/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef IABSTRACTEXTENSION_H
#define IABSTRACTEXTENSION_H


#include "icolor.h"
#include "istring.h"

class iAnimator;
class iAnimatorScript;
class iCrossSectionViewSubject;
class iDataReader;
class iDataReaderExtension;
class iDataType;
class iObjectType;
class iParticlesViewSubject;
class iPicker;
class iPickerExtension;
class iSurfaceViewSubject;
class iTensorFieldViewSubject;
class iVectorFieldViewSubject;
class iViewModule;
class iViewModuleExtension;
class iViewSubject;
class iVolumeViewSubject;


class iAbstractExtension
{

	friend class iObjectFactory;

public:

	inline int GetId() const { return mId; }
	inline int GetSubjectCounter() const { return ++mSubjectCounter; }
	inline const iString& GetName() const { return mName; }

	static int DefineToId(int n);

protected:

	iAbstractExtension(int id, const iString& name);

	virtual iDataReaderExtension*			CreateDataReaderExtension(iDataReader *reader) const = 0;
	virtual iPickerExtension*				CreatePickerExtension(iPicker *picker) const;
	virtual iViewModuleExtension*			CreateViewModuleExtension(iViewModule *vm) const;

	virtual iCrossSectionViewSubject*		CreateCrossSectionSubject(iViewModule *vm, int index) const;
	virtual iParticlesViewSubject*			CreateParticlesSubject(iViewModule *vm, int index) const;
	virtual iSurfaceViewSubject*			CreateSurfaceSubject(iViewModule *vm, int index) const;
	virtual iTensorFieldViewSubject*		CreateTensorFieldSubject(iViewModule *vm, int index) const;
	virtual iVectorFieldViewSubject*		CreateVectorFieldSubject(iViewModule *vm, int index) const;
	virtual iVolumeViewSubject*				CreateVolumeSubject(iViewModule *vm, int index) const;

	virtual iViewSubject*					CreateSpecialSubject(iViewModule *vm, int index, const iObjectType &type) const;

	//
	//  Creater helpers
	//
	iCrossSectionViewSubject*	NewCrossSectionSubject(iViewModule *vm, const iDataType& type) const;
	iParticlesViewSubject*		NewParticlesSubject(iViewModule *vm, const iDataType& type, const iString &name, const iColor &color = iColor::Invalid(), bool shaded = false) const;
	iSurfaceViewSubject*		NewSurfaceSubject(iViewModule *vm, const iDataType& type) const;
	iTensorFieldViewSubject*	NewTensorFieldSubject(iViewModule *vm, const iDataType& type, const iDataType& stype) const;
	iVectorFieldViewSubject*	NewVectorFieldSubject(iViewModule *vm, const iDataType& type, const iDataType& stype) const;
	iVolumeViewSubject*			NewVolumeSubject(iViewModule *vm, const iDataType& type) const;

private:

	const int mId;
	const iString mName;
	mutable int mSubjectCounter;
	
	iAbstractExtension(); // not implemented
	iAbstractExtension(const iAbstractExtension &); // not implemented
	void operator=(const iAbstractExtension&); // not implemented
};

//
//  Useful macros
//
#define IEXTENSION_DECLARE(_class_) \
	public: \
		static int Build(); \
		static int Id(){ return Self()->GetId(); } \
		static int SubjectId(int n = -1){ return (1+Self()->GetId())*1000 + (n == -1 ? Self()->GetSubjectCounter() : n); } \
		static const iString& Name(){ return Self()->GetName(); } \
	private: \
		_class_(); \
		static int mAutoBuild; \
		static const iAbstractExtension* Self()

#define IEXTENSION_DEFINE(_name_,_class_) \
	int _class_::mAutoBuild = Self()->GetId(); \
	const iAbstractExtension* _class_::Self() \
	{ \
		static iAbstractExtension* self = new _class_; \
		return self; \
	} \
	int _class_::Build() \
	{ \
		return Self()->GetId() - mAutoBuild; \
	} \
	_class_::_class_() : iAbstractExtension(IEXTENSION_##_name_,#_name_){}

#define IEXTENSION_DECLARE_CLASS(_prefix_) \
	class _prefix_##Extension : public iAbstractExtension \
	{ \
		IEXTENSION_DECLARE(_prefix_##Extension); \
	private: \
		virtual iDataReaderExtension*			CreateDataReaderExtension(iDataReader *reader) const; \
		virtual iCrossSectionViewSubject*		CreateCrossSectionSubject(iViewModule *vm, int index) const; \
		virtual iParticlesViewSubject*			CreateParticlesSubject(iViewModule *vm, int index) const; \
		virtual iSurfaceViewSubject*			CreateSurfaceSubject(iViewModule *vm, int index) const; \
		virtual iTensorFieldViewSubject*		CreateTensorFieldSubject(iViewModule *vm, int index) const; \
		virtual iVectorFieldViewSubject*		CreateVectorFieldSubject(iViewModule *vm, int index) const; \
		virtual iVolumeViewSubject*				CreateVolumeSubject(iViewModule *vm, int index) const; \
	}

#define IEXTENSION_DEFINE_CLASS(_name_,_prefix_,_stype_,_vtype_,_ttype_,_ptype_) \
	IEXTENSION_DEFINE(_name_,_prefix_##Extension); \
	iCrossSectionViewSubject* _prefix_##Extension::CreateCrossSectionSubject(iViewModule *vm, int index) const \
	{ \
		if(index == 0) return this->NewCrossSectionSubject(vm,_stype_::DataType()); else return 0; \
	} \
	iParticlesViewSubject* _prefix_##Extension::CreateParticlesSubject(iViewModule *vm, int index) const \
	{ \
		if(index == 0) return this->NewParticlesSubject(vm,_ptype_::DataType(),_ptype_::DataType().GetTextName()); else return 0; \
	} \
	iSurfaceViewSubject* _prefix_##Extension::CreateSurfaceSubject(iViewModule *vm, int index) const \
	{ \
		if(index == 0) return this->NewSurfaceSubject(vm,_stype_::DataType()); else return 0; \
	} \
	iTensorFieldViewSubject* _prefix_##Extension::CreateTensorFieldSubject(iViewModule *vm, int index) const \
	{ \
		if(index == 0) return this->NewTensorFieldSubject(vm,_ttype_::DataType(),_stype_::DataType()); else return 0; \
	} \
	iVectorFieldViewSubject* _prefix_##Extension::CreateVectorFieldSubject(iViewModule *vm, int index) const \
	{ \
		if(index == 0) return this->NewVectorFieldSubject(vm,_vtype_::DataType(),_stype_::DataType()); else return 0; \
	} \
	iVolumeViewSubject* _prefix_##Extension::CreateVolumeSubject(iViewModule *vm, int index) const \
	{ \
		if(index == 0) return this->NewVolumeSubject(vm,_stype_::DataType()); else return 0; \
	} \
	iDataReaderExtension* _prefix_##Extension::CreateDataReaderExtension(iDataReader *reader) const \
	{ \
		return new _prefix_##DataReaderExtension(reader,this->GetName()); \
	}

#endif // IABSTRACTEXTENSION_H

