/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.store.remote.utils.cache;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BiFunction;
import org.opensearch.common.cache.RemovalListener;
import org.opensearch.common.cache.RemovalNotification;
import org.opensearch.common.cache.RemovalReason;
import org.opensearch.common.cache.Weigher;
import org.opensearch.index.store.remote.utils.cache.CacheUsage;
import org.opensearch.index.store.remote.utils.cache.Linked;
import org.opensearch.index.store.remote.utils.cache.LinkedDeque;
import org.opensearch.index.store.remote.utils.cache.RefCountedCache;
import org.opensearch.index.store.remote.utils.cache.stats.CacheStats;
import org.opensearch.index.store.remote.utils.cache.stats.DefaultStatsCounter;
import org.opensearch.index.store.remote.utils.cache.stats.StatsCounter;

class LRUCache<K, V>
implements RefCountedCache<K, V> {
    private final long capacity;
    private final HashMap<K, Node<K, V>> data;
    private final LinkedDeque<Node<K, V>> lru;
    private final RemovalListener<K, V> listener;
    private final Weigher<V> weigher;
    private final StatsCounter<K> statsCounter;
    private volatile ReentrantLock lock;
    private long usage;
    private long activeUsage;

    public LRUCache(long capacity, RemovalListener<K, V> listener, Weigher<V> weigher) {
        this.capacity = capacity;
        this.listener = listener;
        this.weigher = weigher;
        this.data = new HashMap();
        this.lru = new LinkedDeque();
        this.lock = new ReentrantLock();
        this.statsCounter = new DefaultStatsCounter();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V get(K key) {
        Objects.requireNonNull(key);
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node == null) {
                this.statsCounter.recordMisses(key, 1);
                V v = null;
                return v;
            }
            if (node.evictable()) {
                this.lru.moveToBack(node);
            }
            this.statsCounter.recordHits(key, 1);
            Object v = node.value;
            return v;
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V put(K key, V value) {
        Objects.requireNonNull(key);
        Objects.requireNonNull(value);
        long weight = this.weigher.weightOf(value);
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null) {
                Object oldValue = node.value;
                long oldWeight = node.weight;
                node.value = value;
                node.weight = weight;
                long weightDiff = weight - oldWeight;
                if (node.refCount > 0) {
                    this.activeUsage += weightDiff;
                }
                if (node.evictable()) {
                    this.lru.moveToBack(node);
                }
                this.usage += weightDiff;
                this.statsCounter.recordReplacement();
                this.listener.onRemoval(new RemovalNotification(key, oldValue, RemovalReason.REPLACED));
                this.evict();
                Object v = oldValue;
                return v;
            }
            Node<K, V> newNode = new Node<K, V>(key, value, weight);
            this.data.put(key, newNode);
            this.lru.add(newNode);
            this.usage += weight;
            this.evict();
            V v = null;
            return v;
        }
        finally {
            lock.unlock();
        }
    }

    @Override
    public void putAll(Map<? extends K, ? extends V> m) {
        for (Map.Entry<K, V> e : m.entrySet()) {
            this.put(e.getKey(), e.getValue());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V computeIfPresent(K key, BiFunction<? super K, ? super V, ? extends V> remappingFunction) {
        Objects.requireNonNull(key);
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null && node.value != null) {
                V v = remappingFunction.apply(key, node.value);
                if (v != null) {
                    Object oldValue = node.value;
                    long oldWeight = node.weight;
                    long weight = this.weigher.weightOf(v);
                    node.value = v;
                    node.weight = weight;
                    long weightDiff = weight - oldWeight;
                    if (node.evictable()) {
                        this.lru.moveToBack(node);
                    }
                    if (node.refCount > 0) {
                        this.activeUsage += weightDiff;
                    }
                    this.usage += weightDiff;
                    this.statsCounter.recordHits(key, 1);
                    if (oldValue != node.value) {
                        this.statsCounter.recordReplacement();
                        this.listener.onRemoval(new RemovalNotification(node.key, oldValue, RemovalReason.REPLACED));
                    }
                    this.evict();
                    V v2 = v;
                    return v2;
                }
                this.data.remove(key);
                if (node.refCount > 0) {
                    this.activeUsage -= node.weight;
                }
                this.usage -= node.weight;
                if (node.evictable()) {
                    this.lru.remove(node);
                }
                this.statsCounter.recordRemoval(node.weight);
                this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.EXPLICIT));
            }
            this.statsCounter.recordMisses(key, 1);
            V v = null;
            return v;
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void remove(K key) {
        Objects.requireNonNull(key);
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            Node<K, V> node = this.data.remove(key);
            if (node != null) {
                if (node.refCount > 0) {
                    this.activeUsage -= node.weight;
                }
                this.usage -= node.weight;
                if (node.evictable()) {
                    this.lru.remove(node);
                }
                this.statsCounter.recordRemoval(node.weight);
                this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.EXPLICIT));
            }
        }
        finally {
            lock.unlock();
        }
    }

    @Override
    public void removeAll(Iterable<? extends K> keys) {
        for (K key : keys) {
            this.remove(key);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void clear() {
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            this.usage = 0L;
            this.activeUsage = 0L;
            this.lru.clear();
            for (Node<K, V> node : this.data.values()) {
                this.data.remove(node.key);
                this.statsCounter.recordRemoval(node.weight);
                this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.EXPLICIT));
            }
        }
        finally {
            lock.unlock();
        }
    }

    @Override
    public long size() {
        return this.data.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void incRef(K key) {
        Objects.requireNonNull(key);
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null) {
                if (node.refCount == 0) {
                    this.activeUsage += node.weight;
                }
                if (node.evictable()) {
                    this.lru.remove(node);
                }
                ++node.refCount;
            }
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void decRef(K key) {
        Objects.requireNonNull(key);
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null && node.refCount > 0) {
                --node.refCount;
                if (node.evictable()) {
                    this.lru.add(node);
                }
                if (node.refCount == 0) {
                    this.activeUsage -= node.weight;
                }
            }
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long prune() {
        long sum = 0L;
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            for (Linked node = (Node)this.lru.peek(); node != null; node = node.getNext()) {
                this.data.remove(node.key, node);
                sum += node.weight;
                this.statsCounter.recordRemoval(node.weight);
                this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.EXPLICIT));
                Node tmp = node;
                this.lru.remove(tmp);
            }
            this.usage -= sum;
        }
        finally {
            lock.unlock();
        }
        return sum;
    }

    @Override
    public CacheUsage usage() {
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            CacheUsage cacheUsage = new CacheUsage(this.usage, this.activeUsage);
            return cacheUsage;
        }
        finally {
            lock.unlock();
        }
    }

    @Override
    public CacheStats stats() {
        ReentrantLock lock = this.lock;
        lock.lock();
        try {
            CacheStats cacheStats = this.statsCounter.snapshot();
            return cacheStats;
        }
        finally {
            lock.unlock();
        }
    }

    boolean hasOverflowed() {
        return this.usage >= this.capacity;
    }

    void evict() {
        while (this.hasOverflowed()) {
            Node node = (Node)this.lru.poll();
            if (node == null) {
                return;
            }
            this.data.remove(node.key, node);
            this.usage -= node.weight;
            this.statsCounter.recordEviction(node.weight);
            this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.CAPACITY));
        }
    }

    static class Node<K, V>
    implements Linked<Node<K, V>> {
        final K key;
        V value;
        long weight;
        Node<K, V> prev;
        Node<K, V> next;
        int refCount;

        Node(K key, V value, long weight) {
            this.key = key;
            this.value = value;
            this.weight = weight;
            this.prev = null;
            this.next = null;
            this.refCount = 0;
        }

        @Override
        public Node<K, V> getPrevious() {
            return this.prev;
        }

        @Override
        public void setPrevious(Node<K, V> prev) {
            this.prev = prev;
        }

        @Override
        public Node<K, V> getNext() {
            return this.next;
        }

        @Override
        public void setNext(Node<K, V> next) {
            this.next = next;
        }

        public boolean evictable() {
            return this.refCount == 0;
        }

        V getValue() {
            return this.value;
        }
    }
}

