/*
 * Decompiled with CFR 0.152.
 */
package ghidra.formats.gfilesystem;

import ghidra.formats.gfilesystem.DerivedFilePushProducer;
import ghidra.formats.gfilesystem.FSUtilities;
import ghidra.formats.gfilesystem.FileCacheEntry;
import ghidra.util.HashingOutputStream;
import ghidra.util.Msg;
import ghidra.util.NumericUtilities;
import ghidra.util.exception.CancelledException;
import ghidra.util.task.TaskMonitor;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.NoSuchAlgorithmException;
import java.util.Date;
import java.util.UUID;
import java.util.regex.Pattern;
import utilities.util.FileUtilities;

public class FileCache {
    private static final Pattern NESTING_DIR_NAME_REGEX = Pattern.compile("[0-9a-fA-F][0-9a-fA-F]");
    private static final int MD5_BYTE_LEN = 16;
    public static final int MD5_HEXSTR_LEN = 32;
    private static final int NESTING_LEVEL = 2;
    private static final long MAX_FILE_AGE_MS = 86400000L;
    private static final long MAINT_INTERVAL_MS = 172800000L;
    private final File cacheDir;
    private final File newDir;
    private final File lastMaintFile;
    private int fileAddCount;
    private int fileReUseCount;
    private long storageEstimateBytes;
    private long lastMaintTS;

    public FileCache(File cacheDir) throws IOException {
        this.cacheDir = cacheDir;
        this.newDir = new File(cacheDir, "new");
        this.lastMaintFile = new File(cacheDir, ".lastmaint");
        if (!cacheDir.exists() && !cacheDir.mkdirs() || !this.newDir.exists() && !this.newDir.mkdirs()) {
            throw new IOException("Unable to initialize cache dir " + cacheDir);
        }
        this.performCacheMaintIfNeeded();
    }

    public synchronized void purge() {
        for (File f : this.cacheDir.listFiles()) {
            String name = f.getName();
            if (!f.isDirectory() || !NESTING_DIR_NAME_REGEX.matcher(name).matches()) continue;
            FileUtilities.deleteDir((File)f);
        }
    }

    public FileCacheEntry addFile(File f, TaskMonitor monitor) throws IOException, CancelledException {
        try (FileInputStream fis = new FileInputStream(f);){
            FileCacheEntry fileCacheEntry = this.addStream(fis, monitor);
            return fileCacheEntry;
        }
    }

    public FileCacheEntry getFile(String md5) {
        FileCacheEntry cfi = this.getFileByMD5(md5);
        if (cfi != null) {
            cfi.file.setLastModified(System.currentTimeMillis());
        }
        return cfi;
    }

    private FileCacheEntry getFileByMD5(String md5) {
        File f = new File(this.cacheDir, this.getCacheRelPath(md5));
        return f.exists() ? new FileCacheEntry(f, md5) : null;
    }

    private void performCacheMaintIfNeeded() throws IOException {
        long l = this.lastMaintTS = this.lastMaintTS == 0L ? this.lastMaintFile.lastModified() : this.lastMaintTS;
        if (this.lastMaintTS + 172800000L < System.currentTimeMillis()) {
            this.performCacheMaint();
            this.lastMaintTS = System.currentTimeMillis();
            FileUtilities.writeStringToFile((File)this.lastMaintFile, (String)("Last maint run at " + new Date()));
        }
    }

    private void performCacheMaint() {
        this.storageEstimateBytes = 0L;
        Msg.info((Object)this, (Object)("Starting cache cleanup: " + this.cacheDir));
        this.cacheMaintForDir(this.cacheDir, 0);
        Msg.info((Object)this, (Object)("Finished cache cleanup, estimated storage used: " + this.storageEstimateBytes));
    }

    private void cacheMaintForDir(File dir, int nestingLevel) {
        if (nestingLevel < 2) {
            for (File f : dir.listFiles()) {
                String name = f.getName();
                if (!f.isDirectory() || !NESTING_DIR_NAME_REGEX.matcher(name).matches()) continue;
                this.cacheMaintForDir(f, nestingLevel + 1);
            }
        } else if (nestingLevel == 2) {
            this.cacheMaintForLeafDir(dir);
        }
    }

    private void cacheMaintForLeafDir(File dir) {
        long cutoffMS = System.currentTimeMillis() - 86400000L;
        for (File f : dir.listFiles()) {
            if (!f.isFile() || !this.isCacheFileName(f.getName())) continue;
            if (f.lastModified() < cutoffMS) {
                if (!f.delete()) {
                    Msg.error((Object)this, (Object)("Failed to delete cache file " + f));
                } else {
                    Msg.info((Object)this, (Object)("Expired cache file " + f));
                    continue;
                }
            }
            this.storageEstimateBytes += f.length();
        }
    }

    private boolean isCacheFileName(String s) {
        byte[] bytes = NumericUtilities.convertStringToBytes((String)s);
        return bytes != null && bytes.length == 16;
    }

    public FileCacheEntry addStream(InputStream is, TaskMonitor monitor) throws IOException, CancelledException {
        File tmpFile = new File(this.newDir, UUID.randomUUID().toString());
        try {
            FileCacheEntry fileCacheEntry;
            try (FileOutputStream fos = new FileOutputStream(tmpFile);){
                FSUtilities.StreamCopyResult copyResults = FSUtilities.streamCopy(is, fos, monitor);
                fos.close();
                String md5 = NumericUtilities.convertBytesToString((byte[])copyResults.md5);
                fileCacheEntry = this.addTmpFileToCache(tmpFile, md5, copyResults.bytesCopied);
            }
            return fileCacheEntry;
        }
        finally {
            if (tmpFile.exists()) {
                Msg.debug((Object)this, (Object)("Removing left-over temp file " + tmpFile));
                tmpFile.delete();
            }
        }
    }

    public FileCacheEntry pushStream(DerivedFilePushProducer pusher, TaskMonitor monitor) throws IOException, CancelledException {
        File tmpFile = new File(this.newDir, UUID.randomUUID().toString());
        try {
            FileCacheEntry fileCacheEntry;
            HashingOutputStream hos = new HashingOutputStream((OutputStream)new FileOutputStream(tmpFile), "MD5");
            try {
                pusher.push((OutputStream)hos);
                hos.close();
                String md5 = NumericUtilities.convertBytesToString((byte[])hos.getDigest());
                long fileSize = tmpFile.length();
                fileCacheEntry = this.addTmpFileToCache(tmpFile, md5, fileSize);
            }
            catch (Throwable throwable) {
                try {
                    try {
                        hos.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
                catch (NoSuchAlgorithmException e) {
                    throw new IOException("Error getting MD5 algo", e);
                }
                catch (Throwable th) {
                    throw new IOException("Error while pushing stream into cache", th);
                }
            }
            hos.close();
            return fileCacheEntry;
        }
        finally {
            if (tmpFile.exists()) {
                Msg.debug((Object)this, (Object)("Removing left-over temp file " + tmpFile));
                tmpFile.delete();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private FileCacheEntry addTmpFileToCache(File tmpFile, String md5, long fileLen) throws IOException {
        String relPath = this.getCacheRelPath(md5);
        File destCacheFile = new File(this.cacheDir, relPath);
        File destCacheFileDir = destCacheFile.getParentFile();
        if (!destCacheFileDir.exists() && !FileUtilities.mkdirs((File)destCacheFileDir)) {
            throw new IOException("Failed to create cache dir " + destCacheFileDir);
        }
        boolean moved = false;
        boolean reused = false;
        if (destCacheFile.exists()) {
            reused = true;
        } else {
            moved = tmpFile.renameTo(destCacheFile);
            boolean bl = reused = !moved && destCacheFile.exists();
        }
        if (!moved && reused) {
            tmpFile.delete();
        } else if (!moved) {
            throw new IOException("Failed to move " + tmpFile + " to " + destCacheFile);
        }
        FileCache fileCache = this;
        synchronized (fileCache) {
            ++this.fileAddCount;
            if (reused) {
                ++this.fileReUseCount;
                destCacheFile.setLastModified(System.currentTimeMillis());
            } else {
                this.storageEstimateBytes += fileLen;
            }
        }
        return new FileCacheEntry(destCacheFile, md5);
    }

    private String getCacheRelPath(String md5) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < 2; ++i) {
            sb.append(md5.substring(i * 2, (i + 1) * 2));
            sb.append('/');
        }
        sb.append(md5);
        return sb.toString();
    }

    public String toString() {
        return "FileCache [cacheDir=" + this.cacheDir + ", fileAddCount=" + this.fileAddCount + ", storageEstimateBytes=" + this.storageEstimateBytes + ", lastMaintTS=" + this.lastMaintTS + "]";
    }

    public int getFileAddCount() {
        return this.fileAddCount;
    }

    public int getFileReUseCount() {
        return this.fileReUseCount;
    }

    public long getStorageEstimateBytes() {
        return this.storageEstimateBytes;
    }

    public long getMaxFileAgeMS() {
        return 86400000L;
    }
}

