/* ************************************************************************
 * Copyright (C) 2016-2025 Advanced Micro Devices, Inc. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * ************************************************************************ */

#include <fstream>
#include <iostream>
#include <stdlib.h>
#include <vector>

#include "testing_common.hpp"

#define TRSM_BLOCK 128

/* ============================================================================================ */

using hipblasTrsmStridedBatchedExModel = ArgumentModel<e_a_type,
                                                       e_side,
                                                       e_uplo,
                                                       e_transA,
                                                       e_diag,
                                                       e_M,
                                                       e_N,
                                                       e_alpha,
                                                       e_lda,
                                                       e_ldb,
                                                       e_stride_scale,
                                                       e_batch_count>;

inline void testname_trsm_strided_batched_ex(const Arguments& arg, std::string& name)
{
    hipblasTrsmStridedBatchedExModel{}.test_name(arg, name);
}

template <typename T>
void testing_trsm_strided_batched_ex_bad_arg(const Arguments& arg)
{
    using Ts     = hipblas_internal_type<T>;
    bool FORTRAN = arg.api == hipblas_client_api::FORTRAN;
    auto hipblasTrsmStridedBatchedExFn
        = FORTRAN ? hipblasTrsmStridedBatchedEx : hipblasTrsmStridedBatchedEx;

    hipblasLocalHandle handle(arg);

    int64_t            M           = 101;
    int64_t            N           = 100;
    int64_t            lda         = 102;
    int64_t            ldb         = 103;
    int64_t            batch_count = 2;
    hipblasSideMode_t  side        = HIPBLAS_SIDE_LEFT;
    hipblasFillMode_t  uplo        = HIPBLAS_FILL_MODE_LOWER;
    hipblasOperation_t transA      = HIPBLAS_OP_N;
    hipblasDiagType_t  diag        = HIPBLAS_DIAG_NON_UNIT;
    hipDataType        computeType = arg.compute_type;

    int64_t K        = side == HIPBLAS_SIDE_LEFT ? M : N;
    int64_t invAsize = TRSM_BLOCK * K;

    hipblasStride stride_A   = K * lda;
    hipblasStride stride_B   = N * ldb;
    hipblasStride strideInvA = invAsize;

    // Allocate device memory
    device_strided_batch_matrix<T> dA(K, K, lda, stride_A, batch_count);
    device_strided_batch_matrix<T> dB(M, N, ldb, stride_B, batch_count);
    device_strided_batch_matrix<T> dinvA(TRSM_BLOCK, TRSM_BLOCK, K, strideInvA, batch_count);

    device_vector<T> d_alpha(1), d_zero(1);
    const Ts         h_alpha{1}, h_zero{0};

    const Ts* alpha = &h_alpha;
    const Ts* zero  = &h_zero;

    for(auto pointer_mode : {HIPBLAS_POINTER_MODE_HOST, HIPBLAS_POINTER_MODE_DEVICE})
    {
        CHECK_HIPBLAS_ERROR(hipblasSetPointerMode(handle, pointer_mode));

        if(pointer_mode == HIPBLAS_POINTER_MODE_DEVICE)
        {
            CHECK_HIP_ERROR(hipMemcpy(d_alpha, alpha, sizeof(*alpha), hipMemcpyHostToDevice));
            CHECK_HIP_ERROR(hipMemcpy(d_zero, zero, sizeof(*zero), hipMemcpyHostToDevice));
            alpha = d_alpha;
            zero  = d_zero;
        }

        EXPECT_HIPBLAS_STATUS(hipblasTrsmStridedBatchedExFn(nullptr,
                                                            side,
                                                            uplo,
                                                            transA,
                                                            diag,
                                                            M,
                                                            N,
                                                            alpha,
                                                            dA,
                                                            lda,
                                                            stride_A,
                                                            dB,
                                                            ldb,
                                                            stride_B,
                                                            batch_count,
                                                            dinvA,
                                                            invAsize,
                                                            strideInvA,
                                                            computeType),
                              HIPBLAS_STATUS_NOT_INITIALIZED);

        EXPECT_HIPBLAS_STATUS(hipblasTrsmStridedBatchedExFn(handle,
                                                            HIPBLAS_SIDE_BOTH,
                                                            uplo,
                                                            transA,
                                                            diag,
                                                            M,
                                                            N,
                                                            alpha,
                                                            dA,
                                                            lda,
                                                            stride_A,
                                                            dB,
                                                            ldb,
                                                            stride_B,
                                                            batch_count,
                                                            dinvA,
                                                            invAsize,
                                                            strideInvA,
                                                            computeType),
                              HIPBLAS_STATUS_INVALID_VALUE);

        EXPECT_HIPBLAS_STATUS(hipblasTrsmStridedBatchedExFn(handle,
                                                            side,
                                                            HIPBLAS_FILL_MODE_FULL,
                                                            transA,
                                                            diag,
                                                            M,
                                                            N,
                                                            alpha,
                                                            dA,
                                                            lda,
                                                            stride_A,
                                                            dB,
                                                            ldb,
                                                            stride_B,
                                                            batch_count,
                                                            dinvA,
                                                            invAsize,
                                                            strideInvA,
                                                            computeType),
                              HIPBLAS_STATUS_INVALID_VALUE);
        EXPECT_HIPBLAS_STATUS(
            hipblasTrsmStridedBatchedExFn(handle,
                                          side,
                                          uplo,
                                          (hipblasOperation_t)HIPBLAS_FILL_MODE_FULL,
                                          diag,
                                          M,
                                          N,
                                          alpha,
                                          dA,
                                          lda,
                                          stride_A,
                                          dB,
                                          ldb,
                                          stride_B,
                                          batch_count,
                                          dinvA,
                                          invAsize,
                                          strideInvA,
                                          computeType),
            HIPBLAS_STATUS_INVALID_ENUM);
        EXPECT_HIPBLAS_STATUS(
            hipblasTrsmStridedBatchedExFn(handle,
                                          side,
                                          uplo,
                                          transA,
                                          (hipblasDiagType_t)HIPBLAS_FILL_MODE_FULL,
                                          M,
                                          N,
                                          alpha,
                                          dA,
                                          lda,
                                          stride_A,
                                          dB,
                                          ldb,
                                          stride_B,
                                          batch_count,
                                          dinvA,
                                          invAsize,
                                          strideInvA,
                                          computeType),
            HIPBLAS_STATUS_INVALID_ENUM);

        EXPECT_HIPBLAS_STATUS(hipblasTrsmStridedBatchedExFn(handle,
                                                            side,
                                                            uplo,
                                                            transA,
                                                            diag,
                                                            M,
                                                            N,
                                                            alpha,
                                                            dA,
                                                            lda,
                                                            stride_A,
                                                            dB,
                                                            ldb,
                                                            stride_B,
                                                            batch_count,
                                                            dinvA,
                                                            invAsize,
                                                            strideInvA,
                                                            HIP_R_16F),
                              HIPBLAS_STATUS_NOT_SUPPORTED);

        EXPECT_HIPBLAS_STATUS(hipblasTrsmStridedBatchedExFn(handle,
                                                            side,
                                                            uplo,
                                                            transA,
                                                            diag,
                                                            M,
                                                            N,
                                                            nullptr,
                                                            dA,
                                                            lda,
                                                            stride_A,
                                                            dB,
                                                            ldb,
                                                            stride_B,
                                                            batch_count,
                                                            dinvA,
                                                            invAsize,
                                                            strideInvA,
                                                            computeType),
                              HIPBLAS_STATUS_INVALID_VALUE);

        if(pointer_mode == HIPBLAS_POINTER_MODE_HOST)
        {
            EXPECT_HIPBLAS_STATUS(hipblasTrsmStridedBatchedExFn(handle,
                                                                side,
                                                                uplo,
                                                                transA,
                                                                diag,
                                                                M,
                                                                N,
                                                                alpha,
                                                                nullptr,
                                                                lda,
                                                                stride_A,
                                                                dB,
                                                                ldb,
                                                                stride_B,
                                                                batch_count,
                                                                dinvA,
                                                                invAsize,
                                                                strideInvA,
                                                                computeType),
                                  HIPBLAS_STATUS_INVALID_VALUE);
            EXPECT_HIPBLAS_STATUS(hipblasTrsmStridedBatchedExFn(handle,
                                                                side,
                                                                uplo,
                                                                transA,
                                                                diag,
                                                                M,
                                                                N,
                                                                alpha,
                                                                dA,
                                                                lda,
                                                                stride_A,
                                                                nullptr,
                                                                ldb,
                                                                stride_B,
                                                                batch_count,
                                                                dinvA,
                                                                invAsize,
                                                                strideInvA,
                                                                computeType),
                                  HIPBLAS_STATUS_INVALID_VALUE);
        }

        // If alpha == 0, then A can be nullptr
        CHECK_HIPBLAS_ERROR(hipblasTrsmStridedBatchedExFn(handle,
                                                          side,
                                                          uplo,
                                                          transA,
                                                          diag,
                                                          M,
                                                          N,
                                                          zero,
                                                          nullptr,
                                                          lda,
                                                          stride_A,
                                                          dB,
                                                          ldb,
                                                          stride_B,
                                                          batch_count,
                                                          dinvA,
                                                          invAsize,
                                                          strideInvA,
                                                          computeType));

        // If M == 0 || N == 0 || batch_count == 0, can have nullptrs
        CHECK_HIPBLAS_ERROR(hipblasTrsmStridedBatchedExFn(handle,
                                                          side,
                                                          uplo,
                                                          transA,
                                                          diag,
                                                          0,
                                                          N,
                                                          nullptr,
                                                          nullptr,
                                                          lda,
                                                          stride_A,
                                                          nullptr,
                                                          ldb,
                                                          stride_B,
                                                          batch_count,
                                                          nullptr,
                                                          invAsize,
                                                          strideInvA,
                                                          computeType));
        CHECK_HIPBLAS_ERROR(hipblasTrsmStridedBatchedExFn(handle,
                                                          side,
                                                          uplo,
                                                          transA,
                                                          diag,
                                                          M,
                                                          0,
                                                          nullptr,
                                                          nullptr,
                                                          lda,
                                                          stride_A,
                                                          nullptr,
                                                          ldb,
                                                          stride_B,
                                                          batch_count,
                                                          nullptr,
                                                          invAsize,
                                                          strideInvA,
                                                          computeType));
        CHECK_HIPBLAS_ERROR(hipblasTrsmStridedBatchedExFn(handle,
                                                          side,
                                                          uplo,
                                                          transA,
                                                          diag,
                                                          M,
                                                          N,
                                                          nullptr,
                                                          nullptr,
                                                          lda,
                                                          stride_A,
                                                          nullptr,
                                                          ldb,
                                                          stride_B,
                                                          0,
                                                          nullptr,
                                                          invAsize,
                                                          strideInvA,
                                                          computeType));

        CHECK_HIPBLAS_ERROR(hipblasTrsmStridedBatchedExFn(handle,
                                                          side,
                                                          uplo,
                                                          transA,
                                                          diag,
                                                          M,
                                                          N,
                                                          alpha,
                                                          dA,
                                                          lda,
                                                          stride_A,
                                                          dB,
                                                          ldb,
                                                          stride_B,
                                                          batch_count,
                                                          nullptr,
                                                          invAsize,
                                                          strideInvA,
                                                          computeType));
    }
}

template <typename T>
void testing_trsm_strided_batched_ex(const Arguments& arg)
{
    using Ts     = hipblas_internal_type<T>;
    bool FORTRAN = arg.api == hipblas_client_api::FORTRAN;
    auto hipblasTrsmStridedBatchedExFn
        = FORTRAN ? hipblasTrsmStridedBatchedEx : hipblasTrsmStridedBatchedEx;

    hipblasSideMode_t  side         = char2hipblas_side(arg.side);
    hipblasFillMode_t  uplo         = char2hipblas_fill(arg.uplo);
    hipblasOperation_t transA       = char2hipblas_operation(arg.transA);
    hipblasDiagType_t  diag         = char2hipblas_diagonal(arg.diag);
    int                M            = arg.M;
    int                N            = arg.N;
    int                lda          = arg.lda;
    int                ldb          = arg.ldb;
    double             stride_scale = arg.stride_scale;
    int                batch_count  = arg.batch_count;

    T h_alpha = arg.get_alpha<T>();

    int K = (side == HIPBLAS_SIDE_LEFT ? M : N);

    hipblasStride stride_A    = size_t(lda) * K * stride_scale;
    hipblasStride stride_B    = size_t(ldb) * N * stride_scale;
    hipblasStride stride_invA = TRSM_BLOCK * size_t(K);
    size_t        A_size      = stride_A * batch_count;
    size_t        B_size      = stride_B * batch_count;
    size_t        invA_size   = stride_invA * batch_count;

    // check here to prevent undefined memory allocation error
    // TODO: Workaround for cuda tests, not actually testing return values
    if(M < 0 || N < 0 || lda < K || ldb < M || batch_count < 0)
    {
        return;
    }
    if(!batch_count)
    {
        return;
    }

    // Naming: `h` is in CPU (host) memory(eg hA), `d` is in GPU (device) memory (eg dA).
    // Allocate host memory
    host_strided_batch_matrix<T> hA(K, K, lda, stride_A, batch_count);
    host_strided_batch_matrix<T> hB_host(M, N, ldb, stride_B, batch_count);
    host_strided_batch_matrix<T> hB_device(M, N, ldb, stride_B, batch_count);
    host_strided_batch_matrix<T> hB_cpu(M, N, ldb, stride_B, batch_count);

    // Check host memory allocation
    CHECK_HIP_ERROR(hA.memcheck());
    CHECK_HIP_ERROR(hB_host.memcheck());
    CHECK_HIP_ERROR(hB_device.memcheck());
    CHECK_HIP_ERROR(hB_cpu.memcheck());

    // Allocate device memory
    device_strided_batch_matrix<T> dA(K, K, lda, stride_A, batch_count);
    device_strided_batch_matrix<T> dB(M, N, ldb, stride_B, batch_count);
    device_strided_batch_matrix<T> dinvA(TRSM_BLOCK, TRSM_BLOCK, K, stride_invA, batch_count);
    device_vector<T>               d_alpha(1);

    // Check device memory allocation
    CHECK_DEVICE_ALLOCATION(dA.memcheck());
    CHECK_DEVICE_ALLOCATION(dB.memcheck());
    CHECK_DEVICE_ALLOCATION(dinvA.memcheck());
    CHECK_DEVICE_ALLOCATION(d_alpha.memcheck());

    double             gpu_time_used, hipblas_error_host, hipblas_error_device;
    hipblasLocalHandle handle(arg);

    // Initial data on CPU
    hipblas_init_matrix(
        hA, arg, hipblas_client_never_set_nan, hipblas_diagonally_dominant_triangular_matrix, true);
    hipblas_init_matrix(hB_host, arg, hipblas_client_never_set_nan, hipblas_general_matrix);

    if(diag == HIPBLAS_DIAG_UNIT)
    {
        make_unit_diagonal(uplo, hA);
    }

    for(int b = 0; b < batch_count; b++)
    {
        // Calculate hB = hA*hX;
        ref_trmm<T>(side,
                    uplo,
                    transA,
                    diag,
                    M,
                    N,
                    T(1.0) / h_alpha,
                    (const T*)hA[b],
                    lda,
                    hB_host[b],
                    ldb);
    }

    hB_device.copy_from(hB_host);
    hB_cpu.copy_from(hB_host);

    CHECK_HIP_ERROR(dA.transfer_from(hA));
    CHECK_HIP_ERROR(dB.transfer_from(hB_host));
    CHECK_HIP_ERROR(hipMemcpy(d_alpha, &h_alpha, sizeof(T), hipMemcpyHostToDevice));

    // calculate invA
    int sub_stride_A    = TRSM_BLOCK * lda + TRSM_BLOCK;
    int sub_stride_invA = TRSM_BLOCK * TRSM_BLOCK;
    int blocks          = K / TRSM_BLOCK;

    for(int b = 0; b < batch_count; b++)
    {
        if(blocks > 0)
        {
            CHECK_HIPBLAS_ERROR(hipblasTrtriStridedBatched<T>(handle,
                                                              uplo,
                                                              diag,
                                                              TRSM_BLOCK,
                                                              dA + b * stride_A,
                                                              lda,
                                                              sub_stride_A,
                                                              dinvA + b * stride_invA,
                                                              TRSM_BLOCK,
                                                              sub_stride_invA,
                                                              blocks));
        }

        if(K % TRSM_BLOCK != 0 || blocks == 0)
        {
            CHECK_HIPBLAS_ERROR(
                hipblasTrtriStridedBatched<T>(handle,
                                              uplo,
                                              diag,
                                              K - TRSM_BLOCK * blocks,
                                              dA + sub_stride_A * blocks + b * stride_A,
                                              lda,
                                              sub_stride_A,
                                              dinvA + sub_stride_invA * blocks + b * stride_invA,
                                              TRSM_BLOCK,
                                              sub_stride_invA,
                                              1));
        }
    }

    if(arg.unit_check || arg.norm_check)
    {
        /* =====================================================================
            HIPBLAS
        =================================================================== */
        CHECK_HIPBLAS_ERROR(hipblasSetPointerMode(handle, HIPBLAS_POINTER_MODE_HOST));
        CHECK_HIPBLAS_ERROR(hipblasTrsmStridedBatchedExFn(handle,
                                                          side,
                                                          uplo,
                                                          transA,
                                                          diag,
                                                          M,
                                                          N,
                                                          reinterpret_cast<Ts*>(&h_alpha),
                                                          dA,
                                                          lda,
                                                          stride_A,
                                                          dB,
                                                          ldb,
                                                          stride_B,
                                                          batch_count,
                                                          dinvA,
                                                          invA_size,
                                                          stride_invA,
                                                          arg.compute_type));

        // copy output from device to CPU
        CHECK_HIP_ERROR(hB_host.transfer_from(dB));
        CHECK_HIP_ERROR(dB.transfer_from(hB_device));

        CHECK_HIPBLAS_ERROR(hipblasSetPointerMode(handle, HIPBLAS_POINTER_MODE_DEVICE));
        CHECK_HIPBLAS_ERROR(hipblasTrsmStridedBatchedExFn(handle,
                                                          side,
                                                          uplo,
                                                          transA,
                                                          diag,
                                                          M,
                                                          N,
                                                          d_alpha,
                                                          dA,
                                                          lda,
                                                          stride_A,
                                                          dB,
                                                          ldb,
                                                          stride_B,
                                                          batch_count,
                                                          dinvA,
                                                          invA_size,
                                                          stride_invA,
                                                          arg.compute_type));

        CHECK_HIP_ERROR(hB_device.transfer_from(dB));

        /* =====================================================================
           CPU BLAS
        =================================================================== */
        for(int b = 0; b < batch_count; b++)
        {
            ref_trsm<T>(
                side, uplo, transA, diag, M, N, h_alpha, (const T*)hA[b], lda, hB_cpu[b], ldb);
        }

        // if enable norm check, norm check is invasive
        real_t<T> eps       = std::numeric_limits<real_t<T>>::epsilon();
        double    tolerance = eps * 40 * M;

        hipblas_error_host
            = norm_check_general<T>('F', M, N, ldb, stride_B, hB_cpu, hB_host, batch_count);
        hipblas_error_device
            = norm_check_general<T>('F', M, N, ldb, stride_B, hB_cpu, hB_device, batch_count);
        if(arg.unit_check)
        {
            unit_check_error(hipblas_error_host, tolerance);
            unit_check_error(hipblas_error_device, tolerance);
        }
    }

    if(arg.timing)
    {
        hipStream_t stream;
        CHECK_HIPBLAS_ERROR(hipblasGetStream(handle, &stream));
        CHECK_HIPBLAS_ERROR(hipblasSetPointerMode(handle, HIPBLAS_POINTER_MODE_DEVICE));

        int runs = arg.cold_iters + arg.iters;
        for(int iter = 0; iter < runs; iter++)
        {
            if(iter == arg.cold_iters)
            {
                gpu_time_used = get_time_us_sync(stream);
            }

            CHECK_HIPBLAS_ERROR(hipblasTrsmStridedBatchedExFn(handle,
                                                              side,
                                                              uplo,
                                                              transA,
                                                              diag,
                                                              M,
                                                              N,
                                                              d_alpha,
                                                              dA,
                                                              lda,
                                                              stride_A,
                                                              dB,
                                                              ldb,
                                                              stride_B,
                                                              batch_count,
                                                              dinvA,
                                                              invA_size,
                                                              stride_invA,
                                                              arg.compute_type));
        }
        gpu_time_used = get_time_us_sync(stream) - gpu_time_used;

        hipblasTrsmStridedBatchedExModel{}.log_args<T>(std::cout,
                                                       arg,
                                                       gpu_time_used,
                                                       trsm_gflop_count<T>(M, N, K),
                                                       trsm_gbyte_count<T>(M, N, K),
                                                       hipblas_error_host,
                                                       hipblas_error_device);
    }
}
