// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Beam.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief Kaon spectra at 3.63, 4.03 and 4.5 GeV
  class PLUTO_1977_I118873 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(PLUTO_1977_I118873);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      declare(Beam(), "Beams");
      declare(UnstableParticles(Cuts::pid==PID::K0S), "UFS");
      book(_h_sigma, 1, 1, 1);
      for (const string& en : _h_sigma.binning().edges<0>()) {
        const size_t idx = en.find("-");
        if (idx != string::npos) {
          const double emin = std::stod(en.substr(0,idx));
          const double emax = std::stod(en.substr(idx+1,string::npos));
          if (inRange(sqrtS()/GeV, emin, emax)) {
            _edge = en;  break;
          }
        }
        else {
          const double eval = std::stod(en);
          if (isCompatibleWithSqrtS(eval)) {
            _edge = en; break;
          }
        }
      }
      size_t ih = 1;
      for (const double eval : {3.63, 4.03, 4.5}) {
        const string label = toString(eval/MeV);
        if (isCompatibleWithSqrtS(eval))  _sqs = label;
        book(_h[label], ++ih, 1, 1);
      }
      raiseBeamErrorIf(_edge.empty() && _sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // Get beams and average beam momentum
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      const double meanBeamMom = 0.5*(beams.first.p3().mod() + beams.second.p3().mod());
      MSG_DEBUG("Avg beam momentum = " << meanBeamMom);
      // unstable particles
      for (const Particle& p : apply<UnstableParticles>(event, "UFS").particles()) {
        _h_sigma->fill(_edge);
        const double xp = p.E()/meanBeamMom;
        const double beta = p.p3().mod()/p.E();
        if (!_sqs.empty()) _h[_sqs]->fill(xp, 1.0/beta);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      const double sf = crossSection()/nanobarn/sumOfWeights();
      scale(_h, sqr(sqrtS())*sf); //< Undefined when merging
      scale(_h_sigma, sf);
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    BinnedHistoPtr<string> _h_sigma;
    string _sqs = "", _edge = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(PLUTO_1977_I118873);


}
