// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"
#include "Rivet/Projections/DecayedParticles.hh"

namespace Rivet {


  /// @brief D0 -> pi+pi0eta and D+ -> pi+pi-eta
  class BESIII_2024_I2777242 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2024_I2777242);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      UnstableParticles ufs = UnstableParticles(Cuts::abspid==411 ||
                                                Cuts::abspid==421);
      declare(ufs, "UFS");
      DecayedParticles DD(ufs);
      DD.addStable(PID::PI0);
      DD.addStable(PID::K0S);
      DD.addStable(PID::ETA);
      declare(DD,"DD");
      // // histograms
      for(unsigned int ix=0;ix<2;++ix) {
        for(unsigned int iy=0;iy<2;++iy)
          book(_h_etapi[ix][iy],1,1+ix,1+iy);
        book(_h_pipi[ix], 1,1+ix,3);
      }
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      static const map<PdgId,unsigned int> mode[2]   = { { { 211,1}, {-211,1}, {221,1} },
                                                         { { 211,1}, { 111,1}, {221,1} } };
      static const map<PdgId,unsigned int> modeCC[2] = { { { 211,1}, {-211,1}, {221,1} },
                                                         { {-211,1}, { 111,1}, {221,1} } };
      DecayedParticles DD = apply<DecayedParticles>(event, "DD");
      // loop over particles
      for(unsigned int ix=0;ix<DD.decaying().size();++ix) {
        unsigned int ip = DD.decaying()[ix].abspid()==411 ? 1 : 0;
        int sign = 1;
        if (DD.decaying()[ix].pid()>0 && DD.modeMatches(ix,3,mode[ip])) {
          sign=1;
        }
        else if  (DD.decaying()[ix].pid()<0 && DD.modeMatches(ix,3,modeCC[ip])) {
          sign=-1;
        }
        else
          continue;
        const Particle & eta = DD.decayProducts()[ix].at(      221)[0];
        const Particle & pim = ip==0 ? DD.decayProducts()[ix].at(-sign*211)[0] : DD.decayProducts()[ix].at(111)[0];
        const Particle & pip = DD.decayProducts()[ix].at( sign*211)[0];
        double mpipi=(pim.momentum()+pip.momentum()).mass();
        // apply KS0 cut
        if( ip==0 && abs(mpipi-.497611) < 0.03) continue;
        // mass dists
        _h_pipi[ip]->fill(mpipi);
        _h_etapi[ip][0]->fill((eta.momentum()+pip.momentum()).mass());
        _h_etapi[ip][1]->fill((eta.momentum()+pim.momentum()).mass());
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      normalize(_h_pipi );
      normalize(_h_etapi);
    }

    /// @}


    /// @name Histograms
    /// @{
    Histo1DPtr _h_pipi[2], _h_etapi[2][2];
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESIII_2024_I2777242);

}
