// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/ChargedFinalState.hh"
#include "Rivet/Projections/Beam.hh"

namespace Rivet {


  /// @brief pi, K and proton spectra at 12 and 30 GeV
  class TASSO_1980_I153656 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1980_I153656);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      declare(Beam(), "Beams");
      declare(ChargedFinalState(), "FS");

      // Book histograms
      size_t ih = 0;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;

        book(_h[en+"pi"], 3*ih+2, 1, 1);
        book(_h[en+"K"],  3*ih+3, 1, 1);
        book(_h[en+"p"],  3*ih+4, 1, 1);

        tribook(en+"pi", 3*ih+ 8, 1, 1);
        tribook(en+"K",  3*ih+ 9, 1, 1);
        tribook(en+"p",  3*ih+10, 1, 1);

        if (ih) {
          _axes[en+"pi"] = YODA::Axis<double>({0.325, 0.375, 0.425, 0.5, 0.575, 0.7, 0.9, 1.1, 1.3, 1.5});
          _axes[en+"K"]  = YODA::Axis<double>({0.4, 0.5, 0.575, 0.7, 0.9, 1.1});
          _axes[en+"p"]  = YODA::Axis<double>({0.5, 0.7, 1.2325, 2.0975});
          _axes[en+"r"]  = YODA::Axis<double>({0.4, 0.5, 0.575, 0.7, 0.9, 1.1});
        }
        else {
          _axes[en+"pi"] = YODA::Axis<double>({0.3, 0.4, 0.5, 0.6, 1.0, 1.6});
          _axes[en+"K"]  = YODA::Axis<double>({0.4, 0.5, 0.6, 1.0});
          _axes[en+"p"]  = YODA::Axis<double>({0.5, 0.7, 0.9, 2.43});
          _axes[en+"r"]  = YODA::Axis<double>({0.4, 0.5, 0.6, 1.0});
        }
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
      _axes["rp"]  = YODA::Axis<double>({0.475, 0.725, 1.0, 2.2});
    }

    void tribook(const string& label, unsigned int d, unsigned int x, unsigned int y) {
      book(_nd["n_"+label], "TMP/n_"+label, refData<YODA::BinnedEstimate<string>>(d, x, y));
      book(_nd["d_"+label], "TMP/d_"+label, refData<YODA::BinnedEstimate<string>>(d, x, y));
      book(_r[label], d, x, y);
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      if (_edges.empty()) {
        for (const auto& item : _h) {
          _edges[item.first] = item.second->xEdges();
        }
        for (const auto& item : _r) {
          _edges["r"+item.first] = _nd["n_"+item.first]->xEdges();
        }
      }
      // First, veto on leptonic events by requiring at least 4 charged FS particles
      const ChargedFinalState& fs = apply<ChargedFinalState>(event, "FS");
      const size_t numParticles = fs.particles().size();

      // Even if we only generate hadronic events, we still need a cut on numCharged >= 2.
      if (numParticles < 2)  vetoEvent;

      // Get beams and average beam momentum
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      const double meanBeamMom = 0.5*(beams.first.p3().mod() + beams.second.p3().mod());
      MSG_DEBUG("Avg beam momentum = " << meanBeamMom);

      for (const Particle& p : fs.particles()) {
        double modp = p.p3().mod();
        fillND("d_"+_sqs+"pi", modp);
        fillND("d_"+_sqs+"K", modp);
        fillND("d_"+_sqs+"p", modp);
        if (p.abspid()==211) {
          fillhist(_sqs+"pi", modp);
          fillND("n_"+_sqs+"pi", modp);
        }
        else if (p.abspid()==321) {
          fillhist(_sqs+"K", modp);
          fillND("n_"+_sqs+"K", modp);
        }
        else if (p.abspid()==2212) {
          fillhist(_sqs+"p", modp);
          fillND("n_"+_sqs+"p", modp);
        }
      }
    }

    void fillhist(const string& label, const double value) {
      string edge = "OTHER";
      const size_t idx = _axes[label].index(value);
      if (idx && idx <= _edges[label].size())  edge = _edges[label][idx-1];
      _h[label]->fill(edge);
    }

    void fillND(const string& label, const double value) {
      string edge = "OTHER";
      const string tag = label.substr(2);
      const string en = tag.substr(0,2);
      const size_t idx = _axes[(tag.substr(2) == "p")? "rp" : en+"r"].index(value);
      if (idx && idx <= _edges["r"+tag].size())  edge = _edges["r"+tag][idx-1];
      _nd[label]->fill(edge);
    }

    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_h, crossSection()/nanobarn/sumOfWeights());
      for (auto& item : _h) {
        for (auto& b: item.second->bins()) {
          b.scaleW(1./_axes[item.first].width(b.index()));
        }
      }
      for (auto& item : _r) {
        divide(_nd["n_"+item.first], _nd["d_"+item.first], item.second);
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,BinnedHistoPtr<string>> _h, _nd;
    map<string,BinnedEstimatePtr<string>> _r;
    map<string, YODA::Axis<double>> _axes;
    map<string, vector<string>> _edges;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(TASSO_1980_I153656);
}
