//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// GEANT4 Class header file
//
//
// File name:     G4AtimaFluctuations
//
// Author:        Jose Luis Rodriguez Sanchez on base of ATIMA code
//
// Creation date: 16.01.2018
//
// Modifications:
// 09/10/2018: Deleted Titeica's correction at high energies
//
//
// Class Description:
//
// This model calculates the energy-loss fluctuations according to the ATIMA code
// developed at GSI, Darmstadt, Germany.
// For details: http://web-docs.gsi.de/~weick/atima/
//
// Helmut Weick, GSI (responsible for fortran version)
// Andrej Prochazka, GSI (responsible for C version)
// Christoph Scheidenberger, GSI (project coordination)
//
// -------------------------------------------------------------------
//
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....
//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

#include "G4AtimaFluctuations.hh"
#include "G4PhysicalConstants.hh"
#include "G4SystemOfUnits.hh"
#include "Randomize.hh"
#include "G4Poisson.hh"
#include "G4MaterialCutsCouple.hh"
#include "G4Material.hh"
#include "G4NistManager.hh"
#include "G4DynamicParticle.hh"
#include "G4Log.hh"
#include "G4Exp.hh"
#include "G4Pow.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

using namespace std;

G4double G4AtimaFluctuations::stepE = 0.0;
G4double G4AtimaFluctuations::tableE[] = {0.0};

G4AtimaFluctuations::G4AtimaFluctuations(const G4String& nam)
  : G4VEmFluctuationModel(nam),
    particle(nullptr),
    particleMass(CLHEP::proton_mass_c2),
    charge(1.0),
    chargeSquare(1.0),
    effChargeSquare(1.0),
    minLoss(0.001*CLHEP::eV)
{
  g4calc = G4Pow::GetInstance();
  kineticEnergy = 0.0;
  beta2 = 0.0;
  MLN10 = 2.30258509299;
  atomic_mass_unit = 931.4940954; // MeV/c^2
  dedx_constant = 0.3070749187;  //4*pi*Na*me*c^2*r_e^2  //MeV cm^2
  electron_mass = 0.510998928;   // MeV/c^2
  fine_structure = 1.0/137.035999139;
  domega2dx_constant = dedx_constant*electron_mass;  //4*pi*Na*me*c^2*r_e^2  //MeV^2 cm^2
  if(tableE[0] == 0.0) {
    G4double logmin = 0.;
    G4double logmax = 5.;
    stepE = (logmax-logmin)/(G4double)(199);
    for(G4int i=0; i<200; ++i){
      tableE[i] = G4Exp(MLN10*(logmin + ((G4double)i)*stepE));
    }
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

G4AtimaFluctuations::~G4AtimaFluctuations() = default;

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

void G4AtimaFluctuations::InitialiseMe(const G4ParticleDefinition* part)
{
  particle       = part;
  particleMass   = part->GetPDGMass();
  charge         = part->GetPDGCharge()/eplus;
  chargeSquare   = charge*charge;
  effChargeSquare= chargeSquare;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

G4double 
G4AtimaFluctuations::SampleFluctuations(const G4MaterialCutsCouple* couple,
                                        const G4DynamicParticle* dp,
                                        const G4double tcut,
                                        const G4double tmax,
                                        const G4double length,
                                        const G4double meanLoss)
{
  //  G4cout << "### meanLoss= " << meanLoss << G4endl;
  if(meanLoss <= minLoss) return meanLoss;

 // G4cout << "G4AtimaFluctuations::SampleFluctuations E(MeV)= " 
 //    << dp->GetKineticEnergy()
 //    << "  Elim(MeV)= " << parameter*charge*particleMass 
 //    << " " << parameter << " " << charge << " " << particleMass << G4endl; 

  const G4Material* material = couple->GetMaterial();
  G4double siga = Dispersion(material,dp,tcut,tmax,length);
  
  CLHEP::HepRandomEngine* rndmEngine = G4Random::getTheEngine();
  //G4cout << "meanLoss= " << meanLoss << " loss= " << siga << G4endl;
  return G4RandGauss::shoot(rndmEngine,meanLoss,std::sqrt(siga));
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

G4double G4AtimaFluctuations::Dispersion(const G4Material* mat,
                                         const G4DynamicParticle* dp,
				         const G4double, const G4double,
                                         const G4double length)
{
  kineticEnergy = dp->GetKineticEnergy();
  const G4ParticleDefinition* p = dp->GetDefinition();
  if(particle != p) { InitialiseMe(p); }

  G4double ap = p->GetPDGMass()/atomic_mass_unit;
  G4double zp = p->GetPDGCharge();
  G4double ep = kineticEnergy/ap;// ep in MeV/u
  G4double zt = mat->GetIonisation()->GetZeffective();
  G4double at = G4NistManager::Instance()->GetAtomicMassAmu(G4lrint(zt));

  G4int z = G4lrint(zp);
  if(z > 109) { z = 109; }

  G4double gamma=1.0 + ep/atomic_mass_unit;
  beta2=1.0-1.0/(gamma*gamma);
  G4double beta = std::sqrt(beta2);
  //z_eff_Pierce_Blann(Z, beta);
  G4double zp_eff = zp*(1.0-G4Exp(-0.95*137.035999139*beta/g4calc->Z23(z)));
  //
  G4double f = domega2dx_constant*zp_eff*zp_eff*zt/at;
  //
  G4double cor =
    24.89 * g4calc->powA(zt,1.2324)/(electron_mass*1e6 * beta2)*
    G4Log( 2.0*electron_mass*1e6*beta2/(33.05*g4calc->powA(zt,1.6364)));
  cor = std::max(cor, 0.0 );

  //Lindhard corrections
  if(ep<tableE[0])ep = tableE[0];
  G4double da = (ap - element_atomic_weights[z])/element_atomic_weights[z];

  G4double v3 = EnergyTable_interpolate(tableE,ep,ls_X_coefficients_a[z-1]);
  G4double v4 = EnergyTable_interpolate(tableE,ep,ls_X_coefficients_ahi[z-1]);
  G4double dif = v4 - v3;
  G4double X = v3+(dif*da/0.05);
  X *= gamma*gamma;
  //
  G4double sse = 0.;
  if(ep<30.0){
    //Energy straggling Firsov
    G4double factor = 4.8184e-3*g4calc->powA(zp+zt,8.0/3.0)/at;
    sse = std::min(f*(X+cor), factor*beta2/fine_structure/fine_structure);
  }else{
   // sse = f*(X+cor); 
    sse = f*X;//09/10/2018
  }
  //
  return sse*length/(cm)*mat->GetDensity()/(g/cm3);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

void G4AtimaFluctuations::SetParticleAndCharge(const G4ParticleDefinition* part,
                                               G4double q2)
{
  if(part != particle) {
    particle       = part;
    particleMass   = part->GetPDGMass();
    charge         = part->GetPDGCharge()/eplus;
    chargeSquare   = charge*charge;
  }
  effChargeSquare  = q2;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4AtimaFluctuations::EnergyTable_interpolate(const G4double* table, G4double xval, const G4double* y){
  G4double r;
  G4int num=200;
  G4double lxval = G4Log(xval)/MLN10;
  if(xval<table[0] || xval>table[num-1])return 0.0;
  if(xval==table[num-1])return y[num-1];
  G4int i = (G4int)(lxval/stepE);
  i = std::min(std::max(i, 0), num-2);
  G4double linstep = table[i+1] - table[i];
  G4double x = 1.0 - ((xval - table[i])/linstep);
  r = (x*y[i]) + ((1-x)*y[i+1]);
  return r;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

const G4double G4AtimaFluctuations::element_atomic_weights[110]={
 0.0,
 1.00794, //H
 4.0026, //He
 6.941, //Li
 9.01218, //Be
 10.811, //B
 12.0107, //C
 14.0067, //N
 15.9994, //O
 18.9984, //F
 20.1797, //Ne
 22.9898, //Na
 24.305, //Mg
 26.9815, //Al
 28.0855, //Si
 30.9738, //P
 32.065, //S
 35.453, //Cl
 39.948, //Ar
 39.0983, //K
 40.078, //Ca
 44.9559, //Sc
 47.867, //Ti
 50.9415, //V
 51.9961, //Cr
 54.938, //Mn
 55.845, //Fe
 58.9332, //Co
 58.6934, //Ni
 63.546, //Cu
 65.409, //Zn
 69.723, //Ga
 72.64, //Ge
 74.9216, //As
 78.96, //Se
 79.904, //Br
 83.798, //Kr
 85.4678, //Rb
 87.62, //Sr
 88.9059, //Y
 91.224, //Zr
 92.9064, //Nb
 95.94, //Mo
 97.9072, //Tc
 101.07, //Ru
 102.906, //Rh
 106.42, //Pd
 107.868, //Ag
 112.411, //Cd
 114.818, //In
 118.71, //Sn
 121.76, //Sb
 127.6, //Te
 126.904, //I
 131.293, //Xe
 132.905, //Cs
 137.327, //Ba
 138.905, //La
 140.116, //Ce
 140.908, //Pr
 144.24, //Nd
 144.913, //Pm
 150.36, //Sm
 151.964, //Eu
 157.25, //Gd
 158.925, //Tb
 162.5, //Dy
 164.93, //Ho
 167.259, //Er
 168.934, //Tm
 173.04, //Yb
 174.967, //Lu
 178.49, //Hf
 180.948, //Ta
 183.84, //W
 186.207, //Re
 190.23, //Os
 192.217, //Ir
 195.078, //Pt
 196.967, //Au
 200.59, //Hg
 204.383, //Tl
 207.2, //Pb
 208.98, //Bi
 208.982, //Po
 209.987, //At
 222.018, //Rn
 223.02, //Fr
 226.025, //Ra
 227.028, //Ac
 232.038, //Th
 231.036, //Pa
 238.029, //U
 237.048, //Np
 244.064, //Pu
 243.061, //Am
 247.07, //Cm
 247.07, //Bk
 251.08, //Cf
 252.083, //Es
 257.095, //Fm
 258.098, //Md
 259.101, //No
 262.11, //Lr
 261.109, //Rf
 262.114, //Db
 266.122, //Sg
 264.125, //Bh
 269.134, //Hs
 268.139 //Mt
};

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

//arrays dimensions are [z][energy], z=1 starts from index=0

//LS X coefficient (dE straggling) for A=atomic weight
const G4double G4AtimaFluctuations::ls_X_coefficients_a[110][200]=
{
	{0.999182,0.999124,0.999062,0.998995,0.998925,0.998851,0.998772,0.998688,0.998598,0.998504,0.998404,0.998297,0.998184,0.998064,0.997937,0.997802,0.997659,0.997507,0.997346,0.997175,0.996994,0.996802,0.996598,0.996382,0.996153,0.99591,0.995653,0.995381,0.995092,0.994785,0.994461,0.994117,0.993753,0.993367,0.992958,0.992526,0.992067,0.991582,0.991068,0.990524,0.989948,0.989338,0.988693,0.988011,0.987289,0.986525,0.985717,0.984863,0.98396,0.983005,0.981996,0.98093,0.979804,0.978614,0.977358,0.976032,0.974632,0.973155,0.971597,0.969955,0.968223,0.966397,0.964474,0.962449,0.960317,0.958074,0.955714,0.953233,0.950626,0.947887,0.945012,0.941995,0.938832,0.935516,0.932044,0.928409,0.924608,0.920635,0.916485,0.912155,0.90764,0.902937,0.898042,0.892953,0.887666,0.88218,0.876495,0.870608,0.864522,0.858236,0.851752,0.845074,0.838206,0.831151,0.823917,0.81651,0.808938,0.801211,0.79334,0.785335,0.77721,0.768979,0.760656,0.752258,0.7438,0.735302,0.72678,0.718255,0.709744,0.701267,0.692844,0.684494,0.676236,0.668089,0.660071,0.6522,0.644492,0.636962,0.629625,0.622494,0.61558,0.608894,0.602443,0.596236,0.590278,0.584573,0.579123,0.573929,0.568991,0.564308,0.559877,0.555692,0.55175,0.548044,0.544568,0.541314,0.538274,0.535439,0.532801,0.53035,0.528078,0.525974,0.52403,0.522236,0.520584,0.519064,0.517667,0.516386,0.515211,0.514137,0.513154,0.512256,0.511437,0.510689,0.510008,0.509387,0.508822,0.508306,0.507837,0.507408,0.507018,0.50666,0.506333,0.506032,0.505756,0.5055,0.505262,0.50504,0.504831,0.504632,0.504442,0.504258,0.504078,0.503901,0.503723,0.503542,0.503357,0.503165,0.502964,0.50275,0.502522,0.502277,0.502011,0.501721,0.501403,0.501053,0.500667,0.500239,0.499764,0.499237,0.49865,0.497996,0.497267,0.496453,0.495544,0.494529,0.493396,0.492131,0.490717,0.489139},
	{0.999599,0.999546,0.999489,0.999428,0.999363,0.999294,0.999221,0.999143,0.99906,0.998971,0.998877,0.998778,0.998671,0.998558,0.998438,0.998311,0.998175,0.998031,0.997878,0.997716,0.997543,0.99736,0.997165,0.996959,0.99674,0.996507,0.99626,0.995998,0.99572,0.995425,0.995112,0.99478,0.994428,0.994054,0.993659,0.993239,0.992795,0.992323,0.991824,0.991295,0.990734,0.990141,0.989512,0.988846,0.988141,0.987395,0.986606,0.98577,0.984886,0.983951,0.982962,0.981917,0.980812,0.979645,0.978411,0.977108,0.975732,0.974279,0.972746,0.971129,0.969423,0.967625,0.965729,0.963732,0.961629,0.959416,0.957086,0.954636,0.952061,0.949354,0.946512,0.943529,0.940401,0.93712,0.933684,0.930086,0.926322,0.922387,0.918276,0.913985,0.90951,0.904848,0.899994,0.894946,0.889702,0.884259,0.878617,0.872775,0.866733,0.860491,0.854053,0.847421,0.840598,0.833589,0.826401,0.81904,0.811515,0.803834,0.796008,0.78805,0.779971,0.771785,0.763507,0.755154,0.746741,0.738286,0.729807,0.721324,0.712855,0.704419,0.696036,0.687725,0.679505,0.671396,0.663414,0.655578,0.647903,0.640406,0.6331,0.625999,0.619114,0.612455,0.606031,0.599849,0.593915,0.588232,0.582803,0.577629,0.57271,0.568044,0.563629,0.559459,0.555531,0.551838,0.548373,0.54513,0.542099,0.539272,0.536641,0.534197,0.531929,0.52983,0.527889,0.526097,0.524445,0.522924,0.521526,0.520242,0.519064,0.517984,0.516995,0.516089,0.51526,0.514501,0.513807,0.513171,0.512587,0.512052,0.511559,0.511105,0.510685,0.510295,0.509931,0.509589,0.509265,0.508957,0.508661,0.508372,0.508089,0.507808,0.507525,0.507238,0.506942,0.506634,0.50631,0.505967,0.505599,0.505203,0.504773,0.504304,0.503789,0.503224,0.502599,0.501908,0.501142,0.500292,0.499346,0.498293,0.497121,0.495814,0.494358,0.492735,0.490925,0.488909,0.486663,0.484161,0.481378,0.478282,0.474842,0.471024},
	{1.00017,1.00012,1.00007,1.00002,0.999959,0.999897,0.99983,0.999759,0.999682,0.999601,0.999514,0.999421,0.999322,0.999217,0.999104,0.998984,0.998857,0.998721,0.998576,0.998422,0.998258,0.998083,0.997898,0.997701,0.997491,0.997268,0.997031,0.996779,0.996512,0.996228,0.995926,0.995606,0.995266,0.994905,0.994522,0.994115,0.993684,0.993226,0.992741,0.992226,0.991681,0.991103,0.99049,0.98984,0.989152,0.988424,0.987652,0.986834,0.985969,0.985054,0.984085,0.98306,0.981976,0.98083,0.979618,0.978338,0.976985,0.975557,0.974048,0.972456,0.970776,0.969004,0.967136,0.965167,0.963092,0.960908,0.958608,0.956188,0.953644,0.95097,0.94816,0.945211,0.942116,0.938871,0.93547,0.931908,0.928181,0.924283,0.92021,0.915958,0.911523,0.9069,0.902087,0.897081,0.891878,0.886478,0.880878,0.875079,0.869081,0.862884,0.85649,0.849902,0.843124,0.836161,0.829018,0.821702,0.814222,0.806587,0.798807,0.790894,0.782859,0.774718,0.766485,0.758175,0.749806,0.741394,0.732958,0.724516,0.716088,0.707692,0.699348,0.691076,0.682894,0.67482,0.666874,0.659072,0.651431,0.643965,0.63669,0.629619,0.622762,0.61613,0.609732,0.603574,0.597663,0.592002,0.586594,0.58144,0.576539,0.571891,0.567491,0.563337,0.559422,0.555742,0.552289,0.549056,0.546034,0.543216,0.540593,0.538155,0.535893,0.533798,0.531861,0.530072,0.528422,0.526902,0.525504,0.524218,0.523038,0.521955,0.520961,0.520049,0.519213,0.518446,0.517741,0.517093,0.516496,0.515945,0.515435,0.514961,0.514518,0.514102,0.513709,0.513335,0.512974,0.512625,0.512282,0.511942,0.511601,0.511255,0.5109,0.510531,0.510144,0.509734,0.509296,0.508825,0.508314,0.507759,0.50715,0.506482,0.505745,0.504931,0.504029,0.503028,0.501917,0.50068,0.499305,0.497774,0.49607,0.494172,0.492059,0.489707,0.487089,0.484179,0.480945,0.477355,0.473372,0.468959,0.464076,0.458682},
	{1.00086,1.00082,1.00078,1.00073,1.00069,1.00063,1.00058,1.00051,1.00045,1.00038,1.0003,1.00021,1.00012,1.00003,0.999924,0.999813,0.999695,0.999568,0.999433,0.999288,0.999134,0.998969,0.998793,0.998606,0.998406,0.998193,0.997967,0.997726,0.997469,0.997196,0.996906,0.996598,0.996269,0.995921,0.99555,0.995157,0.994739,0.994295,0.993824,0.993323,0.992793,0.99223,0.991632,0.990999,0.990327,0.989615,0.988861,0.988062,0.987215,0.986318,0.985369,0.984364,0.983301,0.982176,0.980986,0.979728,0.978398,0.976993,0.975508,0.973941,0.972286,0.97054,0.968699,0.966757,0.96471,0.962554,0.960284,0.957895,0.955381,0.952738,0.949961,0.947044,0.943983,0.940771,0.937405,0.933879,0.930188,0.926327,0.922292,0.918078,0.913682,0.909099,0.904326,0.89936,0.894198,0.888839,0.883282,0.877526,0.87157,0.865416,0.859066,0.852522,0.845789,0.83887,0.831771,0.8245,0.817065,0.809474,0.801739,0.79387,0.78588,0.777782,0.769593,0.761326,0.752999,0.74463,0.736235,0.727834,0.719446,0.71109,0.702784,0.694549,0.686404,0.678366,0.670454,0.662686,0.655077,0.647643,0.640398,0.633355,0.626526,0.619921,0.613548,0.607414,0.601526,0.595887,0.590499,0.585365,0.580482,0.575851,0.571467,0.567328,0.563427,0.559759,0.556318,0.553096,0.550084,0.547275,0.54466,0.542229,0.539973,0.537884,0.535951,0.534165,0.532518,0.531001,0.529604,0.528319,0.527138,0.526054,0.525058,0.524143,0.523303,0.522531,0.52182,0.521164,0.520559,0.519998,0.519476,0.518989,0.51853,0.518097,0.517685,0.517288,0.516903,0.516526,0.516152,0.515776,0.515396,0.515005,0.514601,0.514176,0.513727,0.513248,0.512732,0.512175,0.511567,0.510903,0.510174,0.509371,0.508484,0.507501,0.506412,0.505203,0.50386,0.502366,0.500703,0.498854,0.496795,0.494506,0.491958,0.489126,0.48598,0.482487,0.478613,0.474319,0.469569,0.464319,0.458526,0.452149},
	{1.00162,1.0016,1.00157,1.00154,1.00151,1.00147,1.00143,1.00138,1.00132,1.00127,1.0012,1.00113,1.00105,1.00097,1.00088,1.00078,1.00067,1.00056,1.00043,1.0003,1.00016,1,0.99984,0.999664,0.999476,0.999275,0.99906,0.998831,0.998587,0.998326,0.998048,0.997752,0.997436,0.997101,0.996743,0.996363,0.995959,0.995529,0.995072,0.994586,0.99407,0.993523,0.992941,0.992324,0.991668,0.990973,0.990236,0.989455,0.988626,0.987748,0.986818,0.985833,0.98479,0.983686,0.982517,0.981281,0.979973,0.978591,0.977131,0.975587,0.973958,0.972237,0.970422,0.968507,0.966488,0.96436,0.962119,0.959759,0.957275,0.954663,0.951917,0.949033,0.946004,0.942827,0.939495,0.936004,0.932348,0.928523,0.924525,0.920349,0.915991,0.911446,0.906712,0.901786,0.896665,0.891347,0.885831,0.880117,0.874203,0.868092,0.861785,0.855284,0.848594,0.841718,0.834664,0.827437,0.820045,0.812498,0.804806,0.796981,0.789034,0.78098,0.772832,0.764608,0.756323,0.747995,0.739641,0.73128,0.722931,0.714613,0.706345,0.698147,0.690037,0.682034,0.674156,0.66642,0.658843,0.65144,0.644224,0.63721,0.630408,0.623828,0.61748,0.61137,0.605504,0.599886,0.594519,0.589403,0.584538,0.579923,0.575555,0.57143,0.567543,0.563888,0.560458,0.557247,0.554245,0.551444,0.548837,0.546413,0.544164,0.54208,0.540152,0.53837,0.536726,0.535211,0.533816,0.532532,0.531351,0.530266,0.529268,0.528351,0.527508,0.526731,0.526015,0.525353,0.52474,0.524171,0.523639,0.52314,0.522668,0.52222,0.52179,0.521374,0.520967,0.520566,0.520164,0.519758,0.519343,0.518914,0.518465,0.517992,0.517489,0.516949,0.516366,0.515732,0.51504,0.514282,0.513447,0.512527,0.511508,0.51038,0.509129,0.507739,0.506194,0.504477,0.502567,0.500443,0.498081,0.495455,0.492537,0.489297,0.485701,0.481715,0.4773,0.472417,0.467025,0.46108,0.454538,0.447353},
	{1.00243,1.00243,1.00242,1.00241,1.00239,1.00237,1.00235,1.00232,1.00228,1.00224,1.00219,1.00214,1.00208,1.00201,1.00194,1.00185,1.00176,1.00166,1.00155,1.00144,1.00131,1.00117,1.00102,1.00086,1.00069,1.0005,1.0003,1.00008,0.999853,0.999606,0.999342,0.999061,0.998759,0.998438,0.998095,0.997729,0.99734,0.996925,0.996483,0.996012,0.995512,0.99498,0.994415,0.993814,0.993175,0.992498,0.991778,0.991014,0.990204,0.989345,0.988434,0.987468,0.986445,0.985362,0.984214,0.983,0.981715,0.980355,0.978918,0.977399,0.975794,0.974098,0.972309,0.97042,0.968428,0.966328,0.964115,0.961784,0.95933,0.956748,0.954034,0.951181,0.948185,0.94504,0.941742,0.938285,0.934664,0.930875,0.926913,0.922774,0.918453,0.913946,0.909251,0.904364,0.899283,0.894005,0.88853,0.882856,0.876984,0.870914,0.864649,0.858191,0.851543,0.84471,0.837699,0.830514,0.823166,0.815662,0.808012,0.800229,0.792325,0.784313,0.776208,0.768025,0.75978,0.751492,0.743178,0.734856,0.726545,0.718265,0.710034,0.701872,0.693797,0.685828,0.677983,0.670279,0.662733,0.655359,0.648173,0.641186,0.63441,0.627856,0.621532,0.615445,0.609601,0.604004,0.598656,0.593558,0.588711,0.584113,0.57976,0.575649,0.571775,0.568133,0.564714,0.561513,0.558521,0.555729,0.55313,0.550713,0.54847,0.546392,0.544469,0.542692,0.541051,0.539539,0.538146,0.536864,0.535684,0.534599,0.533601,0.532683,0.531838,0.531059,0.530339,0.529673,0.529055,0.528479,0.52794,0.527432,0.526951,0.526492,0.52605,0.525619,0.525197,0.524777,0.524355,0.523926,0.523485,0.523027,0.522547,0.522038,0.521494,0.52091,0.520276,0.519587,0.518832,0.518004,0.517092,0.516085,0.51497,0.513735,0.512364,0.510842,0.509151,0.50727,0.50518,0.502856,0.500274,0.497404,0.494218,0.490683,0.486765,0.482425,0.477625,0.472323,0.466477,0.460041,0.452973,0.445225},
	{1.00324,1.00327,1.00328,1.0033,1.0033,1.00331,1.00331,1.0033,1.00329,1.00327,1.00324,1.00321,1.00317,1.00313,1.00307,1.00301,1.00294,1.00286,1.00277,1.00267,1.00256,1.00245,1.00231,1.00217,1.00202,1.00185,1.00166,1.00147,1.00125,1.00102,1.00078,1.00051,1.00023,0.999922,0.999595,0.999246,0.998873,0.998475,0.998049,0.997596,0.997113,0.996598,0.996049,0.995466,0.994845,0.994185,0.993484,0.992739,0.991947,0.991107,0.990216,0.98927,0.988268,0.987205,0.986079,0.984886,0.983623,0.982286,0.980872,0.979377,0.977796,0.976126,0.974361,0.972499,0.970534,0.968461,0.966275,0.963973,0.961548,0.958996,0.956312,0.95349,0.950526,0.947414,0.944149,0.940725,0.937139,0.933385,0.929459,0.925355,0.921071,0.916602,0.911945,0.907096,0.902053,0.896815,0.891379,0.885746,0.879914,0.873886,0.867662,0.861245,0.854639,0.847848,0.840878,0.833736,0.826429,0.818967,0.81136,0.803618,0.795755,0.787784,0.77972,0.771577,0.763373,0.755124,0.746848,0.738564,0.730291,0.722047,0.713852,0.705725,0.697684,0.689748,0.681935,0.674263,0.666746,0.659402,0.652243,0.645283,0.638533,0.632003,0.625702,0.619638,0.613815,0.608237,0.602908,0.597829,0.592998,0.588415,0.584077,0.57998,0.576119,0.572488,0.56908,0.565889,0.562905,0.560122,0.55753,0.555119,0.552882,0.550809,0.54889,0.547116,0.545478,0.543967,0.542575,0.541293,0.540113,0.539026,0.538026,0.537105,0.536255,0.535471,0.534745,0.534072,0.533445,0.532859,0.532308,0.531787,0.531291,0.530815,0.530354,0.529902,0.529455,0.529008,0.528555,0.528092,0.527613,0.527112,0.526583,0.526021,0.525417,0.524765,0.524057,0.523284,0.522437,0.521505,0.520478,0.519342,0.518084,0.51669,0.515143,0.513425,0.511516,0.509395,0.507038,0.504419,0.501511,0.498284,0.494705,0.490738,0.486347,0.481492,0.476132,0.470222,0.46372,0.45658,0.448757,0.440208},
	{1.00405,1.0041,1.00415,1.00419,1.00422,1.00425,1.00428,1.0043,1.00431,1.00432,1.00432,1.00432,1.00431,1.00429,1.00426,1.00422,1.00418,1.00412,1.00406,1.00399,1.0039,1.00381,1.0037,1.00358,1.00345,1.0033,1.00314,1.00296,1.00277,1.00256,1.00233,1.00209,1.00182,1.00154,1.00123,1.0009,1.00055,1.00017,0.999763,0.999328,0.998864,0.998368,0.997839,0.997274,0.996673,0.996032,0.99535,0.994624,0.993853,0.993033,0.992162,0.991237,0.990255,0.989214,0.988109,0.986938,0.985698,0.984384,0.982994,0.981522,0.979966,0.97832,0.976581,0.974745,0.972806,0.97076,0.968602,0.966328,0.963933,0.96141,0.958756,0.955965,0.953032,0.949951,0.946719,0.943328,0.939776,0.936056,0.932165,0.928097,0.923849,0.919416,0.914796,0.909985,0.904981,0.899781,0.894384,0.88879,0.882998,0.87701,0.870826,0.86445,0.857884,0.851135,0.844206,0.837104,0.829839,0.822418,0.814851,0.80715,0.799328,0.791397,0.783372,0.775269,0.767103,0.758893,0.750655,0.742409,0.734172,0.725964,0.717803,0.70971,0.701702,0.693799,0.686017,0.678375,0.670887,0.663571,0.656439,0.649505,0.64278,0.636274,0.629996,0.623953,0.61815,0.612592,0.607282,0.602219,0.597405,0.592837,0.588513,0.584429,0.58058,0.57696,0.573563,0.570381,0.567406,0.564631,0.562045,0.559641,0.557409,0.55534,0.553425,0.551654,0.550019,0.54851,0.547118,0.545836,0.544655,0.543567,0.542564,0.54164,0.540786,0.539996,0.539264,0.538583,0.537947,0.53735,0.536788,0.536254,0.535743,0.535249,0.534768,0.534294,0.533823,0.533348,0.532864,0.532366,0.531848,0.531304,0.530727,0.53011,0.529446,0.528727,0.527943,0.527086,0.526146,0.52511,0.523966,0.522701,0.5213,0.519747,0.518023,0.516109,0.513983,0.511622,0.509001,0.50609,0.502861,0.49928,0.495313,0.490923,0.486069,0.480712,0.474806,0.468309,0.461176,0.45336,0.44482,0.435515},
	{1.00483,1.00491,1.00499,1.00506,1.00513,1.00519,1.00525,1.0053,1.00534,1.00538,1.00542,1.00544,1.00546,1.00547,1.00548,1.00547,1.00546,1.00543,1.0054,1.00536,1.0053,1.00523,1.00515,1.00506,1.00495,1.00483,1.0047,1.00455,1.00438,1.0042,1.004,1.00378,1.00354,1.00327,1.00299,1.00268,1.00235,1.002,1.00161,1.0012,1.00076,1.00028,0.999774,0.999231,0.99865,0.998031,0.99737,0.996666,0.995916,0.995117,0.994268,0.993365,0.992405,0.991386,0.990304,0.989156,0.987939,0.986648,0.985282,0.983834,0.982302,0.980682,0.978969,0.977158,0.975246,0.973227,0.971097,0.968851,0.966484,0.963991,0.961366,0.958606,0.955703,0.952654,0.949454,0.946096,0.942577,0.938891,0.935034,0.931001,0.926788,0.922392,0.917808,0.913034,0.908067,0.902905,0.897547,0.891991,0.886238,0.880289,0.874145,0.867808,0.861283,0.854573,0.847684,0.840623,0.833397,0.826016,0.818489,0.810828,0.803045,0.795153,0.787167,0.779103,0.770975,0.762802,0.754601,0.74639,0.738189,0.730015,0.721889,0.713829,0.705853,0.697981,0.69023,0.682616,0.675157,0.667868,0.660763,0.653854,0.647153,0.64067,0.634413,0.628391,0.622608,0.617069,0.611776,0.60673,0.601931,0.597377,0.593067,0.588995,0.585158,0.581548,0.578161,0.574987,0.57202,0.569251,0.566672,0.564273,0.562046,0.55998,0.558067,0.556298,0.554664,0.553155,0.551763,0.550479,0.549295,0.548203,0.547196,0.546265,0.545404,0.544606,0.543863,0.543171,0.542522,0.541911,0.541331,0.540778,0.540245,0.539727,0.539219,0.538714,0.538208,0.537694,0.537167,0.536621,0.536049,0.535444,0.5348,0.534108,0.53336,0.532547,0.531659,0.530686,0.529615,0.528436,0.527132,0.525689,0.524091,0.522318,0.520351,0.518168,0.515745,0.513056,0.510072,0.506763,0.503095,0.499035,0.494543,0.48958,0.484104,0.478072,0.471439,0.464162,0.456193,0.447491,0.438016,0.427731},
	{1.00559,1.0057,1.0058,1.00591,1.00601,1.0061,1.0062,1.00628,1.00636,1.00644,1.00651,1.00657,1.00662,1.00667,1.00671,1.00674,1.00676,1.00677,1.00677,1.00676,1.00674,1.0067,1.00666,1.0066,1.00652,1.00643,1.00633,1.00621,1.00608,1.00592,1.00575,1.00556,1.00535,1.00511,1.00486,1.00458,1.00427,1.00394,1.00358,1.0032,1.00278,1.00233,1.00185,1.00133,1.00077,1.00017,0.999537,0.998857,0.99813,0.997355,0.996529,0.995649,0.994713,0.993718,0.99266,0.991536,0.990343,0.989077,0.987735,0.986312,0.984806,0.983211,0.981524,0.97974,0.977854,0.975863,0.973761,0.971543,0.969204,0.96674,0.964145,0.961415,0.958543,0.955526,0.952357,0.949032,0.945545,0.941893,0.93807,0.934071,0.929893,0.925532,0.920984,0.916246,0.911316,0.906191,0.90087,0.895353,0.889638,0.883727,0.877622,0.871324,0.864837,0.858167,0.851317,0.844295,0.837109,0.829767,0.822279,0.814657,0.806912,0.799059,0.791111,0.783083,0.774993,0.766856,0.758691,0.750516,0.742349,0.734209,0.726116,0.718088,0.710144,0.702302,0.69458,0.686995,0.679564,0.672301,0.665221,0.658337,0.651659,0.645199,0.638964,0.632962,0.627198,0.621677,0.616401,0.611372,0.606588,0.602049,0.597752,0.593693,0.589867,0.586268,0.58289,0.579726,0.576767,0.574006,0.571433,0.56904,0.566818,0.564757,0.562849,0.561083,0.559451,0.557944,0.556553,0.55527,0.554086,0.552994,0.551985,0.551052,0.550188,0.549386,0.54864,0.547942,0.547287,0.546668,0.54608,0.545516,0.544972,0.544441,0.543919,0.543398,0.542873,0.542339,0.541789,0.541217,0.540617,0.53998,0.539299,0.538567,0.537775,0.536912,0.535969,0.534934,0.533796,0.53254,0.531152,0.529615,0.527913,0.526026,0.523932,0.521609,0.519031,0.516171,0.512999,0.509485,0.505592,0.501285,0.496526,0.491271,0.48548,0.479108,0.472109,0.464441,0.456057,0.446916,0.436981,0.426218},
	{1.0063,1.00644,1.00658,1.00672,1.00686,1.00699,1.00712,1.00724,1.00736,1.00747,1.00758,1.00768,1.00777,1.00785,1.00793,1.008,1.00806,1.00811,1.00815,1.00818,1.00819,1.0082,1.00819,1.00817,1.00813,1.00808,1.00801,1.00793,1.00783,1.00771,1.00757,1.00742,1.00724,1.00704,1.00681,1.00657,1.00629,1.00599,1.00566,1.00531,1.00492,1.0045,1.00404,1.00355,1.00302,1.00245,1.00184,1.00119,1.00049,0.999739,0.998939,0.998085,0.997174,0.996204,0.995172,0.994073,0.992906,0.991666,0.99035,0.988954,0.987474,0.985905,0.984245,0.982488,0.98063,0.978666,0.976592,0.974403,0.972093,0.969658,0.967093,0.964393,0.961552,0.958566,0.955428,0.952135,0.948681,0.945062,0.941272,0.937308,0.933164,0.928838,0.924325,0.919623,0.914729,0.909641,0.904357,0.898876,0.893199,0.887326,0.881258,0.874998,0.86855,0.861918,0.855106,0.848123,0.840975,0.833671,0.826221,0.818637,0.81093,0.803114,0.795202,0.787211,0.779157,0.771055,0.762925,0.754784,0.746651,0.738544,0.730482,0.722485,0.714572,0.706759,0.699066,0.691509,0.684104,0.676867,0.669812,0.662951,0.656296,0.649857,0.643642,0.63766,0.631915,0.626411,0.621151,0.616137,0.611368,0.606842,0.602557,0.59851,0.594694,0.591105,0.587736,0.58458,0.581629,0.578874,0.576306,0.573918,0.5717,0.569642,0.567736,0.565971,0.56434,0.562833,0.561441,0.560156,0.558969,0.557873,0.556859,0.55592,0.555049,0.554238,0.553482,0.552772,0.552104,0.55147,0.550865,0.550283,0.549717,0.549162,0.548612,0.548061,0.547503,0.54693,0.546338,0.545718,0.545064,0.544368,0.543622,0.542816,0.541941,0.540987,0.539942,0.538795,0.53753,0.536135,0.534592,0.532884,0.530991,0.528893,0.526566,0.523986,0.521124,0.517952,0.514438,0.510547,0.506243,0.501488,0.496239,0.490456,0.484093,0.477105,0.469449,0.461079,0.451953,0.442034,0.431286,0.419685},
	{1.00697,1.00715,1.00733,1.0075,1.00767,1.00784,1.00801,1.00817,1.00832,1.00847,1.00862,1.00876,1.0089,1.00902,1.00914,1.00925,1.00935,1.00945,1.00953,1.0096,1.00966,1.00971,1.00974,1.00976,1.00977,1.00976,1.00973,1.00969,1.00963,1.00955,1.00945,1.00934,1.0092,1.00903,1.00885,1.00863,1.0084,1.00813,1.00784,1.00752,1.00716,1.00677,1.00635,1.00589,1.00539,1.00485,1.00428,1.00365,1.00298,1.00226,1.00149,1.00066,0.999781,0.99884,0.997835,0.996765,0.995625,0.994413,0.993124,0.991756,0.990303,0.988763,0.98713,0.985401,0.983572,0.981637,0.979591,0.977431,0.975151,0.972746,0.970211,0.967541,0.964731,0.961776,0.95867,0.955409,0.951988,0.948401,0.944645,0.940714,0.936605,0.932313,0.927835,0.923168,0.91831,0.913257,0.908009,0.902565,0.896925,0.891089,0.885058,0.878836,0.872425,0.86583,0.859056,0.85211,0.844999,0.837733,0.83032,0.822773,0.815103,0.807323,0.799447,0.791492,0.783472,0.775406,0.767309,0.759201,0.751101,0.743026,0.734995,0.727029,0.719144,0.711361,0.703695,0.696165,0.688786,0.681574,0.674542,0.667704,0.661071,0.654653,0.648459,0.642495,0.636768,0.631281,0.626038,0.621039,0.616284,0.611771,0.607499,0.603463,0.599658,0.596079,0.592719,0.589571,0.586626,0.583878,0.581317,0.578934,0.57672,0.574666,0.572763,0.571002,0.569372,0.567866,0.566475,0.56519,0.564002,0.562904,0.561888,0.560946,0.560071,0.559256,0.558494,0.557778,0.557102,0.556459,0.555844,0.55525,0.554671,0.554101,0.553534,0.552964,0.552384,0.551789,0.55117,0.55052,0.549833,0.5491,0.548313,0.547461,0.546535,0.545524,0.544417,0.543199,0.541857,0.540375,0.538737,0.536923,0.534913,0.532686,0.530217,0.527479,0.524445,0.521084,0.517362,0.513244,0.508693,0.503668,0.498127,0.492028,0.485325,0.477975,0.46993,0.461149,0.45159,0.441218,0.430001,0.417918},
	{1.00761,1.00782,1.00803,1.00824,1.00845,1.00865,1.00886,1.00906,1.00925,1.00945,1.00963,1.00982,1.00999,1.01016,1.01033,1.01048,1.01063,1.01077,1.0109,1.01101,1.01112,1.01121,1.01129,1.01136,1.01141,1.01145,1.01147,1.01148,1.01146,1.01143,1.01137,1.0113,1.0112,1.01108,1.01094,1.01077,1.01057,1.01035,1.0101,1.00981,1.00949,1.00914,1.00876,1.00834,1.00787,1.00737,1.00683,1.00624,1.0056,1.00491,1.00417,1.00338,1.00253,1.00162,1.00064,0.999603,0.998493,0.997311,0.996052,0.994713,0.993291,0.99178,0.990177,0.988478,0.986677,0.984772,0.982757,0.980626,0.978376,0.976002,0.973498,0.970859,0.96808,0.965156,0.962083,0.958854,0.955465,0.951911,0.948188,0.94429,0.940215,0.935957,0.931514,0.926882,0.922059,0.917042,0.91183,0.906421,0.900817,0.895017,0.889023,0.882837,0.876463,0.869905,0.863168,0.856258,0.849184,0.841954,0.834578,0.827066,0.819432,0.811687,0.803847,0.795926,0.78794,0.779907,0.771843,0.763768,0.755699,0.747655,0.739654,0.731717,0.723861,0.716105,0.708466,0.700962,0.693608,0.68642,0.679411,0.672595,0.665983,0.659585,0.65341,0.647464,0.641754,0.636284,0.631055,0.62607,0.621329,0.616829,0.612568,0.608542,0.604747,0.601177,0.597825,0.594684,0.591746,0.589003,0.586447,0.584068,0.581858,0.579806,0.577905,0.576144,0.574515,0.573009,0.571616,0.570328,0.569137,0.568035,0.567014,0.566065,0.565183,0.564359,0.563586,0.562858,0.562169,0.561511,0.560878,0.560264,0.559664,0.559069,0.558474,0.557873,0.557259,0.556624,0.555962,0.555265,0.554524,0.553731,0.552877,0.551951,0.550943,0.54984,0.54863,0.547299,0.545832,0.544211,0.542418,0.540433,0.538235,0.535799,0.5331,0.53011,0.526798,0.523131,0.519075,0.514593,0.509644,0.504187,0.49818,0.491578,0.484335,0.476407,0.467749,0.45832,0.448084,0.437006,0.425064,0.412243},
	{1.0082,1.00845,1.00869,1.00894,1.00918,1.00943,1.00967,1.00991,1.01015,1.01038,1.01061,1.01084,1.01106,1.01128,1.01149,1.01169,1.01188,1.01207,1.01224,1.01241,1.01257,1.01271,1.01284,1.01296,1.01306,1.01315,1.01322,1.01327,1.01331,1.01333,1.01332,1.0133,1.01325,1.01318,1.01308,1.01296,1.01281,1.01263,1.01242,1.01218,1.01191,1.0116,1.01126,1.01087,1.01045,1.00999,1.00948,1.00893,1.00833,1.00768,1.00698,1.00622,1.0054,1.00453,1.00359,1.00258,1.00151,1.00036,0.99913,0.997823,0.996432,0.994953,0.993381,0.991714,0.989945,0.988071,0.986086,0.983987,0.981769,0.979426,0.976953,0.974346,0.971599,0.968708,0.965666,0.96247,0.959114,0.955593,0.951903,0.948039,0.943998,0.939774,0.935365,0.930768,0.925979,0.920997,0.915821,0.910448,0.904879,0.899115,0.893157,0.887007,0.880669,0.874147,0.867445,0.860572,0.853534,0.846339,0.838998,0.831522,0.823922,0.816212,0.808406,0.800518,0.792566,0.784565,0.776534,0.76849,0.760451,0.752437,0.744466,0.736557,0.728729,0.721,0.713387,0.705908,0.698578,0.691413,0.684427,0.677632,0.671041,0.664662,0.658505,0.652578,0.646884,0.641429,0.636216,0.631245,0.626516,0.622028,0.617779,0.613764,0.609978,0.606417,0.603073,0.59994,0.597009,0.594272,0.591722,0.589347,0.587141,0.585093,0.583194,0.581436,0.579808,0.578303,0.57691,0.575622,0.57443,0.573326,0.572302,0.571351,0.570464,0.569635,0.568856,0.568122,0.567424,0.566757,0.566114,0.565489,0.564875,0.564265,0.563654,0.563034,0.562398,0.56174,0.561051,0.560324,0.55955,0.558721,0.557826,0.556855,0.555796,0.554637,0.553366,0.551966,0.550422,0.548717,0.54683,0.544743,0.542431,0.53987,0.537033,0.533892,0.530414,0.526566,0.522312,0.517614,0.512431,0.506721,0.50044,0.493544,0.485987,0.477724,0.468712,0.458911,0.448286,0.436806,0.424452,0.411212},
	{1.00876,1.00904,1.00932,1.0096,1.00988,1.01016,1.01044,1.01072,1.011,1.01127,1.01155,1.01182,1.01209,1.01235,1.01261,1.01286,1.0131,1.01334,1.01357,1.01378,1.01399,1.01419,1.01437,1.01454,1.0147,1.01484,1.01497,1.01508,1.01516,1.01524,1.01528,1.01531,1.01532,1.0153,1.01526,1.01518,1.01509,1.01496,1.0148,1.01461,1.01438,1.01413,1.01383,1.01349,1.01312,1.0127,1.01224,1.01173,1.01117,1.01056,1.0099,1.00918,1.0084,1.00757,1.00666,1.00569,1.00465,1.00354,1.00235,1.00108,0.999722,0.998277,0.99674,0.995105,0.99337,0.991529,0.989578,0.987512,0.985326,0.983016,0.980576,0.978001,0.975287,0.972429,0.96942,0.966257,0.962934,0.959447,0.95579,0.95196,0.947953,0.943763,0.939389,0.934826,0.930072,0.925125,0.919983,0.914645,0.909112,0.903383,0.89746,0.891346,0.885043,0.878556,0.87189,0.865052,0.858048,0.850888,0.843582,0.83614,0.828574,0.820897,0.813124,0.80527,0.797349,0.78938,0.78138,0.773366,0.765357,0.757372,0.74943,0.741548,0.733747,0.726043,0.718455,0.711,0.703694,0.696551,0.689586,0.682812,0.67624,0.66988,0.66374,0.657829,0.652151,0.646711,0.641512,0.636554,0.631837,0.62736,0.623121,0.619116,0.615339,0.611786,0.608449,0.605322,0.602397,0.599665,0.597118,0.594747,0.592543,0.590497,0.588599,0.586841,0.585212,0.583705,0.58231,0.581019,0.579822,0.578713,0.577683,0.576723,0.575827,0.574988,0.574198,0.573449,0.572736,0.572052,0.571389,0.570742,0.570103,0.569465,0.568823,0.568168,0.567494,0.566792,0.566055,0.565275,0.564441,0.563545,0.562576,0.561523,0.560373,0.559113,0.557728,0.556204,0.554522,0.552663,0.550608,0.548333,0.545815,0.543027,0.539941,0.536525,0.532746,0.52857,0.523957,0.518869,0.513263,0.507095,0.500322,0.492897,0.484777,0.475916,0.466273,0.455812,0.444502,0.432318,0.419248,0.405291},
	{1.00928,1.00959,1.0099,1.01021,1.01053,1.01085,1.01117,1.01149,1.01181,1.01213,1.01245,1.01276,1.01308,1.01339,1.01369,1.01399,1.01429,1.01458,1.01486,1.01513,1.01539,1.01564,1.01588,1.01611,1.01632,1.01652,1.0167,1.01687,1.01702,1.01715,1.01725,1.01734,1.0174,1.01744,1.01746,1.01744,1.0174,1.01733,1.01722,1.01709,1.01692,1.01671,1.01647,1.01618,1.01586,1.01549,1.01508,1.01461,1.0141,1.01354,1.01292,1.01225,1.01152,1.01072,1.00986,1.00893,1.00793,1.00686,1.00571,1.00447,1.00316,1.00175,1.00025,0.998649,0.996949,0.995144,0.993228,0.991197,0.989046,0.98677,0.984364,0.981824,0.979144,0.976319,0.973345,0.970216,0.966927,0.963474,0.959852,0.956056,0.952082,0.947927,0.943587,0.939059,0.93434,0.929427,0.92432,0.919017,0.913519,0.907825,0.901937,0.895857,0.889589,0.883137,0.876506,0.869702,0.862733,0.855607,0.848335,0.840926,0.833393,0.825749,0.818009,0.810186,0.802297,0.794359,0.786388,0.778404,0.770424,0.762467,0.754552,0.746698,0.738922,0.731243,0.72368,0.716248,0.708964,0.701843,0.694899,0.688144,0.681591,0.675249,0.669127,0.663232,0.657569,0.652144,0.646958,0.642012,0.637308,0.632842,0.628613,0.624618,0.62085,0.617304,0.613975,0.610855,0.607935,0.605209,0.602667,0.6003,0.598099,0.596056,0.59416,0.592403,0.590776,0.589269,0.587873,0.586581,0.585383,0.584271,0.583237,0.582274,0.581373,0.580528,0.579731,0.578975,0.578253,0.577558,0.576884,0.576223,0.575569,0.574915,0.574254,0.573579,0.572881,0.572154,0.571388,0.570575,0.569706,0.56877,0.567756,0.566654,0.565449,0.564129,0.562677,0.561078,0.559314,0.557365,0.55521,0.552825,0.550186,0.547265,0.544033,0.540456,0.536503,0.532136,0.527316,0.522003,0.516153,0.509723,0.502669,0.494943,0.486503,0.477303,0.467305,0.456474,0.44478,0.432203,0.418734,0.404379},
	{1.00976,1.0101,1.01045,1.0108,1.01115,1.0115,1.01186,1.01222,1.01258,1.01294,1.01331,1.01367,1.01403,1.01439,1.01474,1.01509,1.01544,1.01578,1.01611,1.01644,1.01676,1.01707,1.01736,1.01765,1.01792,1.01818,1.01842,1.01865,1.01886,1.01905,1.01922,1.01937,1.01949,1.01959,1.01967,1.01972,1.01973,1.01972,1.01968,1.0196,1.01949,1.01934,1.01916,1.01893,1.01866,1.01835,1.01799,1.01758,1.01712,1.01661,1.01604,1.01542,1.01473,1.01399,1.01317,1.01229,1.01134,1.0103,1.0092,1.008,1.00673,1.00536,1.0039,1.00234,1.00068,0.998911,0.997033,0.995039,0.992925,0.990685,0.988316,0.985811,0.983167,0.980378,0.977439,0.974345,0.971091,0.967672,0.964085,0.960324,0.956386,0.952265,0.94796,0.943466,0.938782,0.933904,0.928832,0.923564,0.9181,0.912441,0.906587,0.900542,0.894309,0.887891,0.881293,0.874523,0.867588,0.860495,0.853256,0.84588,0.838379,0.830768,0.823058,0.815266,0.807408,0.799499,0.791558,0.783603,0.77565,0.767721,0.759832,0.752003,0.744252,0.736598,0.729057,0.721647,0.714385,0.707284,0.70036,0.693624,0.687089,0.680764,0.674658,0.668778,0.66313,0.657717,0.652544,0.64761,0.642916,0.638461,0.634241,0.630254,0.626494,0.622955,0.619632,0.616517,0.613602,0.610879,0.608339,0.605975,0.603775,0.601733,0.599837,0.598079,0.596449,0.594939,0.59354,0.592242,0.591038,0.589919,0.588877,0.587904,0.586992,0.586134,0.585322,0.58455,0.583809,0.583093,0.582395,0.581708,0.581025,0.580338,0.57964,0.578923,0.578179,0.577401,0.576578,0.575702,0.574762,0.573748,0.572647,0.571448,0.570136,0.568697,0.567114,0.565369,0.563444,0.561317,0.558965,0.556364,0.553486,0.550303,0.546783,0.542892,0.538595,0.533853,0.528625,0.52287,0.516544,0.509601,0.501996,0.493685,0.484623,0.474769,0.464087,0.452546,0.440123,0.426807,0.412596,0.39751},
	{1.01022,1.01059,1.01096,1.01134,1.01173,1.01212,1.01251,1.01291,1.01331,1.01372,1.01412,1.01453,1.01494,1.01534,1.01575,1.01615,1.01655,1.01694,1.01733,1.01771,1.01809,1.01846,1.01881,1.01916,1.01949,1.01981,1.02012,1.02041,1.02068,1.02094,1.02117,1.02139,1.02158,1.02175,1.02189,1.022,1.02208,1.02214,1.02216,1.02215,1.0221,1.02202,1.02189,1.02173,1.02152,1.02127,1.02097,1.02062,1.02022,1.01976,1.01925,1.01868,1.01805,1.01735,1.01659,1.01576,1.01485,1.01387,1.01281,1.01166,1.01043,1.00911,1.00769,1.00617,1.00455,1.00283,1.00099,0.999035,0.99696,0.994759,0.992428,0.989961,0.987355,0.984603,0.9817,0.978643,0.975425,0.972043,0.968491,0.964766,0.960863,0.956778,0.952508,0.94805,0.9434,0.938558,0.93352,0.928287,0.922858,0.917233,0.911414,0.905403,0.899203,0.892819,0.886255,0.879519,0.872616,0.865556,0.858349,0.851005,0.843536,0.835956,0.828277,0.820515,0.812686,0.804806,0.796894,0.788965,0.78104,0.773137,0.765273,0.757469,0.749742,0.742111,0.734592,0.727204,0.719962,0.712881,0.705975,0.699257,0.692739,0.68643,0.680339,0.674473,0.668839,0.663439,0.658277,0.653354,0.648671,0.644224,0.640013,0.636033,0.63228,0.628747,0.625429,0.622318,0.619406,0.616686,0.614149,0.611785,0.609586,0.607542,0.605644,0.603884,0.602251,0.600736,0.599331,0.598027,0.596814,0.595685,0.594632,0.593646,0.59272,0.591845,0.591014,0.59022,0.589456,0.588714,0.587986,0.587265,0.586545,0.585816,0.585072,0.584303,0.583502,0.58266,0.581766,0.58081,0.579783,0.578671,0.577462,0.576142,0.574697,0.57311,0.571364,0.569438,0.567313,0.564966,0.562371,0.559503,0.556331,0.552825,0.548952,0.544675,0.539956,0.534756,0.529032,0.52274,0.515836,0.508274,0.500009,0.490997,0.481195,0.470567,0.459082,0.446714,0.433449,0.419286,0.40424,0.388342},
	{1.01064,1.01104,1.01144,1.01185,1.01227,1.0127,1.01313,1.01357,1.01401,1.01445,1.0149,1.01535,1.01581,1.01626,1.01671,1.01717,1.01762,1.01807,1.01851,1.01895,1.01938,1.01981,1.02023,1.02064,1.02103,1.02142,1.02179,1.02215,1.02249,1.02281,1.02311,1.02339,1.02365,1.02389,1.0241,1.02428,1.02444,1.02456,1.02466,1.02471,1.02473,1.02472,1.02466,1.02456,1.02442,1.02424,1.024,1.02371,1.02338,1.02298,1.02253,1.02202,1.02145,1.02081,1.0201,1.01933,1.01847,1.01754,1.01653,1.01544,1.01426,1.01298,1.01161,1.01014,1.00857,1.00689,1.00509,1.00318,1.00115,0.99899,0.9967,0.994274,0.991707,0.988994,0.98613,0.983111,0.979931,0.976587,0.973072,0.969384,0.965517,0.961469,0.957235,0.952813,0.948199,0.943392,0.938389,0.933191,0.927797,0.922206,0.916422,0.910445,0.904279,0.897929,0.891398,0.884695,0.877825,0.870798,0.863622,0.85631,0.848872,0.841322,0.833673,0.825941,0.818141,0.81029,0.802405,0.794503,0.786605,0.778727,0.770888,0.763108,0.755404,0.747796,0.740299,0.732932,0.72571,0.718648,0.711761,0.705061,0.698559,0.692266,0.686191,0.68034,0.674718,0.669332,0.664182,0.659271,0.654597,0.650161,0.645959,0.641988,0.638242,0.634717,0.631406,0.628301,0.625395,0.62268,0.620147,0.617788,0.615592,0.613552,0.611657,0.609899,0.608268,0.606755,0.605351,0.604047,0.602835,0.601706,0.600652,0.599666,0.598738,0.597861,0.597028,0.596231,0.595464,0.594717,0.593985,0.593259,0.592533,0.591798,0.591046,0.590269,0.589459,0.588606,0.5877,0.586732,0.58569,0.584563,0.583337,0.581998,0.580532,0.578922,0.57715,0.575197,0.573042,0.570662,0.568031,0.565123,0.561908,0.558356,0.554432,0.5501,0.545323,0.540059,0.534267,0.527902,0.52092,0.513274,0.50492,0.495815,0.485915,0.475183,0.463589,0.451108,0.437725,0.423441,0.408269,0.392241},
	{1.01103,1.01146,1.01189,1.01233,1.01279,1.01325,1.01371,1.01419,1.01467,1.01515,1.01564,1.01614,1.01664,1.01714,1.01764,1.01814,1.01865,1.01915,1.01965,1.02015,1.02064,1.02112,1.0216,1.02208,1.02254,1.02299,1.02343,1.02385,1.02426,1.02465,1.02503,1.02538,1.02571,1.02602,1.02631,1.02656,1.02679,1.02699,1.02716,1.02729,1.02738,1.02744,1.02746,1.02743,1.02736,1.02724,1.02708,1.02686,1.02659,1.02627,1.02588,1.02544,1.02493,1.02435,1.02371,1.02299,1.0222,1.02132,1.02037,1.01933,1.0182,1.01698,1.01566,1.01424,1.01272,1.01109,1.00934,1.00747,1.00549,1.00337,1.00113,0.998743,0.996218,0.993547,0.990724,0.987745,0.984606,0.9813,0.977825,0.974174,0.970346,0.966335,0.962138,0.957752,0.953175,0.948404,0.943437,0.938274,0.932915,0.927359,0.921609,0.915667,0.909535,0.903218,0.89672,0.890049,0.883212,0.876216,0.869073,0.861791,0.854384,0.846863,0.839244,0.83154,0.823768,0.815945,0.808087,0.800212,0.792338,0.784485,0.776671,0.768914,0.761232,0.753645,0.74617,0.738822,0.731619,0.724576,0.717706,0.711023,0.704537,0.698259,0.692198,0.68636,0.680752,0.675377,0.670239,0.665338,0.660674,0.656247,0.652053,0.648089,0.644351,0.640832,0.637526,0.634426,0.631525,0.628814,0.626284,0.623927,0.621734,0.619695,0.617801,0.616043,0.614411,0.612897,0.611491,0.610185,0.60897,0.607837,0.606779,0.605786,0.604852,0.603968,0.603126,0.60232,0.601541,0.600782,0.600035,0.599294,0.59855,0.597795,0.597021,0.596219,0.595382,0.594498,0.593559,0.592553,0.59147,0.590296,0.589019,0.587623,0.586095,0.584416,0.582568,0.580531,0.578283,0.5758,0.573058,0.570027,0.566678,0.562979,0.558895,0.554389,0.549422,0.543953,0.537939,0.531336,0.524098,0.516179,0.507535,0.498123,0.487901,0.476833,0.464891,0.452052,0.438306,0.423657,0.408122,0.391742},
	{1.01139,1.01185,1.01231,1.01278,1.01327,1.01376,1.01426,1.01477,1.01529,1.01581,1.01634,1.01688,1.01743,1.01797,1.01853,1.01908,1.01963,1.02019,1.02075,1.0213,1.02185,1.0224,1.02294,1.02348,1.02401,1.02452,1.02503,1.02552,1.026,1.02647,1.02692,1.02734,1.02775,1.02814,1.0285,1.02883,1.02914,1.02941,1.02966,1.02986,1.03004,1.03017,1.03027,1.03032,1.03032,1.03028,1.03019,1.03005,1.02986,1.0296,1.02929,1.02891,1.02848,1.02797,1.02739,1.02674,1.02601,1.0252,1.02431,1.02333,1.02226,1.02109,1.01983,1.01847,1.017,1.01542,1.01372,1.01191,1.00997,1.0079,1.0057,1.00337,1.00089,0.998259,0.99548,0.992544,0.989446,0.986181,0.982746,0.979136,0.975347,0.971375,0.967216,0.962868,0.958328,0.953593,0.948663,0.943536,0.938212,0.932692,0.926976,0.921068,0.91497,0.908686,0.902222,0.895583,0.888777,0.881813,0.8747,0.867449,0.860072,0.85258,0.84499,0.837314,0.829569,0.821772,0.81394,0.80609,0.798242,0.790412,0.782621,0.774886,0.767226,0.759659,0.752203,0.744875,0.73769,0.730664,0.72381,0.717142,0.710671,0.704407,0.698359,0.692533,0.686936,0.681572,0.676443,0.671551,0.666896,0.662476,0.658289,0.654331,0.650597,0.647082,0.64378,0.640683,0.637784,0.635074,0.632544,0.630187,0.627992,0.625951,0.624053,0.622291,0.620654,0.619133,0.61772,0.616404,0.615179,0.614034,0.612962,0.611953,0.611001,0.610097,0.609233,0.608402,0.607595,0.606804,0.606023,0.605243,0.604455,0.603651,0.602824,0.601962,0.601058,0.600101,0.599079,0.597982,0.596798,0.595512,0.59411,0.592578,0.590897,0.58905,0.587016,0.584774,0.5823,0.579568,0.576552,0.573221,0.569543,0.565484,0.561007,0.556073,0.550642,0.544671,0.538114,0.530928,0.523066,0.514483,0.505136,0.494983,0.483986,0.472115,0.459348,0.445671,0.431084,0.415604,0.399264,0.382119},
	{1.01173,1.01221,1.0127,1.0132,1.01372,1.01424,1.01478,1.01532,1.01588,1.01644,1.01701,1.01759,1.01818,1.01877,1.01937,1.01998,1.02058,1.02119,1.0218,1.02242,1.02303,1.02363,1.02424,1.02484,1.02543,1.02602,1.0266,1.02716,1.02771,1.02825,1.02878,1.02928,1.02976,1.03023,1.03067,1.03108,1.03147,1.03182,1.03215,1.03244,1.03269,1.03291,1.03309,1.03322,1.03331,1.03335,1.03334,1.03328,1.03316,1.03298,1.03275,1.03245,1.03208,1.03165,1.03114,1.03056,1.0299,1.02916,1.02834,1.02743,1.02642,1.02532,1.02412,1.02281,1.0214,1.01988,1.01824,1.01648,1.01459,1.01258,1.01043,1.00814,1.00571,1.00313,1.0004,0.997506,0.994452,0.991231,0.987839,0.984271,0.980522,0.976591,0.972472,0.968163,0.963661,0.958964,0.954071,0.948981,0.943693,0.938209,0.932528,0.926654,0.92059,0.91434,0.907908,0.901302,0.894528,0.887596,0.880513,0.873292,0.865944,0.858482,0.850919,0.843271,0.835553,0.827782,0.819975,0.81215,0.804325,0.796519,0.788749,0.781036,0.773397,0.76585,0.758413,0.751103,0.743936,0.736926,0.730088,0.723435,0.716979,0.710728,0.704692,0.698878,0.693292,0.687938,0.682819,0.677936,0.673288,0.668876,0.664695,0.660743,0.657015,0.653505,0.650206,0.647112,0.644215,0.641507,0.638979,0.636622,0.634426,0.632384,0.630485,0.62872,0.627079,0.625554,0.624135,0.622813,0.62158,0.620426,0.619344,0.618324,0.617359,0.61644,0.61556,0.614709,0.613881,0.613068,0.61226,0.611451,0.610631,0.609791,0.608923,0.608017,0.607064,0.606051,0.604969,0.603804,0.602545,0.601176,0.599683,0.598049,0.596256,0.594285,0.592115,0.589723,0.587083,0.584171,0.580955,0.577406,0.57349,0.569171,0.564411,0.55917,0.553406,0.547076,0.540134,0.532534,0.524231,0.515181,0.50534,0.494669,0.483134,0.470708,0.457374,0.443124,0.427965,0.411925,0.395046,0.377391},
	{1.01205,1.01255,1.01307,1.0136,1.01414,1.01469,1.01526,1.01584,1.01643,1.01703,1.01764,1.01826,1.0189,1.01953,1.02018,1.02083,1.02149,1.02215,1.02282,1.02349,1.02416,1.02483,1.0255,1.02616,1.02682,1.02748,1.02812,1.02876,1.02939,1.03,1.0306,1.03118,1.03175,1.03229,1.03281,1.03331,1.03378,1.03422,1.03463,1.03501,1.03535,1.03565,1.03591,1.03613,1.0363,1.03643,1.03651,1.03653,1.03649,1.0364,1.03625,1.03603,1.03575,1.03539,1.03496,1.03446,1.03388,1.03321,1.03246,1.03161,1.03068,1.02964,1.02851,1.02727,1.02592,1.02446,1.02288,1.02118,1.01935,1.01739,1.0153,1.01306,1.01068,1.00815,1.00547,1.00263,0.999622,0.996447,0.9931,0.989575,0.98587,0.981981,0.977904,0.973635,0.969173,0.964516,0.959661,0.954609,0.949358,0.94391,0.938265,0.932426,0.926396,0.92018,0.913781,0.907207,0.900466,0.893564,0.886512,0.879321,0.872002,0.864568,0.857033,0.849412,0.84172,0.833975,0.826192,0.818391,0.810589,0.802805,0.795058,0.787365,0.779746,0.772218,0.764799,0.757507,0.750356,0.743362,0.73654,0.729901,0.723458,0.71722,0.711196,0.705393,0.699817,0.694473,0.689362,0.684487,0.679847,0.675441,0.671267,0.66732,0.663596,0.66009,0.656795,0.653704,0.650809,0.648102,0.645574,0.643216,0.64102,0.638976,0.637075,0.635306,0.633661,0.632131,0.630706,0.629377,0.628135,0.626972,0.625878,0.624846,0.623866,0.622931,0.622033,0.621163,0.620312,0.619473,0.618638,0.617797,0.616942,0.616064,0.615153,0.614199,0.613192,0.612121,0.610973,0.609737,0.608397,0.60694,0.605349,0.603607,0.601695,0.599592,0.597277,0.594726,0.591912,0.588807,0.585381,0.581601,0.577434,0.572841,0.567783,0.56222,0.556108,0.549402,0.542058,0.534028,0.525268,0.515734,0.505385,0.494183,0.482098,0.469108,0.455199,0.440373,0.424645,0.40805,0.390641,0.372496},
	{1.01235,1.01287,1.01341,1.01397,1.01454,1.01512,1.01572,1.01633,1.01695,1.01759,1.01824,1.0189,1.01958,1.02026,1.02095,1.02165,1.02236,1.02308,1.0238,1.02452,1.02525,1.02598,1.02671,1.02744,1.02817,1.02889,1.02961,1.03032,1.03102,1.03171,1.03239,1.03305,1.0337,1.03432,1.03493,1.03551,1.03607,1.03659,1.03709,1.03756,1.03799,1.03838,1.03873,1.03904,1.03931,1.03952,1.03969,1.0398,1.03985,1.03985,1.03979,1.03965,1.03946,1.03919,1.03884,1.03842,1.03792,1.03733,1.03665,1.03589,1.03503,1.03407,1.033,1.03183,1.03055,1.02916,1.02764,1.02601,1.02424,1.02234,1.02031,1.01813,1.01581,1.01333,1.0107,1.00791,1.00495,1.00183,0.998528,0.99505,0.991391,0.987546,0.983512,0.979286,0.974866,0.970249,0.965435,0.960421,0.955208,0.949797,0.944189,0.938386,0.932391,0.926209,0.919844,0.913303,0.906593,0.899722,0.892701,0.885539,0.878249,0.870844,0.863336,0.855742,0.848076,0.840355,0.832597,0.82482,0.817041,0.809278,0.801552,0.793879,0.786279,0.77877,0.771369,0.764094,0.756959,0.749981,0.743173,0.736548,0.730117,0.723892,0.717879,0.712087,0.706522,0.701186,0.696085,0.691217,0.686585,0.682186,0.678017,0.674076,0.670357,0.666855,0.663564,0.660476,0.657583,0.654878,0.652352,0.649995,0.6478,0.645755,0.643852,0.642082,0.640435,0.638902,0.637473,0.63614,0.634892,0.633723,0.632622,0.631582,0.630593,0.629647,0.628737,0.627853,0.626987,0.626132,0.625278,0.624416,0.623538,0.622634,0.621694,0.620708,0.619666,0.618556,0.617365,0.616081,0.614688,0.613173,0.611517,0.609704,0.607714,0.605526,0.603116,0.60046,0.597532,0.594303,0.590741,0.586813,0.582484,0.577715,0.572468,0.566699,0.560365,0.553421,0.545822,0.537521,0.528473,0.518635,0.507967,0.496433,0.484004,0.470661,0.456394,0.441207,0.425121,0.408176,0.39043,0.371968},
	{1.01262,1.01317,1.01373,1.01431,1.01491,1.01552,1.01615,1.01679,1.01745,1.01812,1.01881,1.01951,1.02022,1.02095,1.02169,1.02244,1.02319,1.02396,1.02474,1.02552,1.0263,1.02709,1.02789,1.02868,1.02948,1.03027,1.03106,1.03184,1.03262,1.03339,1.03414,1.03489,1.03561,1.03632,1.03701,1.03768,1.03833,1.03895,1.03953,1.04009,1.04061,1.0411,1.04154,1.04195,1.04231,1.04262,1.04288,1.04308,1.04323,1.04332,1.04335,1.04331,1.0432,1.04302,1.04277,1.04243,1.04202,1.04152,1.04093,1.04024,1.03946,1.03858,1.03759,1.0365,1.03529,1.03397,1.03253,1.03096,1.02926,1.02742,1.02545,1.02333,1.02107,1.01865,1.01608,1.01334,1.01044,1.00737,1.00412,1.00069,0.997081,0.993283,0.989295,0.985114,0.980737,0.976163,0.971389,0.966416,0.961243,0.95587,0.950299,0.944533,0.938574,0.932426,0.926095,0.919587,0.912909,0.90607,0.899079,0.891947,0.884686,0.877308,0.869828,0.862259,0.854619,0.846923,0.839189,0.831434,0.823677,0.815936,0.808229,0.800576,0.792995,0.785504,0.77812,0.77086,0.763741,0.756777,0.749982,0.74337,0.736952,0.730737,0.724736,0.718953,0.713397,0.70807,0.702976,0.698116,0.69349,0.689096,0.684933,0.680996,0.67728,0.673782,0.670493,0.667406,0.664515,0.66181,0.659283,0.656926,0.654728,0.652681,0.650775,0.649001,0.647349,0.64581,0.644374,0.643032,0.641776,0.640595,0.639482,0.638428,0.637424,0.636461,0.635531,0.634626,0.633736,0.632854,0.63197,0.631075,0.63016,0.629215,0.62823,0.627194,0.626096,0.624924,0.623665,0.622304,0.620828,0.61922,0.617463,0.615537,0.613423,0.611098,0.608538,0.605718,0.602609,0.599182,0.595403,0.59124,0.586653,0.581606,0.576055,0.56996,0.563274,0.555952,0.547949,0.539219,0.529717,0.519401,0.508232,0.496179,0.483216,0.469329,0.454513,0.43878,0.422159,0.4047,0.386471,0.367569},
	{1.01288,1.01345,1.01404,1.01464,1.01526,1.0159,1.01655,1.01723,1.01792,1.01862,1.01935,1.02009,1.02084,1.02161,1.02239,1.02318,1.02399,1.02481,1.02564,1.02647,1.02732,1.02817,1.02902,1.02988,1.03074,1.0316,1.03247,1.03332,1.03417,1.03502,1.03586,1.03668,1.03749,1.03829,1.03907,1.03982,1.04056,1.04127,1.04195,1.0426,1.04322,1.0438,1.04434,1.04485,1.0453,1.04571,1.04607,1.04637,1.04662,1.04681,1.04694,1.04699,1.04698,1.0469,1.04674,1.0465,1.04617,1.04576,1.04526,1.04467,1.04397,1.04317,1.04227,1.04126,1.04013,1.03889,1.03752,1.03602,1.03439,1.03263,1.03073,1.02868,1.02648,1.02412,1.02161,1.01893,1.01609,1.01307,1.00988,1.0065,1.00294,0.999193,0.995253,0.991119,0.986788,0.982258,0.977528,0.972597,0.967465,0.962132,0.9566,0.950871,0.944948,0.938836,0.93254,0.926065,0.91942,0.912613,0.905652,0.89855,0.891318,0.883968,0.876515,0.868972,0.861357,0.853686,0.845975,0.838243,0.830507,0.822787,0.8151,0.807466,0.799903,0.792429,0.785061,0.777817,0.770713,0.763763,0.756982,0.750382,0.743975,0.737772,0.73178,0.726007,0.72046,0.715141,0.710054,0.705201,0.700581,0.696193,0.692035,0.688103,0.684391,0.680896,0.67761,0.674526,0.671636,0.668933,0.666407,0.664049,0.661851,0.659803,0.657895,0.656119,0.654464,0.652921,0.65148,0.650133,0.648871,0.647683,0.646562,0.645499,0.644485,0.64351,0.642567,0.641647,0.640742,0.639841,0.638937,0.638019,0.637079,0.636107,0.635091,0.634022,0.632886,0.631672,0.630367,0.628957,0.627425,0.625755,0.62393,0.62193,0.619734,0.617319,0.614661,0.611732,0.608505,0.604949,0.60103,0.596713,0.59196,0.586732,0.580987,0.574681,0.56777,0.560208,0.551948,0.542944,0.533154,0.522535,0.51105,0.498669,0.485369,0.471137,0.455973,0.439893,0.422931,0.40514,0.386594,0.367396},
	{1.01312,1.01371,1.01432,1.01494,1.01559,1.01625,1.01694,1.01764,1.01836,1.0191,1.01986,1.02063,1.02142,1.02223,1.02306,1.0239,1.02475,1.02562,1.0265,1.02739,1.02829,1.0292,1.03012,1.03104,1.03197,1.0329,1.03383,1.03476,1.03569,1.03661,1.03753,1.03844,1.03933,1.04021,1.04108,1.04193,1.04276,1.04356,1.04434,1.04509,1.0458,1.04648,1.04713,1.04773,1.04829,1.0488,1.04926,1.04967,1.05002,1.05031,1.05054,1.0507,1.05079,1.05081,1.05074,1.0506,1.05038,1.05006,1.04966,1.04915,1.04855,1.04785,1.04703,1.04611,1.04506,1.0439,1.04261,1.0412,1.03965,1.03796,1.03613,1.03415,1.03202,1.02973,1.02728,1.02467,1.02189,1.01893,1.01579,1.01247,1.00897,1.00527,1.00138,0.997299,0.993016,0.988533,0.983848,0.978961,0.973871,0.96858,0.963088,0.957397,0.951512,0.945437,0.939176,0.932736,0.926124,0.919348,0.912419,0.905347,0.898143,0.890822,0.883395,0.875879,0.868289,0.860641,0.852953,0.845243,0.837528,0.829828,0.822161,0.814545,0.806999,0.799542,0.79219,0.78496,0.77787,0.770933,0.764164,0.757576,0.75118,0.744986,0.739004,0.73324,0.7277,0.722388,0.717308,0.712461,0.707846,0.703463,0.699308,0.695379,0.69167,0.688177,0.684892,0.681809,0.678919,0.676215,0.673688,0.671328,0.669127,0.667076,0.665164,0.663382,0.66172,0.66017,0.658721,0.657364,0.656091,0.654891,0.653756,0.652677,0.651645,0.650651,0.649687,0.648743,0.64781,0.646879,0.645941,0.644987,0.644006,0.642988,0.641922,0.640796,0.639599,0.638317,0.636937,0.635442,0.633818,0.632047,0.63011,0.627987,0.625655,0.623091,0.620269,0.617161,0.613737,0.609966,0.605813,0.601241,0.596212,0.590684,0.584615,0.577961,0.570675,0.562712,0.554026,0.544571,0.534304,0.523187,0.511184,0.498268,0.484421,0.469637,0.453921,0.437296,0.419807,0.401515,0.382508,0.362897},
	{1.01335,1.01395,1.01458,1.01523,1.0159,1.01658,1.01729,1.01802,1.01878,1.01955,1.02034,1.02115,1.02198,1.02283,1.02369,1.02458,1.02548,1.02639,1.02732,1.02827,1.02923,1.03019,1.03117,1.03216,1.03315,1.03415,1.03515,1.03616,1.03716,1.03816,1.03916,1.04015,1.04113,1.0421,1.04306,1.044,1.04492,1.04582,1.04669,1.04754,1.04836,1.04914,1.04989,1.05059,1.05126,1.05187,1.05244,1.05296,1.05342,1.05381,1.05415,1.05442,1.05461,1.05474,1.05478,1.05474,1.05462,1.05441,1.0541,1.0537,1.0532,1.05259,1.05187,1.05104,1.05008,1.04901,1.04781,1.04648,1.04501,1.0434,1.04165,1.03975,1.03769,1.03548,1.0331,1.03055,1.02784,1.02494,1.02187,1.01861,1.01516,1.01152,1.00769,1.00366,0.999424,0.99499,0.990353,0.985512,0.980467,0.975218,0.969768,0.964118,0.958273,0.952235,0.94601,0.939605,0.933027,0.926285,0.919387,0.912345,0.905171,0.897878,0.890478,0.882988,0.875423,0.867799,0.860134,0.852446,0.844752,0.837072,0.829424,0.821826,0.814297,0.806856,0.799519,0.792305,0.785228,0.778304,0.771547,0.76497,0.758585,0.752401,0.746428,0.740672,0.73514,0.729836,0.724763,0.719921,0.715312,0.710934,0.706784,0.702859,0.699153,0.695663,0.692381,0.689299,0.686411,0.683708,0.681181,0.678822,0.676621,0.674568,0.672655,0.670871,0.669207,0.667654,0.666201,0.664841,0.663562,0.662357,0.661215,0.660129,0.659089,0.658086,0.657111,0.656155,0.655209,0.654264,0.65331,0.652338,0.651337,0.650297,0.649207,0.648055,0.646829,0.645514,0.644098,0.642565,0.640898,0.639079,0.63709,0.634909,0.632515,0.629883,0.626986,0.623796,0.620284,0.616416,0.612158,0.607472,0.60232,0.596659,0.590448,0.58364,0.57619,0.568053,0.559181,0.549529,0.539056,0.527722,0.515494,0.502344,0.488257,0.473227,0.457263,0.440389,0.422652,0.404116,0.384873,0.365035},
	{1.01356,1.01418,1.01483,1.01549,1.01618,1.0169,1.01763,1.01839,1.01917,1.01997,1.02079,1.02164,1.0225,1.02339,1.0243,1.02522,1.02617,1.02713,1.02811,1.02911,1.03012,1.03115,1.03219,1.03324,1.0343,1.03536,1.03643,1.03751,1.03859,1.03967,1.04075,1.04182,1.04289,1.04395,1.045,1.04603,1.04705,1.04804,1.04902,1.04996,1.05088,1.05177,1.05262,1.05344,1.05421,1.05493,1.05561,1.05624,1.05681,1.05732,1.05776,1.05814,1.05845,1.05869,1.05884,1.05891,1.0589,1.0588,1.0586,1.0583,1.0579,1.05739,1.05677,1.05604,1.05519,1.05421,1.0531,1.05186,1.05048,1.04896,1.04729,1.04547,1.04349,1.04135,1.03905,1.03658,1.03393,1.0311,1.0281,1.0249,1.02151,1.01794,1.01416,1.01018,1.00601,1.00162,0.997036,0.992244,0.987246,0.982042,0.976636,0.971028,0.965223,0.959224,0.953037,0.946668,0.940125,0.933415,0.92655,0.919539,0.912394,0.905129,0.897757,0.890292,0.882752,0.875152,0.867509,0.859842,0.852169,0.844508,0.836878,0.829298,0.821785,0.814359,0.807037,0.799836,0.792771,0.785859,0.779113,0.772546,0.76617,0.759995,0.75403,0.748281,0.742756,0.737457,0.732388,0.727551,0.722946,0.71857,0.714422,0.710499,0.706794,0.703304,0.700021,0.696938,0.694048,0.691342,0.688812,0.686448,0.684242,0.682183,0.680262,0.678469,0.676796,0.675231,0.673766,0.672391,0.671097,0.669874,0.668712,0.667604,0.666539,0.665508,0.664502,0.663512,0.662529,0.661542,0.660542,0.659518,0.65846,0.657357,0.656197,0.654968,0.653656,0.652247,0.650727,0.649078,0.647284,0.645326,0.643182,0.640832,0.638252,0.635414,0.632293,0.628859,0.625078,0.620917,0.61634,0.611308,0.60578,0.599713,0.593064,0.585786,0.577834,0.56916,0.559719,0.549468,0.538366,0.526377,0.513471,0.499629,0.484839,0.469105,0.452444,0.434895,0.416516,0.397384,0.377606,0.357309},
	{1.01375,1.01439,1.01506,1.01574,1.01646,1.01719,1.01795,1.01873,1.01954,1.02037,1.02122,1.0221,1.023,1.02393,1.02487,1.02584,1.02683,1.02784,1.02887,1.02992,1.03098,1.03207,1.03316,1.03428,1.0354,1.03653,1.03768,1.03883,1.03998,1.04114,1.0423,1.04346,1.04461,1.04576,1.04689,1.04802,1.04913,1.05023,1.0513,1.05235,1.05338,1.05437,1.05533,1.05625,1.05714,1.05798,1.05877,1.05951,1.06019,1.06082,1.06138,1.06188,1.0623,1.06265,1.06292,1.06311,1.06321,1.06322,1.06313,1.06295,1.06266,1.06226,1.06175,1.06112,1.06036,1.05949,1.05848,1.05733,1.05605,1.05462,1.05304,1.05131,1.04941,1.04736,1.04514,1.04274,1.04017,1.03742,1.03448,1.03135,1.02803,1.02452,1.0208,1.01688,1.01276,1.00844,1.0039,0.999162,0.994213,0.989058,0.983697,0.978133,0.97237,0.966412,0.960264,0.953932,0.947425,0.940751,0.933918,0.926939,0.919824,0.912588,0.905243,0.897805,0.89029,0.882713,0.875094,0.867448,0.859795,0.852153,0.844541,0.836978,0.829482,0.822071,0.814763,0.807575,0.800523,0.793623,0.786887,0.78033,0.773964,0.767797,0.761839,0.756098,0.750578,0.745285,0.740222,0.735389,0.730787,0.726415,0.72227,0.718349,0.714646,0.711157,0.707874,0.704791,0.701901,0.699193,0.696661,0.694295,0.692085,0.690022,0.688097,0.686299,0.684619,0.683047,0.681573,0.680189,0.678884,0.677649,0.676474,0.675351,0.674269,0.67322,0.672194,0.671182,0.670173,0.669158,0.668127,0.667069,0.665973,0.664829,0.663622,0.662342,0.660973,0.659503,0.657914,0.656189,0.654312,0.652262,0.650019,0.647558,0.644857,0.641887,0.638622,0.635029,0.631076,0.626728,0.621947,0.616694,0.610928,0.604605,0.597679,0.590106,0.581837,0.572828,0.563033,0.552408,0.540916,0.528521,0.515197,0.500926,0.485702,0.469533,0.452442,0.434473,0.415692,0.396182,0.376059,0.355457},
	{1.01393,1.01459,1.01527,1.01598,1.01671,1.01747,1.01825,1.01906,1.01989,1.02075,1.02163,1.02254,1.02348,1.02444,1.02542,1.02643,1.02746,1.02852,1.02959,1.03069,1.03181,1.03295,1.0341,1.03528,1.03646,1.03766,1.03888,1.0401,1.04133,1.04256,1.0438,1.04504,1.04628,1.04752,1.04875,1.04997,1.05118,1.05237,1.05355,1.05471,1.05584,1.05694,1.05801,1.05904,1.06004,1.06099,1.0619,1.06276,1.06356,1.06431,1.06499,1.06561,1.06615,1.06662,1.06701,1.06732,1.06754,1.06767,1.0677,1.06763,1.06746,1.06717,1.06677,1.06626,1.06561,1.06484,1.06394,1.0629,1.06171,1.06038,1.0589,1.05726,1.05546,1.05349,1.05135,1.04904,1.04655,1.04387,1.04101,1.03796,1.03471,1.03126,1.02761,1.02376,1.0197,1.01543,1.01095,1.00626,1.00137,0.996261,0.990948,0.98543,0.979711,0.973795,0.967688,0.961396,0.954925,0.948286,0.941488,0.934541,0.927457,0.92025,0.912933,0.905522,0.898032,0.890479,0.882882,0.875258,0.867625,0.860002,0.852408,0.844861,0.837381,0.829984,0.82269,0.815514,0.808474,0.801584,0.794859,0.788311,0.781953,0.775793,0.769842,0.764107,0.758593,0.753305,0.748245,0.743415,0.738816,0.734446,0.730302,0.726381,0.722679,0.719189,0.715905,0.71282,0.709927,0.707216,0.704679,0.702308,0.700092,0.698022,0.696088,0.694281,0.692591,0.691007,0.689521,0.688122,0.6868,0.685546,0.684351,0.683205,0.682098,0.681021,0.679964,0.678916,0.677869,0.676812,0.675734,0.674624,0.673471,0.672262,0.670986,0.669628,0.668173,0.666608,0.664915,0.663076,0.661072,0.658883,0.656486,0.653858,0.650973,0.647802,0.644315,0.640482,0.636267,0.631633,0.626543,0.620954,0.614825,0.608112,0.600767,0.592745,0.583999,0.574483,0.564154,0.552969,0.540892,0.527893,0.513949,0.499049,0.483192,0.466396,0.448692,0.430137,0.410805,0.390797,0.370235,0.349269},
	{1.0141,1.01478,1.01547,1.0162,1.01695,1.01773,1.01853,1.01936,1.02022,1.02111,1.02202,1.02296,1.02393,1.02492,1.02594,1.02699,1.02806,1.02916,1.03029,1.03143,1.0326,1.0338,1.03501,1.03624,1.03749,1.03876,1.04004,1.04133,1.04263,1.04395,1.04527,1.04659,1.04792,1.04924,1.05057,1.05188,1.05319,1.05448,1.05576,1.05702,1.05826,1.05947,1.06065,1.0618,1.06291,1.06399,1.06501,1.06599,1.06692,1.06778,1.06859,1.06933,1.07,1.0706,1.07112,1.07155,1.07189,1.07215,1.0723,1.07236,1.0723,1.07214,1.07186,1.07145,1.07093,1.07027,1.06948,1.06854,1.06747,1.06624,1.06486,1.06332,1.06161,1.05974,1.05769,1.05547,1.05306,1.05047,1.04769,1.04471,1.04154,1.03816,1.03458,1.0308,1.0268,1.0226,1.01818,1.01355,1.00871,1.00365,0.998392,0.992923,0.987251,0.981379,0.975314,0.969062,0.962631,0.956029,0.949265,0.942352,0.9353,0.928123,0.920835,0.913451,0.905986,0.898458,0.890883,0.883281,0.875668,0.868064,0.860488,0.852958,0.845492,0.83811,0.830829,0.823666,0.816638,0.809758,0.803043,0.796504,0.790153,0.784001,0.778056,0.772327,0.766818,0.761534,0.756478,0.751652,0.747055,0.742686,0.738544,0.734624,0.730921,0.727431,0.724146,0.721059,0.718164,0.71545,0.71291,0.710533,0.708312,0.706236,0.704295,0.70248,0.700781,0.699187,0.697689,0.696276,0.69494,0.693671,0.692458,0.691292,0.690163,0.689061,0.687976,0.686899,0.685819,0.684724,0.683606,0.682451,0.681248,0.679984,0.678647,0.677222,0.675694,0.674048,0.672265,0.670328,0.668216,0.665908,0.663381,0.66061,0.657568,0.654227,0.650554,0.646517,0.642081,0.637207,0.631856,0.625987,0.619555,0.612515,0.604822,0.596428,0.587287,0.577353,0.566583,0.554938,0.542383,0.52889,0.514441,0.499029,0.48266,0.465357,0.44716,0.428133,0.408359,0.387948,0.367033,0.345769},
	{1.01426,1.01495,1.01566,1.0164,1.01717,1.01797,1.01879,1.01965,1.02053,1.02144,1.02238,1.02335,1.02435,1.02538,1.02644,1.02752,1.02864,1.02978,1.03095,1.03214,1.03336,1.03461,1.03588,1.03717,1.03848,1.03981,1.04116,1.04252,1.0439,1.04529,1.04669,1.0481,1.04951,1.05092,1.05234,1.05375,1.05515,1.05655,1.05793,1.0593,1.06064,1.06197,1.06326,1.06453,1.06576,1.06695,1.0681,1.0692,1.07025,1.07125,1.07218,1.07305,1.07385,1.07457,1.07522,1.07578,1.07626,1.07664,1.07693,1.07711,1.07718,1.07714,1.07699,1.07671,1.0763,1.07576,1.07509,1.07427,1.07331,1.07219,1.07091,1.06948,1.06788,1.0661,1.06416,1.06203,1.05971,1.05721,1.05451,1.05162,1.04852,1.04523,1.04172,1.03801,1.03408,1.02994,1.02559,1.02102,1.01623,1.01124,1.00603,1.00061,0.994988,0.989163,0.983143,0.976934,0.970543,0.963979,0.957252,0.950373,0.943354,0.936208,0.92895,0.921593,0.914155,0.906651,0.8991,0.891519,0.883927,0.876342,0.868783,0.86127,0.85382,0.846452,0.839184,0.832033,0.825016,0.818146,0.81144,0.80491,0.798567,0.792421,0.786483,0.780759,0.775254,0.769975,0.764922,0.760099,0.755504,0.751137,0.746996,0.743076,0.739374,0.735883,0.732596,0.729508,0.72661,0.723893,0.721349,0.718968,0.716742,0.714659,0.712712,0.710889,0.70918,0.707576,0.706067,0.704642,0.703292,0.702007,0.700777,0.699592,0.698442,0.697316,0.696206,0.6951,0.693988,0.692858,0.691701,0.690503,0.689253,0.687937,0.686543,0.685054,0.683457,0.681733,0.679866,0.677836,0.675623,0.673203,0.670554,0.667649,0.66446,0.660959,0.657111,0.652885,0.648242,0.643145,0.637553,0.631423,0.62471,0.61737,0.609355,0.600618,0.591114,0.580797,0.569624,0.55756,0.544569,0.530628,0.515722,0.499847,0.483016,0.465256,0.446616,0.427165,0.406995,0.386223,0.36499,0.343458},
	{1.01441,1.01511,1.01584,1.0166,1.01738,1.0182,1.01904,1.01992,1.02082,1.02176,1.02273,1.02373,1.02476,1.02582,1.02691,1.02803,1.02918,1.03037,1.03158,1.03282,1.03409,1.03539,1.03671,1.03806,1.03943,1.04082,1.04224,1.04367,1.04512,1.04659,1.04807,1.04956,1.05106,1.05256,1.05407,1.05557,1.05707,1.05857,1.06006,1.06153,1.06299,1.06442,1.06583,1.06722,1.06857,1.06988,1.07116,1.07238,1.07356,1.07469,1.07575,1.07675,1.07769,1.07855,1.07933,1.08002,1.08063,1.08115,1.08157,1.08188,1.08209,1.08218,1.08216,1.08201,1.08173,1.08132,1.08076,1.08007,1.07922,1.07822,1.07706,1.07574,1.07425,1.07258,1.07073,1.06871,1.06649,1.06408,1.06147,1.05867,1.05566,1.05244,1.04902,1.04538,1.04153,1.03746,1.03317,1.02867,1.02395,1.01901,1.01386,1.00849,1.00292,0.997146,0.991173,0.985008,0.978659,0.972136,0.965447,0.958604,0.951619,0.944505,0.937277,0.929949,0.922537,0.915059,0.907531,0.899972,0.8924,0.884834,0.877293,0.869796,0.862362,0.855008,0.847753,0.840613,0.833606,0.826747,0.820049,0.813526,0.80719,0.801051,0.795118,0.789398,0.783898,0.778621,0.773571,0.76875,0.764156,0.75979,0.755649,0.751728,0.748024,0.744531,0.741242,0.73815,0.735248,0.732526,0.729976,0.727589,0.725354,0.723264,0.721306,0.719472,0.717751,0.716134,0.714609,0.713168,0.711799,0.710493,0.70924,0.708029,0.70685,0.705694,0.704549,0.703404,0.70225,0.701074,0.699865,0.69861,0.697297,0.695912,0.694441,0.692868,0.691178,0.689353,0.687373,0.68522,0.682871,0.680303,0.677491,0.674408,0.671024,0.66731,0.663231,0.658753,0.653837,0.648443,0.642531,0.636056,0.628972,0.621234,0.612795,0.603608,0.593626,0.582807,0.571109,0.558497,0.544941,0.530422,0.514929,0.498465,0.481051,0.462721,0.443535,0.423572,0.402933,0.381748,0.360167,0.338361},
	{1.01455,1.01526,1.016,1.01678,1.01758,1.01841,1.01928,1.02017,1.0211,1.02206,1.02305,1.02408,1.02514,1.02623,1.02736,1.02851,1.0297,1.03093,1.03218,1.03347,1.03479,1.03613,1.03751,1.03892,1.04035,1.0418,1.04328,1.04479,1.04631,1.04785,1.04941,1.05098,1.05256,1.05415,1.05575,1.05735,1.05895,1.06055,1.06214,1.06372,1.06529,1.06684,1.06837,1.06987,1.07134,1.07278,1.07418,1.07554,1.07685,1.0781,1.0793,1.08044,1.08151,1.08251,1.08343,1.08426,1.08501,1.08567,1.08623,1.08668,1.08703,1.08726,1.08737,1.08735,1.08721,1.08693,1.0865,1.08594,1.08522,1.08434,1.0833,1.0821,1.08072,1.07917,1.07743,1.07551,1.07339,1.07108,1.06858,1.06587,1.06295,1.05982,1.05648,1.05292,1.04914,1.04515,1.04094,1.0365,1.03185,1.02697,1.02188,1.01657,1.01106,1.00533,0.99941,0.993292,0.986988,0.980506,0.973858,0.967053,0.960103,0.953023,0.945826,0.938528,0.931144,0.923691,0.916188,0.908651,0.901101,0.893554,0.886032,0.878551,0.871132,0.863792,0.85655,0.849423,0.842427,0.835577,0.828888,0.822373,0.816044,0.809912,0.803984,0.798269,0.792773,0.7875,0.782453,0.777633,0.773041,0.768676,0.764535,0.760615,0.756911,0.753417,0.750126,0.747032,0.744127,0.741402,0.738849,0.736457,0.734218,0.732121,0.730157,0.728316,0.726587,0.72496,0.723425,0.721972,0.720591,0.719271,0.718002,0.716774,0.715576,0.714398,0.71323,0.71206,0.710877,0.70967,0.708426,0.707134,0.705779,0.704348,0.702827,0.701199,0.699447,0.697555,0.695502,0.693269,0.690831,0.688167,0.68525,0.682052,0.678544,0.674693,0.670466,0.665827,0.660738,0.655156,0.649042,0.642349,0.635033,0.627046,0.618342,0.608873,0.598595,0.587463,0.575439,0.562488,0.548581,0.533702,0.517844,0.501012,0.483229,0.464537,0.444998,0.424697,0.40374,0.382262,0.360417,0.338382},
	{1.01467,1.0154,1.01616,1.01695,1.01776,1.01861,1.0195,1.02041,1.02136,1.02234,1.02336,1.02441,1.0255,1.02662,1.02778,1.02897,1.0302,1.03146,1.03276,1.03409,1.03545,1.03685,1.03828,1.03974,1.04123,1.04275,1.04429,1.04586,1.04746,1.04907,1.05071,1.05236,1.05403,1.05571,1.05739,1.05909,1.06079,1.06249,1.06418,1.06587,1.06755,1.06922,1.07086,1.07249,1.07408,1.07565,1.07717,1.07866,1.0801,1.0815,1.08283,1.08411,1.08532,1.08646,1.08752,1.0885,1.08939,1.09019,1.09089,1.09149,1.09198,1.09235,1.09261,1.09273,1.09273,1.09259,1.0923,1.09187,1.09128,1.09053,1.08962,1.08854,1.08729,1.08585,1.08423,1.08242,1.08042,1.07822,1.07581,1.0732,1.07038,1.06734,1.06409,1.06062,1.05693,1.05301,1.04887,1.04451,1.03993,1.03512,1.03009,1.02485,1.01939,1.01372,1.00785,1.00178,0.995523,0.989086,0.982479,0.975713,0.968801,0.961756,0.954591,0.947323,0.939968,0.932542,0.925064,0.91755,0.910021,0.902494,0.894989,0.887526,0.880122,0.872796,0.865566,0.85845,0.851464,0.844624,0.837943,0.831436,0.825113,0.818986,0.813063,0.807351,0.801858,0.796587,0.791542,0.786723,0.782132,0.777766,0.773625,0.769702,0.765996,0.762498,0.759204,0.756106,0.753196,0.750465,0.747904,0.745505,0.743257,0.741151,0.739176,0.737322,0.735579,0.733938,0.732386,0.730915,0.729513,0.72817,0.726876,0.72562,0.724392,0.72318,0.721975,0.720763,0.719535,0.718278,0.716979,0.715625,0.714204,0.712699,0.711096,0.709379,0.707529,0.705528,0.703357,0.700993,0.698413,0.695592,0.692504,0.68912,0.685408,0.681335,0.676867,0.671967,0.666593,0.660706,0.654261,0.647214,0.639517,0.631125,0.621989,0.612063,0.601303,0.589666,0.577115,0.563619,0.549153,0.533704,0.517271,0.499867,0.481522,0.462285,0.442229,0.421448,0.40006,0.378208,0.356057,0.333788},
	{1.01479,1.01553,1.0163,1.0171,1.01794,1.0188,1.0197,1.02064,1.02161,1.02261,1.02365,1.02473,1.02584,1.02699,1.02818,1.02941,1.03067,1.03197,1.03331,1.03468,1.03609,1.03754,1.03902,1.04053,1.04208,1.04365,1.04526,1.0469,1.04856,1.05025,1.05196,1.0537,1.05545,1.05721,1.05899,1.06078,1.06258,1.06438,1.06618,1.06798,1.06977,1.07155,1.07332,1.07506,1.07678,1.07847,1.08013,1.08175,1.08333,1.08486,1.08634,1.08775,1.08911,1.09039,1.0916,1.09273,1.09377,1.09472,1.09557,1.09632,1.09695,1.09747,1.09788,1.09815,1.09829,1.09829,1.09815,1.09786,1.09741,1.0968,1.09602,1.09507,1.09395,1.09264,1.09114,1.08945,1.08756,1.08547,1.08318,1.08067,1.07795,1.07502,1.07186,1.06848,1.06488,1.06105,1.05699,1.05271,1.0482,1.04346,1.0385,1.03332,1.02792,1.02231,1.01649,1.01048,1.00427,0.997881,0.991318,0.984594,0.977721,0.970712,0.963582,0.956347,0.949021,0.941623,0.93417,0.926681,0.919173,0.911667,0.904181,0.896734,0.889346,0.882035,0.874818,0.867714,0.860738,0.853907,0.847235,0.840734,0.834418,0.828296,0.822378,0.816671,0.811181,0.805912,0.800869,0.796052,0.791461,0.787095,0.782953,0.77903,0.775321,0.771822,0.768525,0.765423,0.762509,0.759773,0.757207,0.754801,0.752546,0.750432,0.748448,0.746585,0.744832,0.743178,0.741613,0.740127,0.738708,0.737348,0.736034,0.734756,0.733504,0.732266,0.731031,0.729787,0.728523,0.727227,0.725885,0.724484,0.72301,0.721447,0.719781,0.717995,0.716069,0.713985,0.711722,0.709258,0.706569,0.703629,0.700411,0.696884,0.693017,0.688777,0.684127,0.679029,0.673442,0.667325,0.660633,0.65332,0.64534,0.636646,0.62719,0.616925,0.605809,0.593799,0.58086,0.566963,0.552085,0.536216,0.51936,0.501533,0.48277,0.463127,0.442682,0.421534,0.39981,0.377657,0.355244,0.332759},
	{1.01491,1.01566,1.01644,1.01725,1.0181,1.01898,1.0199,1.02085,1.02184,1.02286,1.02393,1.02503,1.02617,1.02735,1.02856,1.02982,1.03112,1.03246,1.03383,1.03525,1.0367,1.03819,1.03972,1.04129,1.04289,1.04453,1.0462,1.0479,1.04963,1.05139,1.05318,1.05499,1.05683,1.05868,1.06055,1.06243,1.06433,1.06623,1.06814,1.07004,1.07195,1.07384,1.07573,1.07759,1.07944,1.08126,1.08305,1.08481,1.08652,1.08819,1.08981,1.09137,1.09287,1.09431,1.09566,1.09694,1.09814,1.09924,1.10024,1.10114,1.10194,1.10261,1.10317,1.10359,1.10389,1.10404,1.10404,1.1039,1.1036,1.10313,1.10249,1.10168,1.10069,1.09951,1.09815,1.09658,1.09482,1.09285,1.09067,1.08828,1.08567,1.08283,1.07978,1.0765,1.07299,1.06925,1.06528,1.06108,1.05665,1.05199,1.0471,1.04199,1.03665,1.0311,1.02535,1.01939,1.01323,1.00689,1.00037,0.993694,0.986862,0.979892,0.972798,0.965596,0.958301,0.950932,0.943506,0.936041,0.928557,0.921072,0.913605,0.906176,0.898803,0.891506,0.884303,0.87721,0.870245,0.863423,0.856759,0.850266,0.843955,0.837839,0.831925,0.826221,0.820734,0.815468,0.810425,0.805609,0.801018,0.796652,0.792508,0.788583,0.784872,0.78137,0.778069,0.774963,0.772044,0.769302,0.76673,0.764317,0.762054,0.759931,0.757937,0.756062,0.754297,0.752629,0.751049,0.749546,0.74811,0.746729,0.745392,0.74409,0.74281,0.741542,0.740274,0.738994,0.73769,0.736349,0.734959,0.733504,0.731972,0.730345,0.728608,0.726744,0.724733,0.722556,0.72019,0.717614,0.714803,0.711729,0.708364,0.704679,0.700639,0.696211,0.691358,0.68604,0.680216,0.673843,0.666876,0.659269,0.650975,0.641946,0.632135,0.621497,0.609988,0.597568,0.584203,0.569866,0.554539,0.538214,0.520899,0.502616,0.483405,0.463329,0.442473,0.420941,0.398869,0.376409,0.353737,0.331047},
	{1.01501,1.01577,1.01656,1.01739,1.01825,1.01915,1.02008,1.02105,1.02206,1.0231,1.02419,1.02531,1.02647,1.02768,1.02893,1.03022,1.03155,1.03292,1.03433,1.03579,1.03728,1.03882,1.0404,1.04202,1.04367,1.04537,1.0471,1.04886,1.05066,1.0525,1.05436,1.05625,1.05817,1.0601,1.06206,1.06404,1.06603,1.06804,1.07005,1.07206,1.07408,1.07609,1.07809,1.08008,1.08206,1.08401,1.08593,1.08782,1.08968,1.09149,1.09325,1.09496,1.09661,1.0982,1.09971,1.10114,1.10249,1.10375,1.10492,1.10598,1.10693,1.10776,1.10848,1.10906,1.10951,1.10982,1.10998,1.10999,1.10984,1.10952,1.10904,1.10837,1.10752,1.10648,1.10525,1.10382,1.10219,1.10034,1.09829,1.09601,1.09352,1.0908,1.08785,1.08467,1.08126,1.07762,1.07374,1.06963,1.06528,1.0607,1.05589,1.05085,1.04559,1.0401,1.03441,1.02851,1.02241,1.01612,1.00965,1.00302,0.996229,0.989299,0.982243,0.975076,0.967814,0.960475,0.953077,0.945639,0.938178,0.930714,0.923267,0.915856,0.9085,0.901218,0.894028,0.886947,0.879993,0.87318,0.866524,0.860038,0.853734,0.847623,0.841713,0.836013,0.830528,0.825264,0.820223,0.815407,0.810816,0.80645,0.802305,0.798378,0.794665,0.791159,0.787855,0.784745,0.781821,0.779074,0.776495,0.774076,0.771805,0.769673,0.76767,0.765784,0.764007,0.762327,0.760732,0.759214,0.75776,0.75636,0.755002,0.753677,0.752372,0.751076,0.749777,0.748463,0.747122,0.745741,0.744305,0.742801,0.741215,0.739529,0.737727,0.735791,0.733702,0.731439,0.72898,0.726302,0.723378,0.720183,0.716686,0.712856,0.70866,0.704062,0.699025,0.693508,0.68747,0.680867,0.673653,0.665782,0.657207,0.647879,0.637752,0.626781,0.614923,0.602138,0.588396,0.57367,0.557944,0.541215,0.523493,0.504805,0.485197,0.464735,0.44351,0.421632,0.399242,0.376498,0.353579,0.330684},
	{1.0151,1.01587,1.01668,1.01752,1.01839,1.0193,1.02025,1.02123,1.02226,1.02332,1.02443,1.02558,1.02676,1.028,1.02927,1.03059,1.03195,1.03336,1.03481,1.0363,1.03784,1.03942,1.04104,1.04271,1.04442,1.04617,1.04796,1.04979,1.05166,1.05356,1.0555,1.05747,1.05946,1.06149,1.06354,1.06561,1.06769,1.0698,1.07191,1.07404,1.07616,1.07829,1.08042,1.08253,1.08463,1.08671,1.08877,1.0908,1.0928,1.09475,1.09667,1.09852,1.10033,1.10207,1.10373,1.10533,1.10684,1.10826,1.10958,1.11081,1.11192,1.11292,1.1138,1.11455,1.11516,1.11564,1.11596,1.11613,1.11614,1.11598,1.11564,1.11513,1.11443,1.11354,1.11245,1.11116,1.10966,1.10795,1.10602,1.10387,1.1015,1.0989,1.09607,1.093,1.08969,1.08615,1.08237,1.07835,1.07409,1.0696,1.06487,1.05991,1.05472,1.0493,1.04367,1.03784,1.0318,1.02556,1.01915,1.01256,1.00582,0.998932,0.991916,0.984786,0.977559,0.970252,0.962883,0.955471,0.948035,0.940594,0.933168,0.925776,0.918436,0.911169,0.903993,0.896924,0.88998,0.883177,0.876529,0.87005,0.863752,0.857645,0.85174,0.846043,0.84056,0.835297,0.830257,0.825441,0.82085,0.816482,0.812335,0.808405,0.804688,0.801179,0.79787,0.794755,0.791824,0.78907,0.786484,0.784056,0.781775,0.779633,0.777618,0.77572,0.773928,0.772232,0.77062,0.769082,0.767608,0.766184,0.764802,0.763448,0.762112,0.760782,0.759446,0.758092,0.756705,0.755274,0.753784,0.752221,0.750569,0.748811,0.74693,0.744907,0.742723,0.740357,0.737784,0.734982,0.731923,0.72858,0.724922,0.720918,0.716532,0.711728,0.706468,0.700711,0.694414,0.687533,0.680021,0.671832,0.662918,0.653231,0.642724,0.631353,0.619076,0.605856,0.591664,0.576475,0.560278,0.543073,0.524876,0.505718,0.485652,0.464751,0.443111,0.420853,0.398121,0.375081,0.351918,0.328833},
	{1.01519,1.01597,1.01679,1.01764,1.01852,1.01945,1.02041,1.02141,1.02245,1.02353,1.02466,1.02583,1.02704,1.02829,1.0296,1.03094,1.03233,1.03377,1.03526,1.03679,1.03837,1.03999,1.04166,1.04338,1.04514,1.04695,1.0488,1.05069,1.05262,1.05459,1.0566,1.05864,1.06072,1.06283,1.06496,1.06713,1.06931,1.07151,1.07373,1.07597,1.07821,1.08045,1.08269,1.08493,1.08716,1.08938,1.09157,1.09374,1.09588,1.09798,1.10004,1.10205,1.10401,1.1059,1.10773,1.10949,1.11116,1.11275,1.11424,1.11564,1.11692,1.11809,1.11914,1.12005,1.12084,1.12148,1.12197,1.12231,1.12248,1.12248,1.12231,1.12195,1.12141,1.12067,1.11973,1.11859,1.11724,1.11567,1.11388,1.11186,1.10962,1.10714,1.10442,1.10147,1.09828,1.09485,1.09117,1.08725,1.08309,1.07868,1.07404,1.06916,1.06405,1.05871,1.05315,1.04738,1.0414,1.03523,1.02887,1.02233,1.01564,1.0088,1.00182,0.994732,0.987541,0.980268,0.97293,0.965546,0.958136,0.950719,0.943314,0.935942,0.92862,0.921369,0.914206,0.90715,0.900217,0.893424,0.886784,0.880312,0.87402,0.867919,0.862017,0.856323,0.850842,0.845581,0.840541,0.835725,0.831133,0.826763,0.822614,0.818682,0.814962,0.811448,0.808135,0.805014,0.802077,0.799317,0.796723,0.794286,0.791996,0.789843,0.787816,0.785906,0.7841,0.782388,0.78076,0.779203,0.777708,0.776262,0.774855,0.773474,0.772109,0.770746,0.769374,0.767979,0.76655,0.765071,0.763529,0.761908,0.760192,0.758365,0.756408,0.754303,0.752028,0.749563,0.746882,0.743961,0.740774,0.73729,0.73348,0.729309,0.724744,0.719746,0.714276,0.708292,0.701752,0.694609,0.686818,0.67833,0.669099,0.659076,0.648214,0.636471,0.623806,0.610182,0.595572,0.579955,0.563323,0.545679,0.527043,0.507452,0.486962,0.465655,0.443632,0.421018,0.397965,0.374645,0.351245,0.32797},
	{1.01527,1.01606,1.01689,1.01775,1.01865,1.01958,1.02056,1.02157,1.02263,1.02373,1.02487,1.02606,1.0273,1.02858,1.0299,1.03128,1.0327,1.03417,1.03569,1.03726,1.03887,1.04054,1.04226,1.04402,1.04583,1.04769,1.0496,1.05155,1.05354,1.05558,1.05766,1.05978,1.06194,1.06413,1.06635,1.0686,1.07088,1.07319,1.07551,1.07785,1.0802,1.08256,1.08493,1.08729,1.08965,1.09199,1.09432,1.09663,1.09892,1.10117,1.10338,1.10554,1.10766,1.10971,1.11171,1.11363,1.11547,1.11722,1.11889,1.12045,1.12191,1.12325,1.12448,1.12557,1.12653,1.12734,1.12801,1.12852,1.12886,1.12903,1.12903,1.12884,1.12846,1.12788,1.1271,1.12611,1.12491,1.12349,1.12184,1.11996,1.11786,1.11551,1.11292,1.1101,1.10702,1.1037,1.10013,1.09632,1.09226,1.08795,1.0834,1.0786,1.07358,1.06832,1.06283,1.05713,1.05122,1.04511,1.03881,1.03233,1.02569,1.01889,1.01196,1.00491,0.99776,0.990522,0.983217,0.975863,0.96848,0.961088,0.953705,0.946353,0.93905,0.931815,0.924667,0.917623,0.910702,0.903918,0.897286,0.890821,0.884535,0.878438,0.87254,0.866848,0.86137,0.856109,0.851069,0.846252,0.841658,0.837286,0.833134,0.829198,0.825473,0.821954,0.818635,0.815507,0.812562,0.809793,0.807189,0.804741,0.80244,0.800274,0.798233,0.796306,0.794483,0.792752,0.791103,0.789523,0.788003,0.786529,0.785091,0.783676,0.782274,0.78087,0.779453,0.77801,0.776526,0.774988,0.773381,0.771689,0.769895,0.767982,0.765931,0.763723,0.761336,0.758747,0.755932,0.752864,0.749517,0.745859,0.741859,0.737483,0.732694,0.727455,0.721725,0.715461,0.708618,0.701152,0.693015,0.684159,0.674537,0.6641,0.652803,0.640603,0.627462,0.613346,0.598228,0.582093,0.564936,0.546766,0.527608,0.507506,0.486524,0.46475,0.442293,0.41929,0.395895,0.37229,0.348666,0.325232},
	{1.01535,1.01615,1.01698,1.01785,1.01876,1.01971,1.0207,1.02173,1.0228,1.02392,1.02508,1.02628,1.02754,1.02884,1.03019,1.03159,1.03304,1.03454,1.0361,1.0377,1.03936,1.04106,1.04282,1.04463,1.04649,1.0484,1.05037,1.05238,1.05443,1.05654,1.05869,1.06088,1.06311,1.06539,1.0677,1.07004,1.07241,1.07481,1.07724,1.07969,1.08215,1.08463,1.08711,1.0896,1.09209,1.09457,1.09703,1.09949,1.10191,1.10431,1.10668,1.109,1.11127,1.11349,1.11565,1.11774,1.11975,1.12168,1.12352,1.12526,1.12689,1.12842,1.12982,1.13109,1.13223,1.13323,1.13407,1.13476,1.13528,1.13563,1.1358,1.13578,1.13557,1.13517,1.13455,1.13372,1.13268,1.13141,1.12992,1.12819,1.12622,1.12401,1.12156,1.11886,1.11592,1.11272,1.10926,1.10556,1.10161,1.0974,1.09294,1.08825,1.08331,1.07813,1.07272,1.0671,1.06126,1.05521,1.04897,1.04255,1.03597,1.02922,1.02234,1.01534,1.00822,1.00102,0.993752,0.98643,0.979076,0.971709,0.964351,0.95702,0.949736,0.942518,0.935386,0.928356,0.921446,0.914672,0.90805,0.901592,0.895312,0.88922,0.883325,0.877636,0.87216,0.8669,0.86186,0.857043,0.852447,0.848073,0.843918,0.839978,0.836249,0.832725,0.829399,0.826265,0.823313,0.820535,0.817922,0.815464,0.813151,0.810973,0.808919,0.806977,0.805138,0.803389,0.80172,0.800119,0.798574,0.797074,0.795608,0.794162,0.792725,0.791283,0.789825,0.788336,0.786802,0.78521,0.783542,0.781784,0.779918,0.777927,0.77579,0.773487,0.770997,0.768295,0.765357,0.762156,0.758663,0.754847,0.750675,0.746113,0.741123,0.735666,0.729701,0.723184,0.716071,0.708315,0.699869,0.690684,0.680712,0.669907,0.658224,0.645619,0.632056,0.617504,0.601938,0.585345,0.567725,0.549091,0.529473,0.50892,0.487501,0.465313,0.442469,0.419112,0.395404,0.371529,0.347685,0.324083},
	{1.01542,1.01622,1.01706,1.01794,1.01886,1.01982,1.02082,1.02187,1.02296,1.02409,1.02527,1.02649,1.02777,1.02909,1.03047,1.03189,1.03337,1.0349,1.03649,1.03812,1.03981,1.04156,1.04336,1.04522,1.04713,1.04909,1.0511,1.05317,1.05529,1.05746,1.05968,1.06194,1.06425,1.06661,1.069,1.07143,1.0739,1.0764,1.07893,1.08148,1.08405,1.08665,1.08925,1.09186,1.09448,1.09709,1.0997,1.10229,1.10487,1.10742,1.10993,1.11241,1.11485,1.11723,1.11956,1.12182,1.124,1.12611,1.12813,1.13005,1.13187,1.13357,1.13516,1.13662,1.13794,1.13913,1.14016,1.14103,1.14174,1.14227,1.14262,1.14278,1.14275,1.14252,1.14207,1.14142,1.14054,1.13943,1.13809,1.13652,1.1347,1.13264,1.13033,1.12777,1.12496,1.12188,1.11856,1.11497,1.11113,1.10703,1.10268,1.09808,1.09323,1.08814,1.08282,1.07728,1.07151,1.06554,1.05936,1.05301,1.04648,1.03979,1.03296,1.026,1.01893,1.01177,1.00453,0.997244,0.989921,0.982582,0.975249,0.96794,0.960677,0.953477,0.94636,0.939344,0.932446,0.925683,0.91907,0.912619,0.906345,0.900258,0.894367,0.888681,0.883205,0.877946,0.872906,0.868087,0.863489,0.859112,0.854953,0.851008,0.847273,0.843743,0.84041,0.837267,0.834306,0.831519,0.828895,0.826424,0.824098,0.821905,0.819834,0.817874,0.816015,0.814245,0.812552,0.810925,0.809352,0.807821,0.806319,0.804836,0.803357,0.80187,0.800362,0.798818,0.797224,0.795566,0.793826,0.791989,0.790036,0.78795,0.785709,0.783293,0.780678,0.777841,0.774755,0.771393,0.767725,0.763719,0.759341,0.754555,0.749323,0.743605,0.737358,0.730539,0.723102,0.714999,0.706183,0.696606,0.686219,0.674976,0.662832,0.649748,0.635687,0.620621,0.604529,0.587402,0.569244,0.550073,0.529927,0.50886,0.486952,0.464303,0.441038,0.417306,0.393277,0.369139,0.345096,0.32136},
	{1.01548,1.01629,1.01714,1.01803,1.01896,1.01993,1.02094,1.022,1.0231,1.02425,1.02544,1.02669,1.02798,1.02933,1.03073,1.03218,1.03368,1.03524,1.03685,1.03852,1.04025,1.04204,1.04388,1.04577,1.04773,1.04974,1.05181,1.05394,1.05611,1.05835,1.06063,1.06297,1.06535,1.06779,1.07026,1.07278,1.07534,1.07794,1.08057,1.08323,1.08591,1.08862,1.09134,1.09408,1.09683,1.09957,1.10232,1.10505,1.10777,1.11047,1.11315,1.11579,1.11839,1.12094,1.12343,1.12587,1.12823,1.13052,1.13272,1.13482,1.13683,1.13872,1.1405,1.14215,1.14366,1.14504,1.14626,1.14732,1.14822,1.14894,1.14948,1.14983,1.14998,1.14993,1.14967,1.14919,1.14848,1.14755,1.14638,1.14496,1.14331,1.1414,1.13924,1.13682,1.13415,1.13121,1.12801,1.12455,1.12083,1.11684,1.1126,1.1081,1.10336,1.09837,1.09313,1.08767,1.08198,1.07609,1.06998,1.06369,1.05722,1.05059,1.04382,1.03691,1.02988,1.02276,1.01557,1.00832,1.00103,0.993716,0.98641,0.979125,0.971883,0.964703,0.957603,0.950602,0.943717,0.936965,0.930361,0.923919,0.917652,0.91157,0.905683,0.9,0.894526,0.889268,0.884227,0.879407,0.874808,0.870428,0.866265,0.862316,0.858576,0.85504,0.851701,0.84855,0.845581,0.842784,0.84015,0.837668,0.835329,0.833122,0.831036,0.82906,0.827183,0.825393,0.823678,0.822027,0.820427,0.818868,0.817335,0.815817,0.814301,0.812773,0.811219,0.809626,0.807978,0.806259,0.804455,0.802546,0.800516,0.798344,0.79601,0.793492,0.790767,0.787809,0.784592,0.781087,0.777263,0.773088,0.768527,0.763544,0.758099,0.752151,0.745658,0.738574,0.730852,0.722447,0.713309,0.70339,0.692642,0.68102,0.668479,0.65498,0.640489,0.62498,0.608435,0.590847,0.572225,0.552592,0.531989,0.510477,0.488141,0.465089,0.441449,0.417378,0.39305,0.368659,0.344411,0.320517},
	{1.01554,1.01636,1.01721,1.01811,1.01905,1.02003,1.02105,1.02212,1.02324,1.0244,1.02561,1.02687,1.02818,1.02955,1.03097,1.03244,1.03397,1.03556,1.0372,1.0389,1.04067,1.04249,1.04437,1.04631,1.04831,1.05037,1.05249,1.05467,1.05691,1.0592,1.06155,1.06396,1.06642,1.06893,1.07149,1.07409,1.07674,1.07944,1.08216,1.08493,1.08772,1.09055,1.09339,1.09625,1.09912,1.102,1.10489,1.10777,1.11064,1.11349,1.11632,1.11912,1.12188,1.1246,1.12727,1.12988,1.13242,1.13489,1.13728,1.13957,1.14177,1.14385,1.14582,1.14767,1.14938,1.15095,1.15237,1.15363,1.15473,1.15564,1.15638,1.15692,1.15727,1.1574,1.15733,1.15703,1.15651,1.15575,1.15475,1.15351,1.15202,1.15027,1.14827,1.146,1.14348,1.14068,1.13762,1.13429,1.13069,1.12683,1.12271,1.11832,1.11368,1.10879,1.10365,1.09828,1.09268,1.08686,1.08083,1.07461,1.06821,1.06164,1.05491,1.04806,1.04108,1.03401,1.02686,1.01964,1.01238,1.00511,0.997828,0.99057,0.98335,0.97619,0.969108,0.962123,0.955252,0.948511,0.941917,0.935483,0.929222,0.923145,0.917262,0.911581,0.906109,0.900851,0.89581,0.890988,0.886386,0.882002,0.877834,0.87388,0.870134,0.86659,0.863242,0.860082,0.857102,0.854293,0.851646,0.84915,0.846795,0.844571,0.842466,0.84047,0.838569,0.836754,0.835012,0.833331,0.831699,0.830104,0.828532,0.82697,0.825407,0.823826,0.822216,0.82056,0.818844,0.817051,0.815164,0.813166,0.811037,0.808758,0.806307,0.803661,0.800796,0.797686,0.794303,0.790618,0.786599,0.782211,0.777421,0.772189,0.766475,0.760238,0.753434,0.746017,0.73794,0.729154,0.719613,0.709266,0.698067,0.685971,0.672935,0.658921,0.643898,0.627842,0.610741,0.592591,0.573406,0.553216,0.532069,0.510034,0.487203,0.463691,0.439637,0.415204,0.390574,0.365944,0.341525,0.317529},
	{1.01559,1.01641,1.01728,1.01819,1.01913,1.02012,1.02116,1.02224,1.02336,1.02454,1.02577,1.02704,1.02837,1.02976,1.0312,1.03269,1.03425,1.03586,1.03753,1.03927,1.04106,1.04292,1.04483,1.04682,1.04886,1.05097,1.05314,1.05537,1.05766,1.06002,1.06244,1.06491,1.06744,1.07003,1.07267,1.07536,1.0781,1.08089,1.08372,1.08659,1.08949,1.09242,1.09539,1.09837,1.10137,1.10439,1.10741,1.11043,1.11345,1.11646,1.11944,1.12241,1.12534,1.12823,1.13107,1.13386,1.13658,1.13924,1.14181,1.1443,1.14669,1.14897,1.15114,1.15319,1.1551,1.15687,1.1585,1.15996,1.16126,1.16238,1.16331,1.16406,1.1646,1.16494,1.16505,1.16495,1.16461,1.16404,1.16323,1.16216,1.16084,1.15927,1.15743,1.15532,1.15295,1.1503,1.14739,1.1442,1.14073,1.137,1.133,1.12873,1.1242,1.11941,1.11438,1.1091,1.10359,1.09785,1.09191,1.08576,1.07943,1.07292,1.06626,1.05946,1.05254,1.04552,1.03841,1.03123,1.02401,1.01677,1.00952,1.00229,0.995094,0.987956,0.980893,0.973925,0.967069,0.960342,0.953758,0.947334,0.94108,0.935009,0.929131,0.923454,0.917984,0.912726,0.907685,0.902862,0.898258,0.893871,0.8897,0.885741,0.881989,0.878439,0.875084,0.871916,0.868927,0.866108,0.863449,0.860941,0.858573,0.856334,0.854213,0.852198,0.850279,0.848442,0.846677,0.84497,0.84331,0.841684,0.840078,0.83848,0.836877,0.835253,0.833594,0.831886,0.830112,0.828256,0.826301,0.824227,0.822017,0.819648,0.8171,0.814348,0.811368,0.808132,0.804613,0.800779,0.796599,0.792038,0.78706,0.781625,0.775693,0.769222,0.762166,0.75448,0.746115,0.737024,0.727158,0.716468,0.704909,0.692433,0.679002,0.664577,0.64913,0.632639,0.615093,0.596494,0.576859,0.556221,0.534633,0.51217,0.488929,0.465032,0.440622,0.415866,0.390951,0.366079,0.341461,0.31731},
	{1.01563,1.01647,1.01734,1.01825,1.01921,1.02021,1.02125,1.02234,1.02348,1.02467,1.02591,1.0272,1.02855,1.02995,1.03141,1.03293,1.03451,1.03614,1.03784,1.03961,1.04143,1.04332,1.04528,1.0473,1.04939,1.05154,1.05376,1.05604,1.05839,1.06081,1.06329,1.06583,1.06843,1.07109,1.07381,1.07659,1.07942,1.0823,1.08522,1.0882,1.09121,1.09426,1.09734,1.10044,1.10358,1.10672,1.10988,1.11305,1.11622,1.11938,1.12252,1.12565,1.12875,1.13181,1.13483,1.1378,1.14071,1.14355,1.14632,1.149,1.15158,1.15407,1.15644,1.15869,1.16081,1.16279,1.16462,1.1663,1.1678,1.16913,1.17028,1.17123,1.17198,1.17251,1.17283,1.17293,1.17279,1.17241,1.17178,1.17091,1.16977,1.16837,1.1667,1.16477,1.16256,1.16007,1.1573,1.15426,1.15094,1.14734,1.14347,1.13932,1.13491,1.13024,1.12531,1.12013,1.11472,1.10907,1.10321,1.09714,1.09088,1.08445,1.07785,1.07111,1.06425,1.05727,1.05021,1.04308,1.0359,1.02869,1.02147,1.01427,1.0071,0.999986,0.992944,0.985993,0.979152,0.972438,0.965866,0.95945,0.953204,0.947139,0.941265,0.93559,0.930121,0.924864,0.919821,0.914996,0.910388,0.905996,0.901819,0.897853,0.894093,0.890534,0.887168,0.883989,0.880987,0.878153,0.875479,0.872954,0.870566,0.868306,0.866162,0.864123,0.862176,0.860309,0.858511,0.856768,0.855068,0.853398,0.851745,0.850095,0.848434,0.846748,0.84502,0.843237,0.841381,0.839435,0.837381,0.835201,0.832873,0.830376,0.827687,0.824783,0.821636,0.818219,0.814503,0.810456,0.806044,0.801232,0.795982,0.790254,0.784007,0.777196,0.769776,0.7617,0.752921,0.743388,0.733054,0.721871,0.709792,0.696773,0.682776,0.667766,0.651717,0.634611,0.616443,0.597219,0.576964,0.555718,0.533543,0.510521,0.486761,0.46239,0.437564,0.412457,0.387263,0.362187,0.337444,0.313246},
	{1.01567,1.01651,1.01739,1.01831,1.01928,1.02028,1.02134,1.02244,1.02359,1.02479,1.02604,1.02735,1.02871,1.03013,1.03161,1.03315,1.03475,1.03641,1.03814,1.03993,1.04179,1.04371,1.0457,1.04776,1.04989,1.05209,1.05435,1.05669,1.05909,1.06156,1.0641,1.06671,1.06938,1.07212,1.07492,1.07778,1.08069,1.08366,1.08669,1.08976,1.09288,1.09604,1.09924,1.10247,1.10573,1.10901,1.11231,1.11562,1.11893,1.12225,1.12555,1.12884,1.13211,1.13535,1.13854,1.1417,1.14479,1.14783,1.15079,1.15367,1.15646,1.15914,1.16172,1.16418,1.16651,1.16871,1.17075,1.17264,1.17436,1.17591,1.17727,1.17843,1.17939,1.18014,1.18067,1.18097,1.18104,1.18086,1.18043,1.17975,1.1788,1.17759,1.1761,1.17434,1.1723,1.16998,1.16737,1.16448,1.16131,1.15786,1.15412,1.15011,1.14582,1.14127,1.13645,1.13138,1.12607,1.12052,1.11475,1.10877,1.10259,1.09623,1.0897,1.08302,1.07621,1.0693,1.06228,1.0552,1.04806,1.04089,1.03371,1.02654,1.01939,1.0123,1.00528,0.998352,0.991529,0.984829,0.978269,0.971864,0.965626,0.959568,0.953699,0.948028,0.942561,0.937305,0.932262,0.927435,0.922825,0.918429,0.914248,0.910276,0.906509,0.902942,0.899567,0.896377,0.893364,0.890518,0.88783,0.885289,0.882885,0.880607,0.878443,0.876381,0.87441,0.872517,0.870689,0.868915,0.867181,0.865473,0.863779,0.862083,0.860373,0.858632,0.856846,0.854997,0.853071,0.851048,0.84891,0.846637,0.844208,0.841602,0.838794,0.83576,0.832472,0.828902,0.825019,0.820791,0.816184,0.811161,0.805684,0.799711,0.7932,0.786106,0.778384,0.769985,0.760862,0.750965,0.740245,0.728656,0.71615,0.702686,0.688226,0.672737,0.656196,0.638588,0.619911,0.600177,0.579413,0.557668,0.535008,0.511521,0.487323,0.462549,0.437359,0.411933,0.38647,0.361177,0.336273,0.311966},
	{1.01571,1.01655,1.01744,1.01837,1.01934,1.02035,1.02142,1.02253,1.02369,1.0249,1.02617,1.02749,1.02887,1.0303,1.0318,1.03336,1.03498,1.03666,1.03842,1.04023,1.04212,1.04408,1.0461,1.0482,1.05036,1.0526,1.05492,1.0573,1.05976,1.06229,1.06489,1.06756,1.0703,1.07311,1.07598,1.07892,1.08193,1.08499,1.08811,1.09128,1.09451,1.09778,1.10109,1.10445,1.10783,1.11124,1.11468,1.11813,1.1216,1.12507,1.12853,1.13199,1.13542,1.13884,1.14221,1.14555,1.14884,1.15206,1.15522,1.1583,1.1613,1.16419,1.16698,1.16966,1.1722,1.17461,1.17688,1.17899,1.18093,1.1827,1.18428,1.18566,1.18684,1.18781,1.18856,1.18907,1.18935,1.18938,1.18916,1.18868,1.18793,1.18691,1.18561,1.18404,1.18218,1.18003,1.17759,1.17486,1.17185,1.16854,1.16495,1.16108,1.15692,1.15249,1.1478,1.14284,1.13764,1.13219,1.12651,1.12062,1.11453,1.10824,1.10179,1.09518,1.08844,1.08158,1.07462,1.06758,1.06049,1.05336,1.04621,1.03907,1.03196,1.0249,1.0179,1.01099,1.00419,0.997502,0.990956,0.984561,0.978333,0.972281,0.966417,0.96075,0.955285,0.950029,0.944986,0.940157,0.935543,0.931143,0.926955,0.922976,0.919201,0.915624,0.912238,0.909036,0.90601,0.903149,0.900444,0.897885,0.895461,0.893161,0.890972,0.888884,0.886884,0.884959,0.883097,0.881285,0.879509,0.877756,0.876011,0.874261,0.872491,0.870685,0.868828,0.866902,0.86489,0.862774,0.860534,0.85815,0.855601,0.852863,0.849911,0.84672,0.843263,0.839508,0.835425,0.830981,0.826139,0.820863,0.815112,0.808845,0.802018,0.794585,0.786501,0.777716,0.768182,0.757849,0.74667,0.734597,0.721585,0.707593,0.692585,0.676532,0.659413,0.641218,0.621949,0.601624,0.580278,0.557963,0.534756,0.510753,0.486077,0.46087,0.435302,0.409559,0.383845,0.35837,0.333352,0.309},
	{1.01574,1.01659,1.01748,1.01842,1.01939,1.02042,1.02149,1.02261,1.02378,1.025,1.02628,1.02762,1.02901,1.03046,1.03198,1.03355,1.03519,1.0369,1.03868,1.04052,1.04243,1.04442,1.04648,1.04861,1.05082,1.0531,1.05546,1.05789,1.0604,1.06298,1.06564,1.06837,1.07118,1.07406,1.07701,1.08003,1.08312,1.08627,1.08948,1.09276,1.09609,1.09947,1.1029,1.10637,1.10988,1.11343,1.117,1.1206,1.12421,1.12784,1.13146,1.13508,1.13869,1.14228,1.14584,1.14936,1.15284,1.15626,1.15962,1.1629,1.16611,1.16921,1.17222,1.17511,1.17788,1.18051,1.183,1.18533,1.1875,1.1895,1.19131,1.19292,1.19433,1.19552,1.19649,1.19723,1.19773,1.19798,1.19797,1.1977,1.19715,1.19634,1.19524,1.19385,1.19218,1.19021,1.18795,1.1854,1.18255,1.1794,1.17596,1.17223,1.16822,1.16392,1.15935,1.15452,1.14943,1.14409,1.13851,1.13272,1.12671,1.12051,1.11414,1.1076,1.10092,1.09413,1.08722,1.08024,1.07319,1.06611,1.059,1.0519,1.04482,1.03779,1.03082,1.02393,1.01714,1.01047,1.00394,0.99756,0.991341,0.985298,0.97944,0.973777,0.968315,0.96306,0.958016,0.953185,0.948568,0.944164,0.939971,0.935985,0.932202,0.928616,0.92522,0.922006,0.918966,0.91609,0.913369,0.910792,0.908348,0.906026,0.903814,0.9017,0.899671,0.897715,0.895819,0.893969,0.892152,0.890355,0.888562,0.886758,0.88493,0.88306,0.881133,0.879131,0.877036,0.87483,0.872491,0.87,0.867333,0.864467,0.861377,0.858035,0.854414,0.850481,0.846206,0.841553,0.836487,0.830967,0.824955,0.818407,0.811279,0.803524,0.795095,0.785944,0.776021,0.765277,0.753664,0.741135,0.727647,0.71316,0.697639,0.681058,0.6634,0.644658,0.624838,0.603964,0.582076,0.559233,0.535517,0.511034,0.485912,0.460302,0.434378,0.408333,0.382372,0.356712,0.331568,0.307147},
	{1.01577,1.01662,1.01752,1.01846,1.01944,1.02047,1.02155,1.02268,1.02386,1.0251,1.02639,1.02773,1.02914,1.03061,1.03214,1.03373,1.03539,1.03712,1.03892,1.04079,1.04273,1.04475,1.04684,1.049,1.05125,1.05357,1.05597,1.05845,1.06101,1.06364,1.06636,1.06915,1.07203,1.07497,1.078,1.0811,1.08427,1.08751,1.09082,1.09419,1.09762,1.10111,1.10466,1.10825,1.11189,1.11556,1.11928,1.12302,1.12678,1.13055,1.13434,1.13813,1.14191,1.14567,1.14941,1.15312,1.15679,1.16041,1.16398,1.16747,1.17088,1.1742,1.17743,1.18054,1.18353,1.18639,1.18911,1.19167,1.19408,1.1963,1.19835,1.20019,1.20184,1.20326,1.20446,1.20543,1.20616,1.20663,1.20684,1.20679,1.20646,1.20586,1.20496,1.20378,1.2023,1.20053,1.19845,1.19608,1.1934,1.19042,1.18714,1.18357,1.1797,1.17554,1.17111,1.1664,1.16143,1.1562,1.15074,1.14504,1.13913,1.13302,1.12673,1.12027,1.11367,1.10693,1.10009,1.09316,1.08617,1.07913,1.07207,1.065,1.05796,1.05095,1.04401,1.03715,1.03038,1.02373,1.01721,1.01084,1.00463,0.998599,0.992747,0.987088,0.981628,0.976373,0.971328,0.966494,0.961873,0.957462,0.953262,0.949267,0.945473,0.941874,0.938464,0.935235,0.932177,0.929283,0.92654,0.92394,0.921471,0.91912,0.916877,0.914729,0.912664,0.910667,0.908727,0.906828,0.904958,0.903103,0.901246,0.899373,0.897469,0.895516,0.893498,0.891396,0.889193,0.886868,0.884401,0.881769,0.878948,0.875916,0.872644,0.869104,0.865268,0.861103,0.856576,0.85165,0.846289,0.840452,0.834098,0.827183,0.819661,0.811486,0.802608,0.792979,0.78255,0.771272,0.759096,0.745978,0.731875,0.71675,0.700571,0.683317,0.664974,0.64554,0.62503,0.603474,0.580919,0.557436,0.533116,0.508074,0.482448,0.456399,0.43011,0.40378,0.377619,0.351847,0.326676,0.302309},
	{1.01579,1.01665,1.01755,1.01849,1.01949,1.02052,1.02161,1.02275,1.02394,1.02518,1.02648,1.02784,1.02926,1.03074,1.03229,1.0339,1.03558,1.03733,1.03915,1.04104,1.04301,1.04505,1.04718,1.04938,1.05165,1.05402,1.05646,1.05898,1.06159,1.06428,1.06705,1.0699,1.07284,1.07585,1.07895,1.08213,1.08538,1.08871,1.0921,1.09557,1.09911,1.10271,1.10636,1.11008,1.11384,1.11765,1.1215,1.12538,1.12929,1.13322,1.13717,1.14112,1.14507,1.14902,1.15294,1.15684,1.16071,1.16453,1.16829,1.17199,1.17562,1.17916,1.1826,1.18594,1.18916,1.19226,1.19521,1.19801,1.20065,1.20312,1.2054,1.20749,1.20937,1.21104,1.21248,1.21369,1.21465,1.21535,1.2158,1.21597,1.21587,1.21548,1.21481,1.21383,1.21256,1.21099,1.2091,1.20692,1.20442,1.20162,1.19851,1.19509,1.19138,1.18738,1.18308,1.17851,1.17367,1.16856,1.16321,1.15762,1.15182,1.1458,1.1396,1.13322,1.12669,1.12003,1.11326,1.10639,1.09945,1.09246,1.08545,1.07843,1.07142,1.06445,1.05753,1.0507,1.04395,1.03733,1.03083,1.02447,1.01828,1.01225,1.00641,1.00076,0.995303,0.990052,0.985009,0.980176,0.975554,0.971142,0.966938,0.962938,0.959139,0.955534,0.952116,0.948878,0.945811,0.942904,0.94015,0.937535,0.935051,0.932683,0.930422,0.928254,0.926166,0.924145,0.922178,0.920251,0.91835,0.916459,0.914565,0.912652,0.910703,0.908702,0.906631,0.904473,0.902207,0.899814,0.897273,0.894562,0.891655,0.888528,0.885155,0.881506,0.877551,0.873258,0.868592,0.863518,0.857997,0.851988,0.84545,0.838338,0.830606,0.822206,0.813091,0.80321,0.792514,0.780955,0.768484,0.755057,0.740632,0.725172,0.708648,0.691036,0.672328,0.652523,0.631636,0.609701,0.586768,0.56291,0.53822,0.512817,0.486842,0.460459,0.433852,0.407225,0.380788,0.35476,0.329356,0.304778},
	{1.0158,1.01667,1.01758,1.01853,1.01952,1.02057,1.02166,1.0228,1.024,1.02526,1.02657,1.02794,1.02937,1.03087,1.03243,1.03405,1.03575,1.03752,1.03936,1.04128,1.04327,1.04534,1.04749,1.04973,1.05204,1.05444,1.05692,1.05949,1.06214,1.06488,1.06771,1.07062,1.07362,1.0767,1.07987,1.08312,1.08645,1.08986,1.09335,1.09691,1.10055,1.10425,1.10802,1.11186,1.11574,1.11968,1.12366,1.12769,1.13175,1.13583,1.13994,1.14406,1.14818,1.1523,1.15642,1.16051,1.16457,1.16859,1.17256,1.17647,1.18031,1.18408,1.18775,1.19132,1.19477,1.1981,1.20129,1.20433,1.20722,1.20993,1.21246,1.2148,1.21693,1.21884,1.22053,1.22198,1.22318,1.22413,1.22481,1.22522,1.22535,1.22519,1.22474,1.22399,1.22293,1.22156,1.21988,1.21789,1.21558,1.21296,1.21003,1.20679,1.20324,1.19939,1.19525,1.19082,1.18611,1.18114,1.17591,1.17043,1.16473,1.15882,1.15271,1.14642,1.13997,1.13339,1.12668,1.11988,1.113,1.10607,1.0991,1.09212,1.08515,1.07822,1.07134,1.06453,1.05781,1.0512,1.04472,1.03838,1.0322,1.02619,1.02035,1.01471,1.00925,1.00401,0.998962,0.994127,0.989501,0.985084,0.980873,0.976866,0.973056,0.96944,0.966008,0.962755,0.95967,0.956745,0.95397,0.951332,0.948822,0.946427,0.944135,0.941933,0.939808,0.937747,0.935736,0.933761,0.931806,0.929859,0.927902,0.925919,0.923895,0.921812,0.919652,0.917396,0.915025,0.912516,0.909849,0.907,0.903944,0.900655,0.897106,0.893266,0.889104,0.884587,0.879679,0.874344,0.868542,0.862231,0.855368,0.847908,0.839804,0.831008,0.82147,0.811142,0.799974,0.787917,0.774924,0.760951,0.745958,0.729912,0.712784,0.694557,0.675223,0.65479,0.633277,0.610725,0.587191,0.562756,0.537521,0.511614,0.485184,0.458401,0.431458,0.404563,0.377928,0.351775,0.326316,0.301745},
	{1.01582,1.01669,1.0176,1.01855,1.01955,1.0206,1.0217,1.02286,1.02406,1.02532,1.02664,1.02803,1.02947,1.03098,1.03255,1.0342,1.03591,1.0377,1.03956,1.0415,1.04351,1.04561,1.04779,1.05005,1.0524,1.05484,1.05736,1.05997,1.06267,1.06545,1.06833,1.0713,1.07436,1.07751,1.08074,1.08407,1.08748,1.09097,1.09455,1.09821,1.10195,1.10576,1.10964,1.11358,1.11759,1.12166,1.12578,1.12994,1.13415,1.13839,1.14266,1.14694,1.15124,1.15554,1.15984,1.16412,1.16838,1.1726,1.17678,1.18091,1.18497,1.18896,1.19286,1.19666,1.20035,1.20392,1.20735,1.21064,1.21378,1.21674,1.21952,1.22211,1.2245,1.22666,1.22861,1.23031,1.23177,1.23296,1.23389,1.23455,1.23492,1.235,1.23477,1.23425,1.23341,1.23227,1.2308,1.22901,1.22691,1.22448,1.22173,1.21867,1.2153,1.21161,1.20763,1.20335,1.19878,1.19394,1.18884,1.18349,1.1779,1.1721,1.16609,1.15989,1.15353,1.14703,1.1404,1.13366,1.12685,1.11997,1.11306,1.10613,1.0992,1.09231,1.08546,1.07868,1.07199,1.06541,1.05895,1.05263,1.04646,1.04046,1.03463,1.029,1.02355,1.01831,1.01326,1.00843,1.0038,0.999382,0.995167,0.991153,0.987336,0.98371,0.980267,0.977002,0.973903,0.970962,0.968169,0.965512,0.96298,0.960562,0.958243,0.956013,0.953857,0.951762,0.949713,0.947697,0.945699,0.943703,0.941693,0.939653,0.937566,0.935415,0.93318,0.930843,0.928382,0.925778,0.923006,0.920044,0.916865,0.913443,0.909749,0.905752,0.901421,0.896722,0.891618,0.88607,0.88004,0.873485,0.86636,0.85862,0.850218,0.841104,0.83123,0.820545,0.809001,0.796548,0.783141,0.768736,0.753294,0.736784,0.719179,0.700464,0.680636,0.659702,0.63769,0.614643,0.590623,0.565716,0.540029,0.513695,0.486868,0.459724,0.432459,0.405283,0.378412,0.352068,0.326459,0.301781},
	{1.01583,1.0167,1.01761,1.01857,1.01958,1.02064,1.02174,1.0229,1.02411,1.02538,1.02671,1.0281,1.02956,1.03108,1.03267,1.03433,1.03606,1.03786,1.03974,1.0417,1.04374,1.04586,1.04807,1.05036,1.05274,1.05521,1.05777,1.06042,1.06316,1.066,1.06893,1.07195,1.07507,1.07828,1.08159,1.08498,1.08847,1.09205,1.09571,1.09946,1.1033,1.10721,1.1112,1.11526,1.11939,1.12359,1.12784,1.13215,1.1365,1.14089,1.14532,1.14977,1.15424,1.15872,1.1632,1.16768,1.17213,1.17656,1.18095,1.1853,1.18958,1.19379,1.19792,1.20196,1.20589,1.20971,1.21339,1.21693,1.22032,1.22354,1.22658,1.22943,1.23207,1.2345,1.23671,1.23867,1.24039,1.24184,1.24303,1.24393,1.24455,1.24488,1.2449,1.24461,1.24401,1.24309,1.24184,1.24027,1.23837,1.23615,1.2336,1.23072,1.22753,1.22402,1.2202,1.21608,1.21166,1.20697,1.202,1.19678,1.19131,1.18562,1.17972,1.17362,1.16736,1.16094,1.15439,1.14772,1.14098,1.13416,1.1273,1.12042,1.11355,1.10669,1.09988,1.09314,1.08648,1.07992,1.07348,1.06718,1.06103,1.05504,1.04922,1.04359,1.03815,1.03291,1.02787,1.02303,1.0184,1.01398,1.00975,1.00573,1.0019,0.998265,0.994809,0.991527,0.988409,0.985448,0.982632,0.979949,0.977389,0.97494,0.972588,0.97032,0.968123,0.965984,0.963886,0.961817,0.95976,0.9577,0.955621,0.953505,0.951336,0.949094,0.946761,0.944317,0.941741,0.939009,0.9361,0.932988,0.929646,0.926048,0.922162,0.917959,0.913405,0.908463,0.903099,0.897271,0.890939,0.88406,0.876589,0.868478,0.859681,0.850147,0.839827,0.828672,0.816631,0.803657,0.789705,0.774733,0.758704,0.741588,0.723363,0.704018,0.683553,0.661983,0.63934,0.615675,0.591057,0.565579,0.539358,0.512534,0.485269,0.457746,0.430167,0.402746,0.375701,0.349252,0.323607,0.298951},
	{1.01583,1.01671,1.01763,1.01859,1.0196,1.02066,1.02177,1.02294,1.02416,1.02544,1.02677,1.02817,1.02964,1.03117,1.03277,1.03444,1.03619,1.03801,1.03991,1.04189,1.04395,1.0461,1.04833,1.05065,1.05306,1.05557,1.05816,1.06085,1.06364,1.06652,1.0695,1.07257,1.07575,1.07902,1.08239,1.08586,1.08942,1.09308,1.09683,1.10067,1.1046,1.10862,1.11271,1.11689,1.12114,1.12546,1.12985,1.13429,1.13879,1.14334,1.14792,1.15254,1.15719,1.16185,1.16651,1.17118,1.17584,1.18047,1.18507,1.18964,1.19414,1.19859,1.20295,1.20723,1.2114,1.21547,1.2194,1.2232,1.22685,1.23033,1.23363,1.23675,1.23966,1.24236,1.24483,1.24706,1.24905,1.25077,1.25222,1.25338,1.25426,1.25484,1.25511,1.25507,1.25471,1.25403,1.25301,1.25167,1.24999,1.24798,1.24563,1.24295,1.23995,1.23662,1.23298,1.22902,1.22477,1.22022,1.2154,1.21031,1.20498,1.1994,1.19362,1.18763,1.18146,1.17513,1.16867,1.16209,1.15541,1.14866,1.14186,1.13504,1.12821,1.1214,1.11463,1.10793,1.1013,1.09477,1.08835,1.08207,1.07594,1.06997,1.06417,1.05854,1.05311,1.04787,1.04284,1.038,1.03337,1.02894,1.02472,1.02069,1.01685,1.0132,1.00973,1.00644,1.00331,1.00033,0.997492,0.994788,0.992204,0.989728,0.987348,0.985048,0.982817,0.980639,0.978501,0.976386,0.97428,0.972167,0.970029,0.967849,0.96561,0.963293,0.960878,0.958344,0.95567,0.952833,0.949809,0.946572,0.943096,0.939352,0.935309,0.930935,0.926197,0.921058,0.91548,0.909424,0.902847,0.895705,0.887953,0.879543,0.870427,0.860556,0.849878,0.838345,0.825907,0.812517,0.798131,0.782707,0.76621,0.748612,0.729894,0.710047,0.689074,0.666994,0.643843,0.619675,0.594567,0.568617,0.541945,0.514698,0.487042,0.459164,0.431271,0.403579,0.376306,0.349675,0.323887,0.299128},
	{1.01583,1.01671,1.01763,1.0186,1.01962,1.02068,1.0218,1.02297,1.02419,1.02548,1.02683,1.02824,1.02971,1.03125,1.03287,1.03455,1.03631,1.03815,1.04007,1.04206,1.04415,1.04632,1.04857,1.05092,1.05336,1.0559,1.05853,1.06126,1.06408,1.06701,1.07004,1.07316,1.07639,1.07973,1.08316,1.0867,1.09033,1.09407,1.0979,1.10183,1.10586,1.10997,1.11418,1.11847,1.12284,1.12728,1.1318,1.13639,1.14103,1.14573,1.15047,1.15526,1.16007,1.16491,1.16977,1.17463,1.17948,1.18433,1.18914,1.19392,1.19866,1.20333,1.20793,1.21245,1.21688,1.22119,1.22539,1.22945,1.23336,1.23711,1.24068,1.24407,1.24725,1.25023,1.25297,1.25548,1.25774,1.25973,1.26146,1.26289,1.26404,1.26488,1.26542,1.26563,1.26552,1.26508,1.26431,1.2632,1.26175,1.25996,1.25783,1.25536,1.25255,1.24942,1.24596,1.24218,1.23809,1.23371,1.22904,1.22409,1.21889,1.21345,1.20778,1.2019,1.19584,1.18961,1.18324,1.17674,1.17014,1.16346,1.15673,1.14997,1.1432,1.13644,1.12971,1.12304,1.11645,1.10995,1.10356,1.09731,1.09119,1.08524,1.07945,1.07384,1.06842,1.06319,1.05815,1.05332,1.04869,1.04426,1.04003,1.03599,1.03215,1.02849,1.02501,1.0217,1.01855,1.01556,1.0127,1.00998,1.00737,1.00487,1.00246,1.00012,0.997858,0.995642,0.993461,0.991301,0.989145,0.986976,0.984779,0.982534,0.980224,0.97783,0.975331,0.972707,0.969934,0.96699,0.963851,0.960488,0.956876,0.952985,0.948783,0.944238,0.939315,0.933978,0.928186,0.921901,0.915078,0.907674,0.899641,0.890933,0.881501,0.871293,0.86026,0.848353,0.835522,0.821721,0.806905,0.791035,0.774077,0.756005,0.736801,0.71646,0.694988,0.672407,0.648758,0.624098,0.59851,0.572095,0.54498,0.517316,0.489272,0.461043,0.432836,0.404868,0.377362,0.350537,0.324594,0.299714},
	{1.01583,1.01671,1.01763,1.01861,1.01962,1.02069,1.02182,1.02299,1.02422,1.02552,1.02687,1.02829,1.02977,1.03132,1.03295,1.03465,1.03642,1.03827,1.04021,1.04222,1.04433,1.04652,1.0488,1.05117,1.05364,1.05621,1.05887,1.06164,1.0645,1.06747,1.07055,1.07372,1.07701,1.0804,1.0839,1.0875,1.09121,1.09502,1.09894,1.10295,1.10707,1.11129,1.1156,1.12,1.12448,1.12905,1.1337,1.13842,1.14321,1.14806,1.15296,1.15791,1.1629,1.16792,1.17296,1.17802,1.18307,1.18812,1.19315,1.19816,1.20312,1.20803,1.21287,1.21763,1.22231,1.22688,1.23134,1.23566,1.23984,1.24386,1.24771,1.25138,1.25484,1.2581,1.26113,1.26392,1.26646,1.26874,1.27074,1.27246,1.27388,1.275,1.2758,1.27628,1.27643,1.27625,1.27573,1.27486,1.27365,1.27209,1.27018,1.26793,1.26533,1.2624,1.25913,1.25554,1.25163,1.24742,1.24291,1.23812,1.23306,1.22775,1.22221,1.21645,1.2105,1.20438,1.1981,1.19169,1.18518,1.17858,1.17191,1.16521,1.1585,1.15179,1.14512,1.13849,1.13193,1.12547,1.11911,1.11288,1.10679,1.10085,1.09508,1.08948,1.08407,1.07885,1.07382,1.06899,1.06436,1.05993,1.05569,1.05165,1.0478,1.04414,1.04064,1.03732,1.03416,1.03115,1.02827,1.02552,1.02289,1.02036,1.01792,1.01556,1.01326,1.011,1.00878,1.00657,1.00436,1.00214,0.999882,0.997572,0.995191,0.992719,0.990136,0.98742,0.984549,0.981497,0.978241,0.974752,0.971004,0.966965,0.962605,0.957889,0.952781,0.947245,0.941241,0.934728,0.927661,0.919996,0.911686,0.902682,0.892936,0.882396,0.871013,0.858737,0.845519,0.831312,0.816075,0.799768,0.782358,0.763822,0.744143,0.723319,0.701359,0.678289,0.654152,0.629012,0.602953,0.576083,0.548532,0.520456,0.492029,0.463448,0.434923,0.406675,0.378925,0.351893,0.325778,0.300758},
	{1.01582,1.01671,1.01763,1.01861,1.01963,1.0207,1.02183,1.02301,1.02425,1.02555,1.02691,1.02833,1.02982,1.03139,1.03302,1.03473,1.03652,1.03838,1.04033,1.04237,1.04449,1.0467,1.04901,1.0514,1.0539,1.0565,1.05919,1.06199,1.0649,1.06791,1.07103,1.07425,1.07759,1.08104,1.0846,1.08826,1.09204,1.09593,1.09993,1.10403,1.10824,1.11255,1.11696,1.12147,1.12608,1.13077,1.13555,1.1404,1.14533,1.15033,1.15539,1.16051,1.16567,1.17087,1.1761,1.18134,1.1866,1.19186,1.19711,1.20233,1.20752,1.21267,1.21775,1.22277,1.2277,1.23253,1.23725,1.24184,1.24629,1.25059,1.25472,1.25867,1.26243,1.26597,1.26929,1.27237,1.2752,1.27777,1.28006,1.28207,1.28377,1.28517,1.28625,1.28701,1.28743,1.28752,1.28725,1.28664,1.28568,1.28436,1.28269,1.28066,1.27828,1.27556,1.2725,1.2691,1.26538,1.26134,1.257,1.25237,1.24746,1.2423,1.23689,1.23126,1.22543,1.21941,1.21324,1.20693,1.2005,1.19398,1.18739,1.18076,1.17411,1.16746,1.16083,1.15425,1.14774,1.14131,1.13498,1.12878,1.12271,1.11679,1.11104,1.10546,1.10005,1.09484,1.08981,1.08499,1.08035,1.07592,1.07168,1.06763,1.06377,1.06009,1.05658,1.05324,1.05006,1.04703,1.04413,1.04135,1.03869,1.03612,1.03365,1.03124,1.02889,1.02659,1.02431,1.02204,1.01977,1.01748,1.01514,1.01275,1.01027,1.0077,1.00501,1.00218,0.999176,0.995986,0.992579,0.988928,0.985004,0.980775,0.976208,0.971271,0.965925,0.960132,0.953852,0.947043,0.93966,0.931657,0.922987,0.9136,0.903448,0.892478,0.880642,0.867889,0.854171,0.839443,0.823663,0.806794,0.788806,0.769678,0.749397,0.727964,0.705393,0.681715,0.656979,0.631256,0.604636,0.577233,0.549186,0.520655,0.491823,0.46289,0.434071,0.405589,0.377666,0.350518,0.324341,0.299307},
	{1.01581,1.0167,1.01763,1.0186,1.01963,1.02071,1.02184,1.02302,1.02426,1.02557,1.02694,1.02837,1.02987,1.03144,1.03308,1.0348,1.0366,1.03848,1.04045,1.0425,1.04464,1.04687,1.04919,1.05162,1.05414,1.05676,1.05949,1.06232,1.06527,1.06832,1.07148,1.07475,1.07814,1.08164,1.08526,1.08899,1.09284,1.0968,1.10087,1.10506,1.10936,1.11377,1.11828,1.1229,1.12762,1.13243,1.13733,1.14233,1.1474,1.15255,1.15776,1.16304,1.16838,1.17375,1.17917,1.18461,1.19007,1.19554,1.201,1.20645,1.21187,1.21725,1.22258,1.22785,1.23304,1.23813,1.24312,1.24798,1.25272,1.2573,1.26172,1.26596,1.27,1.27384,1.27746,1.28084,1.28396,1.28683,1.28942,1.29172,1.29372,1.29541,1.29679,1.29783,1.29853,1.29889,1.2989,1.29855,1.29785,1.29678,1.29535,1.29356,1.29142,1.28892,1.28606,1.28287,1.27934,1.27549,1.27133,1.26687,1.26212,1.25711,1.25185,1.24635,1.24065,1.23475,1.22869,1.22248,1.21614,1.20971,1.2032,1.19665,1.19006,1.18347,1.1769,1.17037,1.1639,1.15751,1.15122,1.14505,1.13901,1.13311,1.12738,1.12181,1.11642,1.11122,1.1062,1.10138,1.09675,1.09231,1.08807,1.08402,1.08015,1.07646,1.07294,1.06959,1.06639,1.06333,1.06041,1.05761,1.05492,1.05233,1.04982,1.04738,1.04499,1.04264,1.04032,1.038,1.03567,1.03332,1.03092,1.02845,1.0259,1.02324,1.02046,1.01753,1.01442,1.01111,1.00758,1.00379,0.999718,0.99533,0.990593,0.985471,0.979927,0.973921,0.967414,0.960361,0.952718,0.944438,0.935473,0.925774,0.91529,0.903972,0.891767,0.878627,0.864505,0.849356,0.833137,0.815815,0.797361,0.777754,0.756985,0.735057,0.711988,0.687812,0.662582,0.636371,0.609277,0.581415,0.552931,0.523989,0.494773,0.465489,0.436354,0.407591,0.379425,0.352068,0.325716,0.300537},
	{1.0158,1.01668,1.01762,1.0186,1.01962,1.0207,1.02184,1.02303,1.02427,1.02558,1.02696,1.0284,1.0299,1.03148,1.03314,1.03487,1.03668,1.03857,1.04055,1.04261,1.04477,1.04702,1.04937,1.05181,1.05436,1.05701,1.05977,1.06263,1.06561,1.0687,1.0719,1.07522,1.07866,1.08221,1.08589,1.08968,1.0936,1.09763,1.10178,1.10605,1.11044,1.11494,1.11955,1.12427,1.1291,1.13403,1.13907,1.14419,1.1494,1.1547,1.16007,1.16551,1.17102,1.17657,1.18217,1.18781,1.19347,1.19915,1.20483,1.2105,1.21616,1.22178,1.22736,1.23287,1.23832,1.24368,1.24894,1.25409,1.2591,1.26397,1.26868,1.27321,1.27756,1.2817,1.28562,1.2893,1.29273,1.29591,1.2988,1.30141,1.30371,1.30571,1.30737,1.30871,1.3097,1.31035,1.31064,1.31057,1.31013,1.30933,1.30815,1.30661,1.3047,1.30243,1.29981,1.29683,1.2935,1.28985,1.28587,1.28159,1.27701,1.27216,1.26705,1.2617,1.25612,1.25035,1.2444,1.2383,1.23207,1.22573,1.2193,1.21282,1.20631,1.19978,1.19327,1.18679,1.18037,1.17402,1.16777,1.16163,1.15561,1.14974,1.14403,1.13847,1.1331,1.1279,1.12289,1.11807,1.11344,1.109,1.10475,1.10068,1.0968,1.0931,1.08956,1.08618,1.08296,1.07988,1.07692,1.07409,1.07136,1.06873,1.06617,1.06367,1.06123,1.05882,1.05643,1.05403,1.05162,1.04918,1.04668,1.04411,1.04144,1.03866,1.03574,1.03265,1.02939,1.02591,1.02218,1.01819,1.0139,1.00927,1.00428,0.998879,0.993037,0.986713,0.979862,0.972443,0.964408,0.95571,0.9463,0.936129,0.925146,0.913299,0.900539,0.886818,0.872087,0.856304,0.83943,0.821432,0.802284,0.781971,0.760487,0.737842,0.714058,0.689177,0.66326,0.636389,0.608668,0.580223,0.551208,0.521794,0.492174,0.462559,0.433171,0.404233,0.375968,0.348587,0.322276,0.297194},
	{1.01578,1.01667,1.0176,1.01858,1.01961,1.0207,1.02183,1.02303,1.02428,1.02559,1.02697,1.02841,1.02993,1.03152,1.03318,1.03492,1.03674,1.03864,1.04063,1.04271,1.04489,1.04715,1.04952,1.05199,1.05456,1.05724,1.06002,1.06292,1.06593,1.06906,1.0723,1.07566,1.07915,1.08275,1.08648,1.09034,1.09432,1.09842,1.10264,1.10699,1.11146,1.11606,1.12077,1.1256,1.13054,1.13559,1.14074,1.146,1.15135,1.15679,1.16232,1.16792,1.1736,1.17933,1.18512,1.19095,1.19681,1.2027,1.2086,1.2145,1.22038,1.22625,1.23207,1.23785,1.24355,1.24918,1.25472,1.26015,1.26545,1.27061,1.27562,1.28045,1.2851,1.28955,1.29378,1.29777,1.30152,1.30501,1.30822,1.31114,1.31375,1.31606,1.31803,1.31967,1.32097,1.32191,1.32249,1.3227,1.32255,1.32202,1.32111,1.31983,1.31817,1.31614,1.31375,1.31099,1.30789,1.30444,1.30066,1.29657,1.29217,1.28748,1.28253,1.27733,1.2719,1.26626,1.26043,1.25445,1.24832,1.24208,1.23575,1.22935,1.22291,1.21646,1.21001,1.20359,1.19722,1.19092,1.1847,1.1786,1.17262,1.16677,1.16108,1.15555,1.15018,1.145,1.14,1.13518,1.13055,1.12611,1.12186,1.11779,1.1139,1.11019,1.10664,1.10325,1.1,1.0969,1.09392,1.09106,1.0883,1.08562,1.08303,1.08049,1.078,1.07554,1.07309,1.07063,1.06816,1.06564,1.06306,1.0604,1.05764,1.05476,1.05173,1.04853,1.04513,1.04151,1.03764,1.03348,1.02901,1.0242,1.019,1.01338,1.00731,1.00073,0.993615,0.985909,0.977569,0.968546,0.958792,0.948256,0.936888,0.924635,0.911449,0.897281,0.882085,0.865818,0.848443,0.829929,0.810252,0.789398,0.767366,0.744168,0.719832,0.694402,0.667945,0.640547,0.612318,0.58339,0.55392,0.524085,0.494083,0.464127,0.434441,0.40525,0.376777,0.349229,0.32279,0.297612},
	{1.01575,1.01665,1.01758,1.01856,1.0196,1.02068,1.02182,1.02302,1.02428,1.02559,1.02698,1.02843,1.02995,1.03154,1.03321,1.03496,1.03679,1.0387,1.04071,1.0428,1.04499,1.04727,1.04966,1.05215,1.05474,1.05744,1.06025,1.06318,1.06622,1.06939,1.07267,1.07607,1.0796,1.08326,1.08704,1.09096,1.095,1.09917,1.10347,1.10789,1.11245,1.11713,1.12194,1.12687,1.13192,1.13708,1.14236,1.14775,1.15324,1.15882,1.1645,1.17027,1.17611,1.18202,1.18799,1.19402,1.20008,1.20618,1.21229,1.21842,1.22454,1.23065,1.23672,1.24275,1.24873,1.25463,1.26044,1.26616,1.27175,1.27721,1.28251,1.28766,1.29262,1.29738,1.30192,1.30624,1.3103,1.31411,1.31764,1.32089,1.32382,1.32645,1.32874,1.3307,1.3323,1.33355,1.33443,1.33494,1.33507,1.33483,1.3342,1.33318,1.33179,1.33001,1.32786,1.32534,1.32246,1.31923,1.31566,1.31176,1.30755,1.30304,1.29826,1.29322,1.28793,1.28243,1.27674,1.27087,1.26486,1.25873,1.25249,1.24618,1.23983,1.23345,1.22706,1.2207,1.21439,1.20813,1.20196,1.1959,1.18995,1.18413,1.17846,1.17295,1.1676,1.16243,1.15743,1.15262,1.14799,1.14355,1.13929,1.13521,1.13131,1.12758,1.12401,1.12059,1.11732,1.11419,1.11117,1.10827,1.10547,1.10275,1.1001,1.09751,1.09496,1.09242,1.0899,1.08736,1.0848,1.08218,1.0795,1.07672,1.07384,1.07082,1.06764,1.06428,1.06071,1.0569,1.05282,1.04845,1.04375,1.03868,1.0332,1.02729,1.0209,1.01399,1.00651,0.998414,0.989661,0.9802,0.969981,0.958953,0.947064,0.934264,0.920505,0.905736,0.889915,0.873,0.854956,0.835754,0.815375,0.793809,0.77106,0.747145,0.722097,0.69597,0.668837,0.640792,0.611954,0.582462,0.552483,0.522201,0.491819,0.461556,0.431638,0.402291,0.373736,0.346173,0.319779,0.294696},
	{1.01573,1.01662,1.01756,1.01854,1.01958,1.02066,1.02181,1.02301,1.02427,1.02559,1.02697,1.02843,1.02996,1.03156,1.03323,1.03499,1.03683,1.03875,1.04077,1.04287,1.04508,1.04738,1.04978,1.05229,1.0549,1.05763,1.06047,1.06342,1.06649,1.06969,1.07301,1.07646,1.08003,1.08374,1.08757,1.09154,1.09564,1.09988,1.10425,1.10875,1.11339,1.11816,1.12306,1.12809,1.13324,1.13852,1.14392,1.14944,1.15506,1.16079,1.16662,1.17255,1.17856,1.18464,1.1908,1.19702,1.20328,1.20959,1.21593,1.22228,1.22864,1.23498,1.24131,1.2476,1.25384,1.26002,1.26612,1.27212,1.278,1.28376,1.28938,1.29483,1.30011,1.30519,1.31006,1.3147,1.31909,1.32323,1.32709,1.33067,1.33394,1.33689,1.33951,1.34179,1.34372,1.34528,1.34648,1.34729,1.34773,1.34778,1.34743,1.3467,1.34558,1.34407,1.34217,1.3399,1.33726,1.33425,1.3309,1.32721,1.3232,1.31888,1.31427,1.30939,1.30427,1.29892,1.29337,1.28763,1.28174,1.27571,1.26958,1.26337,1.2571,1.2508,1.24449,1.2382,1.23194,1.22574,1.21962,1.2136,1.20769,1.2019,1.19626,1.19077,1.18544,1.18028,1.1753,1.1705,1.16587,1.16143,1.15717,1.15309,1.14918,1.14543,1.14185,1.13841,1.13512,1.13196,1.12892,1.12599,1.12315,1.1204,1.1177,1.11506,1.11246,1.10987,1.10728,1.10468,1.10203,1.09934,1.09657,1.09369,1.09071,1.08757,1.08427,1.08078,1.07707,1.0731,1.06886,1.06431,1.05941,1.05413,1.04844,1.04229,1.03564,1.02846,1.02068,1.01228,1.0032,0.993387,0.982797,0.971377,0.959076,0.945844,0.93163,0.916389,0.900075,0.88265,0.86408,0.844337,0.823406,0.801278,0.777962,0.753477,0.727862,0.701174,0.673491,0.644914,0.615564,0.585589,0.555157,0.524459,0.493703,0.463107,0.432903,0.403314,0.37456,0.34684,0.320325,0.295152},
	{1.0157,1.01659,1.01753,1.01852,1.01955,1.02064,1.02179,1.02299,1.02425,1.02558,1.02697,1.02843,1.02996,1.03156,1.03325,1.03501,1.03686,1.03879,1.04082,1.04294,1.04515,1.04747,1.04988,1.05241,1.05504,1.05779,1.06065,1.06364,1.06674,1.06997,1.07332,1.07681,1.08043,1.08418,1.08806,1.09209,1.09625,1.10055,1.10498,1.10956,1.11428,1.11913,1.12413,1.12925,1.13452,1.13991,1.14543,1.15107,1.15683,1.1627,1.16868,1.17476,1.18094,1.1872,1.19354,1.19995,1.20642,1.21293,1.21949,1.22607,1.23266,1.23925,1.24583,1.25238,1.2589,1.26535,1.27173,1.27802,1.28421,1.29027,1.2962,1.30197,1.30756,1.31297,1.31817,1.32314,1.32788,1.33235,1.33655,1.34046,1.34407,1.34736,1.35032,1.35293,1.35519,1.35708,1.3586,1.35974,1.36049,1.36084,1.3608,1.36036,1.35952,1.35828,1.35665,1.35464,1.35224,1.34948,1.34635,1.34288,1.33907,1.33495,1.33053,1.32584,1.32088,1.31568,1.31028,1.30468,1.29892,1.29301,1.28699,1.28088,1.27471,1.26849,1.26226,1.25604,1.24984,1.2437,1.23763,1.23165,1.22578,1.22003,1.21442,1.20895,1.20364,1.1985,1.19353,1.18873,1.18411,1.17967,1.1754,1.17131,1.16739,1.16363,1.16003,1.15658,1.15326,1.15007,1.147,1.14403,1.14115,1.13834,1.1356,1.1329,1.13023,1.12758,1.12491,1.12223,1.1195,1.1167,1.11383,1.11084,1.10773,1.10446,1.10102,1.09737,1.09348,1.08933,1.0849,1.08013,1.075,1.06948,1.06352,1.05709,1.05013,1.04262,1.0345,1.02573,1.01626,1.00603,0.995001,0.983117,0.970328,0.956583,0.941834,0.926033,0.90914,0.891115,0.871927,0.851552,0.829976,0.807196,0.783224,0.758084,0.731821,0.704498,0.676199,0.647031,0.617124,0.58663,0.555725,0.524606,0.493483,0.462582,0.432132,0.402357,0.373476,0.345681,0.319138,0.293973},
	{1.01567,1.01656,1.0175,1.01849,1.01952,1.02061,1.02176,1.02296,1.02423,1.02556,1.02695,1.02841,1.02995,1.03156,1.03325,1.03502,1.03687,1.03882,1.04085,1.04298,1.04521,1.04754,1.04997,1.05251,1.05517,1.05794,1.06082,1.06383,1.06696,1.07022,1.07361,1.07713,1.08079,1.08459,1.08852,1.0926,1.09681,1.10117,1.10568,1.11033,1.11513,1.12006,1.12515,1.13037,1.13573,1.14124,1.14687,1.15264,1.15853,1.16454,1.17067,1.17691,1.18325,1.18968,1.19621,1.20281,1.20948,1.2162,1.22297,1.22978,1.23661,1.24345,1.25028,1.2571,1.26388,1.27061,1.27728,1.28387,1.29036,1.29673,1.30297,1.30906,1.31499,1.32072,1.32626,1.33157,1.33665,1.34147,1.34602,1.35028,1.35423,1.35787,1.36117,1.36413,1.36673,1.36896,1.37082,1.37228,1.37335,1.37402,1.37429,1.37415,1.37361,1.37266,1.37131,1.36957,1.36743,1.36491,1.36203,1.35879,1.3552,1.35129,1.34707,1.34255,1.33777,1.33275,1.32749,1.32204,1.31642,1.31064,1.30474,1.29873,1.29266,1.28653,1.28039,1.27424,1.26812,1.26204,1.25602,1.25009,1.24426,1.23855,1.23297,1.22753,1.22225,1.21712,1.21216,1.20738,1.20276,1.19833,1.19406,1.18996,1.18603,1.18226,1.17864,1.17517,1.17182,1.16861,1.1655,1.1625,1.15958,1.15673,1.15393,1.15118,1.14845,1.14573,1.14299,1.14022,1.13741,1.13452,1.13154,1.12844,1.12521,1.12181,1.11822,1.11442,1.11037,1.10605,1.10142,1.09644,1.0911,1.08533,1.07912,1.07241,1.06517,1.05735,1.0489,1.03977,1.02993,1.0193,1.00786,0.995533,0.982281,0.968051,0.952796,0.936469,0.919029,0.90044,0.880672,0.859703,0.837523,0.814132,0.789545,0.763791,0.73692,0.709,0.680121,0.650395,0.619957,0.588968,0.557607,0.526074,0.494586,0.46337,0.432655,0.402668,0.373622,0.345707,0.319082,0.293866},
	{1.01563,1.01652,1.01746,1.01845,1.01949,1.02058,1.02173,1.02293,1.0242,1.02553,1.02693,1.02839,1.02993,1.03155,1.03324,1.03502,1.03688,1.03883,1.04088,1.04302,1.04525,1.0476,1.05004,1.0526,1.05527,1.05806,1.06097,1.064,1.06716,1.07045,1.07387,1.07743,1.08113,1.08496,1.08895,1.09307,1.09734,1.10176,1.10634,1.11106,1.11593,1.12095,1.12612,1.13143,1.1369,1.14251,1.14826,1.15414,1.16017,1.16632,1.17259,1.17899,1.18549,1.1921,1.1988,1.20559,1.21246,1.21939,1.22639,1.23342,1.24049,1.24757,1.25466,1.26174,1.26879,1.27581,1.28277,1.28965,1.29645,1.30313,1.30969,1.31611,1.32237,1.32844,1.33432,1.33998,1.3454,1.35058,1.35548,1.3601,1.36441,1.3684,1.37207,1.37538,1.37833,1.38091,1.38311,1.38492,1.38632,1.38732,1.38791,1.38809,1.38785,1.3872,1.38614,1.38468,1.38281,1.38056,1.37792,1.37492,1.37157,1.36787,1.36386,1.35954,1.35495,1.3501,1.34501,1.33971,1.33423,1.32859,1.32281,1.31693,1.31096,1.30493,1.29887,1.29281,1.28676,1.28075,1.27479,1.26891,1.26313,1.25746,1.25192,1.24651,1.24125,1.23615,1.2312,1.22643,1.22182,1.21739,1.21312,1.20902,1.20508,1.20129,1.19766,1.19416,1.1908,1.18755,1.18441,1.18137,1.1784,1.17551,1.17266,1.16985,1.16706,1.16426,1.16145,1.1586,1.1557,1.15271,1.14962,1.14641,1.14305,1.13952,1.13578,1.13182,1.1276,1.12309,1.11826,1.11308,1.1075,1.10149,1.09501,1.08803,1.08048,1.07234,1.06355,1.05407,1.04383,1.03281,1.02093,1.00816,0.994437,0.979714,0.963943,0.947082,0.929088,0.909927,0.889572,0.868005,0.845215,0.821208,0.796003,0.769633,0.742153,0.713635,0.684176,0.653893,0.622927,0.591444,0.559628,0.527685,0.495835,0.464306,0.433329,0.403129,0.373918,0.34588,0.319169,0.293898},
	{1.01559,1.01648,1.01742,1.01841,1.01945,1.02054,1.02169,1.0229,1.02417,1.0255,1.0269,1.02837,1.02991,1.03153,1.03323,1.03501,1.03688,1.03884,1.04089,1.04304,1.04529,1.04764,1.0501,1.05267,1.05536,1.05817,1.0611,1.06415,1.06734,1.07065,1.07411,1.0777,1.08143,1.08531,1.08934,1.09351,1.09784,1.10231,1.10695,1.11174,1.11668,1.12178,1.12703,1.13244,1.13801,1.14372,1.14958,1.15559,1.16174,1.16803,1.17445,1.18099,1.18766,1.19444,1.20132,1.2083,1.21537,1.22251,1.22972,1.23698,1.24429,1.25162,1.25896,1.26631,1.27364,1.28093,1.28818,1.29537,1.30247,1.30947,1.31636,1.32311,1.3297,1.33612,1.34234,1.34836,1.35414,1.35967,1.36494,1.36992,1.3746,1.37896,1.38299,1.38667,1.38998,1.39292,1.39548,1.39763,1.39939,1.40073,1.40165,1.40216,1.40224,1.40191,1.40115,1.39998,1.39839,1.39641,1.39404,1.39129,1.38817,1.38471,1.38092,1.37681,1.37241,1.36775,1.36283,1.3577,1.35237,1.34687,1.34123,1.33546,1.32961,1.32369,1.31773,1.31175,1.30578,1.29984,1.29395,1.28813,1.2824,1.27678,1.27127,1.2659,1.26066,1.25558,1.25066,1.2459,1.2413,1.23687,1.2326,1.2285,1.22455,1.22075,1.2171,1.21359,1.2102,1.20692,1.20375,1.20067,1.19766,1.19472,1.19182,1.18895,1.1861,1.18323,1.18034,1.17741,1.17441,1.17133,1.16813,1.1648,1.16132,1.15765,1.15377,1.14965,1.14526,1.14056,1.13553,1.13014,1.12433,1.11808,1.11134,1.10408,1.09624,1.08778,1.07865,1.06881,1.0582,1.04678,1.03448,1.02127,1.00708,0.991879,0.975606,0.958223,0.939689,0.919973,0.899048,0.876898,0.853517,0.828913,0.803109,0.776141,0.74807,0.718972,0.688949,0.658124,0.626642,0.594674,0.56241,0.530058,0.497842,0.465992,0.434739,0.404309,0.374908,0.34672,0.319892,0.29453},
	{1.01555,1.01644,1.01738,1.01837,1.01941,1.0205,1.02165,1.02285,1.02412,1.02546,1.02686,1.02833,1.02988,1.0315,1.0332,1.03499,1.03686,1.03883,1.04089,1.04304,1.0453,1.04767,1.05014,1.05273,1.05543,1.05825,1.0612,1.06428,1.06749,1.07083,1.07431,1.07794,1.08171,1.08562,1.08969,1.09391,1.09829,1.10282,1.10752,1.11237,1.11739,1.12256,1.1279,1.1334,1.13906,1.14488,1.15085,1.15698,1.16325,1.16967,1.17623,1.18293,1.18976,1.19671,1.20377,1.21093,1.2182,1.22555,1.23297,1.24046,1.248,1.25558,1.26318,1.27079,1.2784,1.28598,1.29352,1.30101,1.30843,1.31575,1.32296,1.33005,1.33698,1.34375,1.35033,1.3567,1.36284,1.36875,1.37438,1.37974,1.38479,1.38953,1.39393,1.39798,1.40167,1.40498,1.4079,1.41042,1.41254,1.41423,1.4155,1.41635,1.41676,1.41675,1.4163,1.41544,1.41415,1.41245,1.41035,1.40787,1.405,1.40178,1.39822,1.39433,1.39014,1.38566,1.38094,1.37598,1.37081,1.36546,1.35995,1.35432,1.34859,1.34278,1.33692,1.33103,1.32515,1.31928,1.31346,1.3077,1.30203,1.29645,1.29098,1.28564,1.28044,1.27538,1.27048,1.26573,1.26114,1.25671,1.25244,1.24833,1.24437,1.24056,1.23689,1.23335,1.22992,1.22661,1.2234,1.22027,1.21722,1.21422,1.21125,1.20831,1.20538,1.20243,1.19945,1.19642,1.19331,1.1901,1.18677,1.18331,1.17967,1.17583,1.17177,1.16746,1.16286,1.15794,1.15267,1.14702,1.14093,1.13439,1.12733,1.11973,1.11153,1.10268,1.09315,1.08287,1.07181,1.0599,1.0471,1.03335,1.01861,1.00282,0.985941,0.967929,0.948747,0.928364,0.906757,0.883914,0.859832,0.834523,0.808015,0.780352,0.751598,0.721839,0.691181,0.659754,0.627712,0.595232,0.562509,0.529758,0.497204,0.465081,0.433619,0.403042,0.373553,0.345325,0.318501,0.293175},
	{1.0155,1.01639,1.01733,1.01832,1.01936,1.02045,1.0216,1.02281,1.02408,1.02541,1.02682,1.02829,1.02984,1.03146,1.03317,1.03496,1.03684,1.03881,1.04087,1.04304,1.04531,1.04768,1.05016,1.05276,1.05548,1.05832,1.06129,1.06438,1.06761,1.07098,1.07449,1.07815,1.08195,1.0859,1.09001,1.09428,1.09871,1.10329,1.10805,1.11296,1.11805,1.1233,1.12872,1.1343,1.14006,1.14597,1.15205,1.1583,1.1647,1.17125,1.17795,1.1848,1.19178,1.1989,1.20613,1.21349,1.22095,1.2285,1.23615,1.24386,1.25164,1.25946,1.26732,1.2752,1.28308,1.29095,1.29879,1.30658,1.31431,1.32196,1.3295,1.33692,1.34421,1.35133,1.35827,1.365,1.37152,1.3778,1.38381,1.38955,1.39499,1.40011,1.40489,1.40933,1.4134,1.4171,1.4204,1.42329,1.42577,1.42783,1.42946,1.43066,1.43142,1.43174,1.43163,1.43108,1.4301,1.4287,1.42689,1.42468,1.42208,1.41911,1.41578,1.41213,1.40815,1.40389,1.39935,1.39458,1.38958,1.38439,1.37904,1.37354,1.36794,1.36225,1.3565,1.35071,1.34492,1.33913,1.33339,1.3277,1.32208,1.31655,1.31114,1.30584,1.30067,1.29564,1.29075,1.28602,1.28144,1.27702,1.27275,1.26864,1.26467,1.26085,1.25716,1.25359,1.25015,1.24681,1.24356,1.24039,1.23728,1.23423,1.23121,1.2282,1.2252,1.22217,1.2191,1.21597,1.21276,1.20945,1.206,1.2024,1.19862,1.19463,1.19041,1.18592,1.18113,1.17601,1.17052,1.16463,1.15829,1.15148,1.14414,1.13623,1.1277,1.11851,1.10861,1.09795,1.08648,1.07414,1.06088,1.04666,1.03142,1.01512,0.997699,0.979131,0.959376,0.938404,0.916195,0.892739,0.868037,0.842105,0.814973,0.786691,0.757327,0.726972,0.695738,0.663761,0.631198,0.598233,0.565064,0.53191,0.498999,0.466566,0.434842,0.404049,0.374385,0.346022,0.319094,0.293689},
	{1.01545,1.01634,1.01728,1.01827,1.0193,1.0204,1.02155,1.02275,1.02402,1.02536,1.02676,1.02824,1.02979,1.03142,1.03313,1.03492,1.0368,1.03878,1.04085,1.04302,1.04529,1.04768,1.05017,1.05278,1.05551,1.05837,1.06135,1.06447,1.06772,1.07111,1.07465,1.07833,1.08216,1.08615,1.0903,1.09461,1.09908,1.10372,1.10853,1.11351,1.11866,1.12399,1.12948,1.13515,1.141,1.14701,1.1532,1.15955,1.16607,1.17276,1.1796,1.18659,1.19373,1.20101,1.20842,1.21596,1.22362,1.23138,1.23923,1.24717,1.25519,1.26326,1.27137,1.27952,1.28768,1.29583,1.30397,1.31207,1.32012,1.32809,1.33597,1.34373,1.35137,1.35885,1.36615,1.37326,1.38015,1.38681,1.39321,1.39934,1.40517,1.41068,1.41586,1.4207,1.42516,1.42925,1.43294,1.43622,1.43908,1.44152,1.44352,1.44508,1.4462,1.44687,1.44709,1.44688,1.44622,1.44513,1.44362,1.44169,1.43937,1.43667,1.4336,1.43018,1.42643,1.42238,1.41806,1.41347,1.40865,1.40363,1.39844,1.39309,1.38762,1.38206,1.37642,1.37074,1.36504,1.35935,1.35368,1.34806,1.34251,1.33704,1.33167,1.32641,1.32128,1.31627,1.31141,1.3067,1.30213,1.29772,1.29345,1.28933,1.28535,1.28151,1.2778,1.27421,1.27073,1.26735,1.26406,1.26085,1.25769,1.25457,1.25148,1.2484,1.24531,1.24219,1.23903,1.23579,1.23246,1.22902,1.22543,1.22168,1.21774,1.21358,1.20917,1.20447,1.19946,1.19411,1.18837,1.18221,1.17559,1.16846,1.16079,1.15253,1.14363,1.13405,1.12373,1.11262,1.10068,1.08785,1.07408,1.05932,1.04351,1.02662,1.00859,0.989394,0.96899,0.947354,0.924467,0.900324,0.874928,0.848301,0.820478,0.791514,0.761483,0.730481,0.698629,0.666067,0.63296,0.599497,0.565882,0.532337,0.499094,0.46639,0.434453,0.403502,0.373733,0.34531,0.318357,0.292956},
	{1.01539,1.01628,1.01722,1.01821,1.01925,1.02034,1.02149,1.02269,1.02397,1.0253,1.02671,1.02818,1.02973,1.03136,1.03307,1.03487,1.03676,1.03873,1.04081,1.04299,1.04527,1.04766,1.05016,1.05278,1.05553,1.0584,1.06139,1.06453,1.0678,1.07121,1.07477,1.07848,1.08235,1.08637,1.09056,1.09491,1.09943,1.10411,1.10898,1.11401,1.11923,1.12462,1.13019,1.13595,1.14188,1.14799,1.15428,1.16075,1.16738,1.17419,1.18117,1.18831,1.1956,1.20304,1.21063,1.21835,1.2262,1.23416,1.24223,1.2504,1.25865,1.26696,1.27533,1.28375,1.29218,1.30063,1.30907,1.31748,1.32584,1.33414,1.34236,1.35047,1.35846,1.3663,1.37398,1.38147,1.38874,1.39579,1.40259,1.40911,1.41534,1.42126,1.42684,1.43208,1.43695,1.44144,1.44553,1.44921,1.45247,1.4553,1.45768,1.45962,1.4611,1.46214,1.46272,1.46284,1.46252,1.46175,1.46055,1.45893,1.4569,1.45447,1.45166,1.4485,1.445,1.44117,1.43706,1.43268,1.42805,1.42321,1.41819,1.41299,1.40767,1.40224,1.39673,1.39116,1.38557,1.37997,1.37438,1.36884,1.36336,1.35795,1.35263,1.34742,1.34232,1.33735,1.33252,1.32782,1.32327,1.31886,1.3146,1.31048,1.30649,1.30264,1.29891,1.2953,1.29179,1.28837,1.28504,1.28177,1.27856,1.27538,1.27223,1.26907,1.2659,1.2627,1.25943,1.25609,1.25264,1.24907,1.24535,1.24145,1.23735,1.23302,1.22842,1.22353,1.21831,1.21272,1.20673,1.20031,1.19341,1.18598,1.17799,1.16939,1.16013,1.15016,1.13944,1.12791,1.11552,1.10221,1.08794,1.07266,1.05632,1.03886,1.02025,1.00046,0.979434,0.957165,0.933634,0.908837,0.882783,0.855496,0.827016,0.797403,0.766737,0.735118,0.702673,0.669549,0.635915,0.601965,0.567909,0.533971,0.500386,0.46739,0.435213,0.404071,0.374155,0.345622,0.318592,0.293138},
	{1.01533,1.01622,1.01716,1.01815,1.01918,1.02027,1.02142,1.02263,1.0239,1.02524,1.02664,1.02812,1.02967,1.0313,1.03301,1.03481,1.0367,1.03868,1.04076,1.04294,1.04523,1.04763,1.05014,1.05277,1.05552,1.0584,1.06142,1.06456,1.06785,1.07129,1.07487,1.07861,1.0825,1.08656,1.09078,1.09517,1.09973,1.10446,1.10938,1.11447,1.11975,1.12521,1.13085,1.13669,1.1427,1.14891,1.1553,1.16187,1.16863,1.17556,1.18267,1.18995,1.19739,1.205,1.21275,1.22066,1.2287,1.23686,1.24514,1.25353,1.26201,1.27058,1.2792,1.28788,1.2966,1.30534,1.31407,1.3228,1.33148,1.34011,1.34867,1.35714,1.36548,1.37369,1.38174,1.38961,1.39728,1.40472,1.41192,1.41885,1.42549,1.43182,1.43782,1.44347,1.44876,1.45366,1.45816,1.46225,1.46592,1.46914,1.47193,1.47425,1.47612,1.47753,1.47848,1.47896,1.47898,1.47855,1.47768,1.47637,1.47464,1.4725,1.46998,1.46708,1.46382,1.46024,1.45636,1.45219,1.44777,1.44311,1.43826,1.43324,1.42807,1.42278,1.4174,1.41195,1.40647,1.40097,1.39547,1.39001,1.3846,1.37926,1.374,1.36883,1.36378,1.35885,1.35404,1.34937,1.34483,1.34043,1.33617,1.33205,1.32805,1.32418,1.32043,1.31679,1.31325,1.3098,1.30642,1.30311,1.29984,1.29659,1.29337,1.29013,1.28687,1.28357,1.2802,1.27674,1.27317,1.26947,1.2656,1.26154,1.25727,1.25275,1.24796,1.24285,1.2374,1.23157,1.22532,1.21861,1.21141,1.20367,1.19534,1.18637,1.17673,1.16636,1.15521,1.14322,1.13035,1.11655,1.10176,1.08593,1.06902,1.05097,1.03175,1.01132,0.989655,0.966726,0.942523,0.917046,0.890308,0.862338,0.833179,0.802897,0.771578,0.739329,0.70628,0.672586,0.638421,0.603984,0.56949,0.535166,0.50125,0.467978,0.435578,0.404263,0.37422,0.345599,0.318513,0.293027},
	{1.01527,1.01616,1.0171,1.01808,1.01912,1.02021,1.02135,1.02256,1.02383,1.02516,1.02657,1.02804,1.0296,1.03123,1.03294,1.03474,1.03663,1.03862,1.0407,1.04288,1.04518,1.04758,1.0501,1.05274,1.0555,1.05839,1.06142,1.06458,1.06789,1.07134,1.07494,1.0787,1.08263,1.08671,1.09096,1.09539,1.09999,1.10477,1.10973,1.11488,1.12022,1.12574,1.13146,1.13737,1.14347,1.14977,1.15625,1.16293,1.1698,1.17686,1.18409,1.19151,1.19911,1.20687,1.21479,1.22288,1.2311,1.23947,1.24796,1.25657,1.26529,1.27409,1.28298,1.29193,1.30092,1.30995,1.31899,1.32802,1.33703,1.346,1.3549,1.36372,1.37243,1.38101,1.38944,1.3977,1.40576,1.41361,1.42121,1.42855,1.43561,1.44236,1.44878,1.45486,1.46057,1.4659,1.47083,1.47534,1.47942,1.48306,1.48625,1.48899,1.49125,1.49305,1.49437,1.49523,1.49561,1.49553,1.495,1.49402,1.49261,1.49077,1.48853,1.48591,1.48293,1.4796,1.47595,1.47201,1.4678,1.46334,1.45868,1.45383,1.44882,1.44368,1.43844,1.43312,1.42776,1.42236,1.41697,1.4116,1.40626,1.40099,1.39579,1.39068,1.38568,1.38078,1.37601,1.37136,1.36684,1.36245,1.35819,1.35407,1.35007,1.34619,1.34242,1.33875,1.33518,1.33169,1.32826,1.3249,1.32157,1.31826,1.31496,1.31164,1.3083,1.30489,1.30142,1.29784,1.29415,1.2903,1.28629,1.28207,1.27762,1.27292,1.26792,1.2626,1.25692,1.25084,1.24433,1.23734,1.22983,1.22177,1.21309,1.20377,1.19374,1.18296,1.17138,1.15895,1.14561,1.13131,1.116,1.09963,1.08215,1.06352,1.0437,1.02265,1.00034,0.976762,0.951898,0.925754,0.898344,0.869703,0.839879,0.808942,0.776983,0.744114,0.710473,0.676217,0.641528,0.60661,0.571679,0.536968,0.502714,0.469156,0.43652,0.405016,0.374824,0.346092,0.318922,0.293375},
	{1.01521,1.01609,1.01703,1.01801,1.01904,1.02013,1.02128,1.02248,1.02375,1.02509,1.02649,1.02796,1.02951,1.03115,1.03286,1.03466,1.03655,1.03854,1.04062,1.04281,1.04511,1.04752,1.05004,1.05269,1.05546,1.05836,1.0614,1.06458,1.0679,1.07137,1.07499,1.07877,1.08272,1.08683,1.09112,1.09558,1.10022,1.10504,1.11005,1.11525,1.12064,1.12623,1.13201,1.138,1.14418,1.15056,1.15714,1.16392,1.1709,1.17808,1.18544,1.193,1.20074,1.20866,1.21675,1.225,1.23342,1.24198,1.25069,1.25952,1.26846,1.27751,1.28665,1.29587,1.30514,1.31446,1.3238,1.33314,1.34248,1.35178,1.36103,1.3702,1.37928,1.38824,1.39706,1.40571,1.41417,1.42243,1.43045,1.43821,1.44569,1.45287,1.45973,1.46624,1.47239,1.47815,1.48351,1.48846,1.49297,1.49704,1.50065,1.5038,1.50647,1.50867,1.51039,1.51163,1.51239,1.51267,1.51249,1.51185,1.51077,1.50925,1.50732,1.50499,1.50228,1.49921,1.49581,1.49211,1.48812,1.48388,1.47941,1.47474,1.46991,1.46493,1.45983,1.45465,1.44941,1.44413,1.43884,1.43356,1.42831,1.42311,1.41797,1.41292,1.40796,1.40311,1.39836,1.39374,1.38924,1.38486,1.38061,1.37648,1.37247,1.36857,1.36477,1.36108,1.35747,1.35393,1.35045,1.34703,1.34363,1.34025,1.33686,1.33345,1.33,1.32649,1.32289,1.31918,1.31534,1.31134,1.30715,1.30275,1.29811,1.29319,1.28796,1.28239,1.27645,1.27009,1.26327,1.25596,1.24811,1.23968,1.23062,1.22089,1.21042,1.19919,1.18712,1.17417,1.16029,1.14543,1.12953,1.11255,1.09444,1.07515,1.05465,1.0329,1.00989,0.985582,0.959983,0.933097,0.904945,0.875566,0.845013,0.813364,0.780714,0.747184,0.712916,0.678076,0.642852,0.607451,0.572095,0.537021,0.502468,0.468672,0.435858,0.404232,0.373968,0.345203,0.318034,0.292508},
	{1.01514,1.01602,1.01695,1.01794,1.01897,1.02005,1.0212,1.0224,1.02367,1.025,1.0264,1.02788,1.02943,1.03106,1.03277,1.03457,1.03646,1.03845,1.04054,1.04273,1.04503,1.04744,1.04997,1.05262,1.0554,1.05831,1.06136,1.06455,1.06788,1.07137,1.07501,1.07881,1.08278,1.08692,1.09123,1.09573,1.1004,1.10527,1.11032,1.11557,1.12102,1.12666,1.13251,1.13857,1.14483,1.1513,1.15797,1.16485,1.17194,1.17922,1.18672,1.1944,1.20229,1.21036,1.21861,1.22704,1.23564,1.2444,1.25331,1.26236,1.27154,1.28083,1.29022,1.2997,1.30926,1.31886,1.32851,1.33817,1.34783,1.35747,1.36707,1.3766,1.38605,1.39539,1.4046,1.41365,1.42252,1.43119,1.43963,1.44782,1.45574,1.46335,1.47065,1.47761,1.4842,1.49041,1.49622,1.50161,1.50657,1.51107,1.51512,1.5187,1.5218,1.52441,1.52654,1.52818,1.52932,1.52999,1.53018,1.52989,1.52915,1.52797,1.52635,1.52432,1.52191,1.51912,1.51598,1.51253,1.50878,1.50476,1.5005,1.49602,1.49137,1.48656,1.48162,1.47658,1.47147,1.46632,1.46114,1.45596,1.4508,1.44568,1.44062,1.43563,1.43072,1.42591,1.42121,1.41661,1.41213,1.40777,1.40353,1.3994,1.39538,1.39147,1.38766,1.38393,1.38029,1.37671,1.37319,1.36971,1.36625,1.36279,1.35933,1.35583,1.35229,1.34867,1.34495,1.34112,1.33714,1.33299,1.32865,1.32408,1.31925,1.31413,1.30869,1.30289,1.2967,1.29007,1.28298,1.27537,1.2672,1.25843,1.24902,1.2389,1.22804,1.21638,1.20387,1.19045,1.17608,1.16071,1.14428,1.12674,1.10805,1.08817,1.06705,1.04468,1.02102,0.996055,0.969791,0.942235,0.913411,0.883362,0.852146,0.819845,0.78656,0.752417,0.717563,0.682168,0.646426,0.610548,0.574761,0.539301,0.504411,0.470326,0.437269,0.405442,0.375016,0.346122,0.318849,0.293239},
	{1.01506,1.01595,1.01688,1.01786,1.01889,1.01997,1.02111,1.02231,1.02358,1.02491,1.02631,1.02778,1.02933,1.03096,1.03267,1.03447,1.03636,1.03835,1.04044,1.04263,1.04493,1.04735,1.04988,1.05254,1.05532,1.05824,1.0613,1.0645,1.06784,1.07134,1.075,1.07882,1.08281,1.08697,1.09132,1.09584,1.10055,1.10545,1.11055,1.11584,1.12134,1.12705,1.13296,1.13908,1.14542,1.15197,1.15873,1.1657,1.17289,1.1803,1.18791,1.19573,1.20375,1.21197,1.22039,1.22899,1.23777,1.24672,1.25584,1.2651,1.27451,1.28404,1.29369,1.30343,1.31326,1.32316,1.33311,1.34308,1.35307,1.36305,1.373,1.38289,1.39272,1.40244,1.41204,1.4215,1.43078,1.43987,1.44874,1.45737,1.46572,1.47379,1.48154,1.48895,1.496,1.50266,1.50893,1.51478,1.52019,1.52515,1.52965,1.53367,1.5372,1.54025,1.54279,1.54484,1.5464,1.54746,1.54802,1.54811,1.54773,1.54689,1.5456,1.5439,1.54178,1.53928,1.53642,1.53323,1.52973,1.52594,1.5219,1.51763,1.51317,1.50854,1.50377,1.49888,1.49391,1.48889,1.48382,1.47875,1.47369,1.46865,1.46367,1.45874,1.45389,1.44913,1.44447,1.4399,1.43545,1.4311,1.42686,1.42274,1.41871,1.41479,1.41095,1.4072,1.40352,1.39989,1.39632,1.39277,1.38925,1.38571,1.38216,1.37857,1.37492,1.37119,1.36734,1.36337,1.35925,1.35494,1.35041,1.34565,1.34062,1.33528,1.32961,1.32356,1.3171,1.31019,1.30279,1.29486,1.28634,1.27721,1.2674,1.25687,1.24558,1.23346,1.22047,1.20655,1.19165,1.17572,1.15871,1.14058,1.12127,1.10075,1.07897,1.05592,1.03157,1.00591,0.978938,0.95067,0.921134,0.890377,0.858464,0.82548,0.791534,0.756755,0.721298,0.685338,0.649074,0.612722,0.576511,0.540682,0.505476,0.47113,0.437863,0.405873,0.375325,0.346344,0.31901,0.293358},
	{1.01499,1.01587,1.0168,1.01777,1.0188,1.01988,1.02102,1.02222,1.02348,1.02481,1.02621,1.02768,1.02922,1.03085,1.03256,1.03436,1.03625,1.03824,1.04033,1.04252,1.04483,1.04724,1.04978,1.05244,1.05523,1.05815,1.06122,1.06442,1.06778,1.07129,1.07497,1.0788,1.08281,1.087,1.09136,1.09591,1.10066,1.10559,1.11073,1.11607,1.12162,1.12738,1.13335,1.13954,1.14594,1.15257,1.15942,1.16649,1.17378,1.18129,1.18902,1.19697,1.20513,1.2135,1.22207,1.23084,1.2398,1.24894,1.25826,1.26774,1.27737,1.28714,1.29704,1.30705,1.31716,1.32734,1.33759,1.34788,1.3582,1.36852,1.37882,1.38908,1.39928,1.40939,1.41939,1.42926,1.43896,1.44848,1.45778,1.46685,1.47566,1.48418,1.49239,1.50026,1.50777,1.51491,1.52165,1.52796,1.53384,1.53927,1.54422,1.5487,1.55269,1.55618,1.55916,1.56164,1.56361,1.56508,1.56604,1.56652,1.56651,1.56603,1.56509,1.56371,1.56192,1.55972,1.55715,1.55423,1.55099,1.54745,1.54364,1.53959,1.53533,1.53089,1.52629,1.52157,1.51675,1.51186,1.50693,1.50197,1.49701,1.49207,1.48716,1.48231,1.47752,1.47282,1.46819,1.46367,1.45924,1.45491,1.45069,1.44656,1.44253,1.43859,1.43474,1.43096,1.42724,1.42357,1.41995,1.41634,1.41275,1.40914,1.4055,1.40182,1.39806,1.39421,1.39024,1.38613,1.38186,1.37739,1.37269,1.36775,1.36251,1.35696,1.35105,1.34476,1.33803,1.33084,1.32314,1.31489,1.30604,1.29654,1.28636,1.27543,1.26371,1.25114,1.23768,1.22327,1.20786,1.1914,1.17383,1.15512,1.13521,1.11407,1.09167,1.06797,1.04295,1.01662,0.988966,0.960014,0.929794,0.898358,0.865774,0.832132,0.797546,0.762151,0.726107,0.689593,0.652813,0.615986,0.579346,0.543135,0.507594,0.472959,0.43945,0.407258,0.376544,0.347428,0.319983,0.294238},
	{1.01491,1.01579,1.01671,1.01768,1.01871,1.01979,1.02092,1.02212,1.02338,1.0247,1.0261,1.02757,1.02911,1.03074,1.03245,1.03424,1.03613,1.03812,1.04021,1.0424,1.0447,1.04712,1.04966,1.05232,1.05512,1.05804,1.06111,1.06433,1.06769,1.07122,1.0749,1.07876,1.08278,1.08699,1.09138,1.09595,1.10072,1.1057,1.11087,1.11625,1.12184,1.12765,1.13368,1.13993,1.14641,1.15311,1.16004,1.1672,1.17459,1.18221,1.19006,1.19813,1.20642,1.21493,1.22366,1.23259,1.24173,1.25106,1.26057,1.27027,1.28012,1.29013,1.30028,1.31055,1.32093,1.33141,1.34196,1.35257,1.36321,1.37387,1.38453,1.39515,1.40573,1.41623,1.42663,1.43691,1.44704,1.45699,1.46673,1.47625,1.48551,1.4945,1.50317,1.51152,1.51951,1.52713,1.53434,1.54114,1.5475,1.5534,1.55883,1.56378,1.56823,1.57217,1.57561,1.57853,1.58094,1.58283,1.5842,1.58508,1.58545,1.58535,1.58477,1.58374,1.58228,1.5804,1.57813,1.5755,1.57253,1.56925,1.56568,1.56186,1.55781,1.55357,1.54916,1.54461,1.53995,1.5352,1.5304,1.52556,1.5207,1.51586,1.51104,1.50626,1.50154,1.49689,1.49231,1.48782,1.48342,1.47911,1.47489,1.47077,1.46673,1.46278,1.45889,1.45508,1.45132,1.4476,1.44392,1.44024,1.43657,1.43287,1.42914,1.42534,1.42146,1.41748,1.41337,1.40911,1.40466,1.40001,1.39512,1.38995,1.38449,1.37869,1.37252,1.36595,1.35892,1.35141,1.34337,1.33475,1.32551,1.31561,1.30499,1.29361,1.28141,1.26834,1.25435,1.23938,1.22339,1.20632,1.18813,1.16877,1.14819,1.12636,1.10324,1.07881,1.05306,1.02598,0.997572,0.967863,0.936891,0.90471,0.871395,0.837043,0.801772,0.765725,0.729067,0.691983,0.654683,0.61739,0.580342,0.543781,0.507952,0.473086,0.4394,0.407082,0.376283,0.347116,0.319646,0.293892},
	{1.01482,1.0157,1.01662,1.01759,1.01861,1.01969,1.02082,1.02202,1.02327,1.02459,1.02598,1.02745,1.02899,1.03061,1.03232,1.03411,1.036,1.03798,1.04007,1.04226,1.04456,1.04698,1.04952,1.05219,1.05498,1.05791,1.06099,1.06421,1.06758,1.07112,1.07481,1.07868,1.08272,1.08694,1.09135,1.09595,1.10075,1.10575,1.11096,1.11638,1.12202,1.12788,1.13396,1.14027,1.14681,1.15359,1.1606,1.16784,1.17533,1.18305,1.191,1.1992,1.20762,1.21627,1.22515,1.23424,1.24355,1.25307,1.26278,1.27268,1.28276,1.293,1.30339,1.31393,1.32458,1.33535,1.3462,1.35712,1.36809,1.3791,1.39011,1.4011,1.41206,1.42296,1.43376,1.44446,1.45501,1.4654,1.47559,1.48556,1.49529,1.50474,1.5139,1.52273,1.5312,1.53931,1.54701,1.5543,1.56115,1.56754,1.57346,1.57889,1.58382,1.58824,1.59214,1.59552,1.59837,1.6007,1.60251,1.60379,1.60457,1.60486,1.60466,1.60399,1.60287,1.60133,1.59938,1.59705,1.59436,1.59135,1.58803,1.58445,1.58063,1.57659,1.57238,1.56801,1.56351,1.55892,1.55426,1.54954,1.54481,1.54006,1.53534,1.53064,1.52599,1.52139,1.51687,1.51241,1.50804,1.50375,1.49954,1.49542,1.49137,1.4874,1.48349,1.47964,1.47584,1.47207,1.46832,1.46457,1.46081,1.45702,1.45318,1.44927,1.44527,1.44114,1.43688,1.43245,1.42783,1.42298,1.41788,1.4125,1.40679,1.40074,1.39429,1.38742,1.38008,1.37223,1.36383,1.35484,1.3452,1.33487,1.3238,1.31195,1.29925,1.28565,1.27111,1.25558,1.23899,1.2213,1.20246,1.18242,1.16115,1.13861,1.11476,1.08959,1.06309,1.03524,1.00607,0.97559,0.943855,0.91092,0.876867,0.841799,0.80584,0.769139,0.731868,0.694217,0.656402,0.61865,0.581202,0.544302,0.508194,0.473107,0.439256,0.40682,0.375946,0.346736,0.319246,0.293489},
	{1.01474,1.01561,1.01653,1.0175,1.01851,1.01959,1.02071,1.0219,1.02316,1.02447,1.02586,1.02732,1.02886,1.03048,1.03218,1.03397,1.03586,1.03784,1.03992,1.04211,1.04441,1.04683,1.04937,1.05203,1.05483,1.05777,1.06084,1.06407,1.06745,1.07099,1.0747,1.07857,1.08263,1.08687,1.09129,1.09592,1.10074,1.10577,1.11101,1.11647,1.12214,1.12805,1.13418,1.14055,1.14715,1.154,1.16108,1.16841,1.17598,1.1838,1.19187,1.20018,1.20873,1.21752,1.22654,1.23579,1.24527,1.25497,1.26487,1.27498,1.28527,1.29575,1.30639,1.31718,1.32811,1.33916,1.35031,1.36155,1.37285,1.38419,1.39556,1.40693,1.41827,1.42956,1.44077,1.45189,1.46287,1.4737,1.48435,1.49478,1.50498,1.51491,1.52455,1.53387,1.54285,1.55145,1.55966,1.56745,1.5748,1.5817,1.58812,1.59405,1.59947,1.60438,1.60876,1.61261,1.61593,1.61871,1.62095,1.62267,1.62387,1.62456,1.62475,1.62446,1.6237,1.62251,1.62089,1.61887,1.61649,1.61376,1.61071,1.60738,1.6038,1.59998,1.59597,1.5918,1.58748,1.58305,1.57853,1.57396,1.56934,1.56471,1.56008,1.55547,1.55089,1.54636,1.54189,1.53748,1.53314,1.52887,1.52468,1.52056,1.51651,1.51252,1.50859,1.5047,1.50086,1.49703,1.49322,1.4894,1.48556,1.48168,1.47773,1.47371,1.46958,1.46532,1.4609,1.45631,1.45151,1.44647,1.44116,1.43555,1.42961,1.4233,1.41658,1.40942,1.40177,1.39359,1.38484,1.37547,1.36544,1.35469,1.34318,1.33086,1.31767,1.30357,1.2885,1.2724,1.25523,1.23693,1.21746,1.19678,1.17484,1.15161,1.12707,1.10118,1.07394,1.04536,1.01545,0.984233,0.951762,0.918101,0.883337,0.847577,0.810952,0.773616,0.735746,0.697538,0.659213,0.621,0.583143,0.545889,0.50948,0.474146,0.440093,0.4075,0.376505,0.347204,0.319645,0.293833},
	{1.01465,1.01551,1.01643,1.01739,1.01841,1.01948,1.0206,1.02179,1.02303,1.02435,1.02573,1.02719,1.02872,1.03034,1.03204,1.03382,1.0357,1.03768,1.03976,1.04195,1.04425,1.04666,1.0492,1.05186,1.05466,1.0576,1.06068,1.06391,1.06729,1.07084,1.07455,1.07844,1.0825,1.08676,1.0912,1.09584,1.10069,1.10574,1.11101,1.1165,1.12222,1.12816,1.13434,1.14076,1.14743,1.15434,1.16149,1.1689,1.17656,1.18448,1.19265,1.20107,1.20974,1.21866,1.22783,1.23724,1.24688,1.25676,1.26685,1.27716,1.28767,1.29837,1.30926,1.3203,1.3315,1.34284,1.35429,1.36584,1.37747,1.38916,1.40088,1.41262,1.42434,1.43603,1.44766,1.4592,1.47062,1.48189,1.493,1.5039,1.51458,1.525,1.53513,1.54495,1.55443,1.56354,1.57227,1.58057,1.58844,1.59585,1.60279,1.60923,1.61516,1.62057,1.62544,1.62978,1.63358,1.63683,1.63954,1.6417,1.64334,1.64445,1.64504,1.64515,1.64477,1.64394,1.64266,1.64098,1.63891,1.63648,1.63372,1.63065,1.62732,1.62374,1.61994,1.61597,1.61184,1.60758,1.60323,1.59879,1.59431,1.5898,1.58527,1.58076,1.57626,1.5718,1.56739,1.56303,1.55873,1.55449,1.55031,1.5462,1.54214,1.53814,1.53419,1.53028,1.52639,1.52252,1.51864,1.51475,1.51083,1.50686,1.50281,1.49867,1.49441,1.49002,1.48545,1.4807,1.47572,1.47049,1.46498,1.45915,1.45297,1.44641,1.43942,1.43197,1.42401,1.41551,1.40641,1.39668,1.38626,1.37511,1.36317,1.3504,1.33674,1.32214,1.30654,1.28991,1.27217,1.25329,1.23322,1.21191,1.18933,1.16544,1.14022,1.11364,1.08571,1.05642,1.0258,0.993871,0.960694,0.926335,0.890885,0.854456,0.817185,0.77923,0.740772,0.702014,0.663178,0.624499,0.586221,0.548593,0.511857,0.47624,0.441947,0.409152,0.377987,0.348543,0.32086,0.29494},
	{1.01455,1.01542,1.01633,1.01729,1.0183,1.01936,1.02048,1.02166,1.02291,1.02422,1.0256,1.02705,1.02858,1.03019,1.03188,1.03366,1.03554,1.03751,1.03959,1.04177,1.04407,1.04648,1.04901,1.05168,1.05447,1.05741,1.06049,1.06372,1.06711,1.07066,1.07438,1.07827,1.08235,1.08661,1.09107,1.09573,1.10059,1.10567,1.11097,1.11649,1.12224,1.12822,1.13445,1.14092,1.14764,1.15461,1.16183,1.16932,1.17706,1.18507,1.19334,1.20187,1.21066,1.21971,1.22902,1.23858,1.24838,1.25843,1.26871,1.27922,1.28994,1.30087,1.31199,1.3233,1.33476,1.34638,1.35813,1.36999,1.38195,1.39398,1.40606,1.41817,1.43028,1.44237,1.45441,1.46637,1.47824,1.48997,1.50154,1.51291,1.52408,1.53499,1.54562,1.55595,1.56595,1.57558,1.58483,1.59366,1.60206,1.61,1.61746,1.62443,1.63088,1.63681,1.6422,1.64704,1.65133,1.65507,1.65825,1.66088,1.66297,1.66452,1.66554,1.66605,1.66607,1.66562,1.66471,1.66337,1.66163,1.65952,1.65705,1.65427,1.65119,1.64786,1.6443,1.64054,1.63661,1.63254,1.62835,1.62408,1.61974,1.61535,1.61094,1.60653,1.60213,1.59774,1.5934,1.58909,1.58483,1.58063,1.57648,1.57238,1.56833,1.56432,1.56035,1.55641,1.55249,1.54857,1.54464,1.54069,1.53669,1.53263,1.52849,1.52424,1.51987,1.51534,1.51064,1.50573,1.50058,1.49518,1.48947,1.48343,1.47703,1.47023,1.46299,1.45527,1.44702,1.43821,1.42879,1.41872,1.40794,1.39641,1.38407,1.37088,1.35678,1.34172,1.32565,1.30851,1.29026,1.27084,1.25021,1.22833,1.20515,1.18066,1.15482,1.12761,1.09904,1.0691,1.03782,1.00524,0.971409,0.936399,0.900307,0.863249,0.825363,0.786813,0.747783,0.70848,0.669127,0.629964,0.591237,0.553194,0.516078,0.480116,0.445511,0.412433,0.381012,0.351334,0.323438,0.297319},
	{1.01445,1.01531,1.01622,1.01718,1.01818,1.01924,1.02036,1.02153,1.02277,1.02408,1.02545,1.0269,1.02842,1.03003,1.03171,1.03349,1.03536,1.03733,1.0394,1.04158,1.04387,1.04628,1.04881,1.05147,1.05427,1.0572,1.06028,1.06351,1.0669,1.07045,1.07417,1.07807,1.08216,1.08643,1.0909,1.09557,1.10046,1.10556,1.11088,1.11642,1.12221,1.12823,1.13449,1.14101,1.14778,1.15481,1.1621,1.16966,1.17748,1.18557,1.19394,1.20257,1.21148,1.22066,1.2301,1.2398,1.24977,1.25998,1.27045,1.28115,1.29208,1.30323,1.31459,1.32615,1.33788,1.34978,1.36182,1.374,1.38628,1.39865,1.41109,1.42357,1.43607,1.44856,1.46101,1.47341,1.48571,1.4979,1.50994,1.5218,1.53345,1.54486,1.556,1.56685,1.57737,1.58754,1.59732,1.60669,1.61563,1.62411,1.63211,1.63962,1.64661,1.65307,1.65898,1.66435,1.66915,1.67339,1.67706,1.68018,1.68273,1.68473,1.6862,1.68714,1.68757,1.68751,1.68698,1.686,1.6846,1.68282,1.68066,1.67817,1.67537,1.6723,1.66898,1.66545,1.66173,1.65785,1.65385,1.64973,1.64554,1.64129,1.637,1.6327,1.62839,1.62409,1.61981,1.61556,1.61135,1.60718,1.60305,1.59897,1.59492,1.59091,1.58692,1.58295,1.57898,1.57501,1.57102,1.567,1.56292,1.55876,1.55452,1.55015,1.54565,1.54098,1.53612,1.53104,1.52571,1.52011,1.51419,1.50793,1.50128,1.49422,1.4867,1.47868,1.47012,1.46098,1.45121,1.44076,1.42959,1.41765,1.40488,1.39123,1.37665,1.36109,1.3445,1.32682,1.308,1.288,1.26677,1.24427,1.22046,1.19531,1.1688,1.14091,1.11165,1.08102,1.04904,1.01576,0.981235,0.945539,0.908771,0.871053,0.832528,0.793364,0.753749,0.713894,0.674026,0.634386,0.595224,0.556788,0.519321,0.483048,0.44817,0.414853,0.383222,0.353359,0.325296,0.299026},
	{1.01435,1.01521,1.01611,1.01706,1.01806,1.01912,1.02023,1.0214,1.02263,1.02393,1.0253,1.02674,1.02826,1.02986,1.03154,1.03331,1.03518,1.03714,1.03921,1.04138,1.04366,1.04607,1.04859,1.05125,1.05404,1.05697,1.06005,1.06327,1.06666,1.07022,1.07394,1.07785,1.08194,1.08622,1.0907,1.09538,1.10028,1.1054,1.11074,1.11631,1.12212,1.12817,1.13448,1.14104,1.14786,1.15494,1.16229,1.16991,1.17781,1.18599,1.19444,1.20318,1.2122,1.22149,1.23106,1.24091,1.25102,1.26141,1.27205,1.28294,1.29408,1.30544,1.31704,1.32883,1.34083,1.353,1.36534,1.37782,1.39043,1.40314,1.41593,1.42878,1.44166,1.45455,1.46742,1.48025,1.493,1.50564,1.51815,1.53049,1.54264,1.55455,1.56621,1.57758,1.58863,1.59933,1.60966,1.61957,1.62906,1.63809,1.64665,1.6547,1.66223,1.66923,1.67569,1.68158,1.68691,1.69166,1.69584,1.69944,1.70247,1.70494,1.70686,1.70823,1.70908,1.70943,1.70928,1.70868,1.70764,1.70618,1.70434,1.70215,1.69964,1.69683,1.69375,1.69045,1.68694,1.68325,1.67942,1.67546,1.67141,1.66729,1.66312,1.6589,1.65468,1.65044,1.64622,1.64201,1.63782,1.63366,1.62953,1.62543,1.62135,1.61729,1.61324,1.6092,1.60515,1.60108,1.59697,1.59281,1.58859,1.58427,1.57984,1.57527,1.57055,1.56565,1.56053,1.55517,1.54955,1.54362,1.53735,1.53072,1.52368,1.51619,1.50822,1.49972,1.49065,1.48096,1.47061,1.45956,1.44774,1.43512,1.42163,1.40723,1.39187,1.37549,1.35804,1.33946,1.31972,1.29876,1.27653,1.25301,1.22816,1.20194,1.17435,1.14537,1.115,1.08327,1.05021,1.01586,0.980284,0.943579,0.905851,0.867231,0.827874,0.787959,0.747684,0.70727,0.666952,0.626975,0.58759,0.549047,0.511582,0.475412,0.440726,0.407674,0.376363,0.346856,0.319169,0.293275},
	{1.01424,1.0151,1.01599,1.01694,1.01794,1.01899,1.02009,1.02126,1.02249,1.02378,1.02514,1.02658,1.02809,1.02968,1.03135,1.03312,1.03498,1.03693,1.03899,1.04116,1.04344,1.04584,1.04836,1.05101,1.05379,1.05672,1.05979,1.06302,1.0664,1.06996,1.07368,1.07759,1.08168,1.08597,1.09045,1.09515,1.10006,1.10519,1.11055,1.11614,1.12198,1.12806,1.1344,1.141,1.14786,1.155,1.16241,1.17009,1.17806,1.18632,1.19486,1.20369,1.21281,1.22222,1.23192,1.2419,1.25217,1.26271,1.27353,1.28461,1.29594,1.30753,1.31934,1.33139,1.34364,1.35609,1.36872,1.3815,1.39443,1.40748,1.42063,1.43385,1.44712,1.46041,1.4737,1.48696,1.50016,1.51327,1.52625,1.53908,1.55173,1.56416,1.57634,1.58824,1.59984,1.61109,1.62197,1.63244,1.6425,1.65209,1.66121,1.66983,1.67793,1.68549,1.6925,1.69895,1.70482,1.7101,1.71481,1.71892,1.72246,1.72542,1.72781,1.72964,1.73094,1.73171,1.73198,1.73176,1.7311,1.73,1.7285,1.72663,1.72442,1.7219,1.71909,1.71603,1.71275,1.70928,1.70565,1.70188,1.698,1.69402,1.68998,1.68589,1.68177,1.67763,1.67348,1.66934,1.6652,1.66108,1.65698,1.65289,1.64881,1.64475,1.64068,1.6366,1.63251,1.62838,1.62421,1.61997,1.61565,1.61123,1.60668,1.60199,1.59712,1.59206,1.58677,1.58123,1.5754,1.56926,1.56276,1.55588,1.54857,1.5408,1.53252,1.5237,1.5143,1.50425,1.49353,1.48207,1.46983,1.45677,1.44282,1.42794,1.41207,1.39516,1.37716,1.35801,1.33768,1.31611,1.29326,1.26909,1.24357,1.21668,1.1884,1.15872,1.12765,1.09521,1.06143,1.02637,0.990096,0.952697,0.914289,0.875008,0.835013,0.794487,0.753633,0.712674,0.671848,0.631405,0.591595,0.552669,0.514862,0.47839,0.443436,0.410149,0.37863,0.348937,0.321081,0.295031},
	{1.01413,1.01498,1.01587,1.01682,1.01781,1.01885,1.01995,1.02111,1.02233,1.02362,1.02497,1.0264,1.02791,1.02949,1.03116,1.03292,1.03477,1.03672,1.03877,1.04093,1.0432,1.04559,1.0481,1.05075,1.05353,1.05645,1.05951,1.06274,1.06612,1.06967,1.07339,1.0773,1.08139,1.08568,1.09017,1.09488,1.0998,1.10494,1.11031,1.11593,1.12178,1.12789,1.13426,1.1409,1.1478,1.15498,1.16244,1.17019,1.17823,1.18656,1.19518,1.2041,1.21333,1.22285,1.23266,1.24278,1.25319,1.26389,1.27487,1.28613,1.29766,1.30946,1.3215,1.33378,1.34629,1.35901,1.37192,1.38501,1.39826,1.41165,1.42515,1.43874,1.45239,1.46609,1.4798,1.49349,1.50714,1.52071,1.53418,1.5475,1.56065,1.5736,1.58632,1.59876,1.6109,1.62271,1.63415,1.6452,1.65583,1.666,1.6757,1.6849,1.69358,1.70171,1.70929,1.7163,1.72273,1.72857,1.73381,1.73846,1.74251,1.74597,1.74885,1.75117,1.75292,1.75413,1.75483,1.75503,1.75475,1.75403,1.75288,1.75135,1.74945,1.74722,1.7447,1.7419,1.73886,1.73562,1.73219,1.72861,1.7249,1.72108,1.71718,1.71321,1.7092,1.70515,1.70108,1.697,1.69292,1.68884,1.68476,1.68068,1.6766,1.67252,1.66842,1.66431,1.66016,1.65596,1.65171,1.64738,1.64295,1.6384,1.63372,1.62888,1.62385,1.61861,1.61312,1.60737,1.60131,1.59492,1.58816,1.581,1.57339,1.56529,1.55668,1.54749,1.5377,1.52725,1.5161,1.50419,1.49148,1.47791,1.46344,1.44801,1.43156,1.41406,1.39544,1.37565,1.35465,1.33239,1.30884,1.28394,1.25768,1.23003,1.20098,1.17053,1.13867,1.10544,1.07088,1.03504,0.997997,0.959846,0.920708,0.880724,0.840059,0.798901,0.757457,0.715957,0.674641,0.633761,0.59357,0.554317,0.516235,0.479537,0.444402,0.41097,0.379337,0.349553,0.321621,0.295506},
	{1.01402,1.01486,1.01575,1.01669,1.01767,1.01871,1.0198,1.02096,1.02217,1.02345,1.0248,1.02622,1.02772,1.02929,1.03096,1.03271,1.03455,1.03649,1.03853,1.04068,1.04295,1.04533,1.04783,1.05047,1.05324,1.05615,1.05921,1.06243,1.06581,1.06935,1.07307,1.07698,1.08107,1.08536,1.08986,1.09456,1.09949,1.10464,1.11003,1.11566,1.12153,1.12767,1.13406,1.14073,1.14767,1.15489,1.1624,1.1702,1.1783,1.1867,1.1954,1.20441,1.21373,1.22336,1.23329,1.24353,1.25408,1.26493,1.27608,1.28751,1.29924,1.31123,1.3235,1.33602,1.34877,1.36176,1.37496,1.38835,1.40191,1.41563,1.42948,1.44344,1.45749,1.47158,1.48571,1.49984,1.51394,1.52798,1.54193,1.55575,1.56942,1.58289,1.59614,1.60913,1.62183,1.63421,1.64623,1.65786,1.66907,1.67983,1.69012,1.69992,1.70919,1.71791,1.72608,1.73367,1.74068,1.74708,1.75288,1.75808,1.76267,1.76665,1.77004,1.77285,1.77508,1.77676,1.7779,1.77852,1.77866,1.77832,1.77755,1.77637,1.7748,1.77289,1.77066,1.76814,1.76536,1.76235,1.75914,1.75576,1.75224,1.74859,1.74484,1.74101,1.73711,1.73317,1.72919,1.72518,1.72116,1.71712,1.71307,1.70901,1.70494,1.70085,1.69674,1.69258,1.68839,1.68413,1.67981,1.67539,1.67086,1.6662,1.6614,1.65641,1.65123,1.64581,1.64015,1.63419,1.62792,1.6213,1.61429,1.60685,1.59896,1.59056,1.58163,1.5721,1.56195,1.55112,1.53956,1.52723,1.51407,1.50004,1.48508,1.46915,1.45217,1.43412,1.41492,1.39454,1.37293,1.35004,1.32583,1.30027,1.27333,1.24499,1.21523,1.18406,1.15148,1.11753,1.08224,1.04567,1.00792,0.969062,0.929236,0.888584,0.847276,0.805503,0.763478,0.721431,0.679608,0.638261,0.597645,0.55801,0.519587,0.482586,0.447183,0.413514,0.38167,0.351695,0.32359,0.297314},
	{1.0139,1.01474,1.01562,1.01655,1.01753,1.01856,1.01965,1.0208,1.022,1.02328,1.02462,1.02603,1.02752,1.02909,1.03074,1.03248,1.03432,1.03625,1.03828,1.04042,1.04268,1.04505,1.04755,1.05017,1.05294,1.05584,1.05889,1.0621,1.06547,1.06901,1.07273,1.07663,1.08072,1.085,1.0895,1.09421,1.09914,1.1043,1.10969,1.11533,1.12123,1.12738,1.1338,1.14049,1.14746,1.15472,1.16228,1.17013,1.17829,1.18675,1.19553,1.20462,1.21402,1.22375,1.2338,1.24416,1.25484,1.26583,1.27714,1.28875,1.30065,1.31285,1.32533,1.33808,1.35108,1.36433,1.3778,1.39149,1.40537,1.41942,1.43362,1.44794,1.46237,1.47687,1.49142,1.50598,1.52053,1.53504,1.54947,1.5638,1.57798,1.59198,1.60577,1.61932,1.63258,1.64553,1.65813,1.67035,1.68215,1.69352,1.70441,1.71481,1.72469,1.73402,1.74278,1.75097,1.75856,1.76555,1.77193,1.77768,1.78282,1.78735,1.79126,1.79458,1.79731,1.79946,1.80106,1.80213,1.80268,1.80275,1.80236,1.80154,1.80032,1.79873,1.7968,1.79456,1.79205,1.78929,1.78631,1.78313,1.7798,1.77632,1.77272,1.76902,1.76525,1.7614,1.75751,1.75357,1.7496,1.7456,1.74158,1.73753,1.73345,1.72933,1.72518,1.72098,1.71671,1.71238,1.70795,1.70342,1.69877,1.69396,1.689,1.68383,1.67846,1.67283,1.66693,1.66073,1.65419,1.64728,1.63996,1.6322,1.62396,1.61519,1.60585,1.59591,1.58531,1.57401,1.56196,1.54911,1.53541,1.52081,1.50525,1.48868,1.47106,1.45233,1.43243,1.41132,1.38895,1.36529,1.34028,1.31391,1.28613,1.25694,1.22633,1.19429,1.16084,1.12602,1.08987,1.05246,1.01387,0.97421,0.933608,0.892217,0.850213,0.807796,0.765181,0.722607,0.68032,0.638578,0.597633,0.557736,0.519113,0.48197,0.446475,0.412754,0.38089,0.350917,0.322827,0.296571},
	{1.01378,1.01461,1.01549,1.01641,1.01739,1.01841,1.01949,1.02063,1.02183,1.02309,1.02443,1.02583,1.02731,1.02887,1.03052,1.03225,1.03407,1.03599,1.03802,1.04015,1.04239,1.04475,1.04724,1.04986,1.05261,1.0555,1.05855,1.06175,1.06511,1.06864,1.07235,1.07624,1.08033,1.08461,1.0891,1.09381,1.09874,1.1039,1.10931,1.11495,1.12086,1.12703,1.13347,1.14018,1.14719,1.15448,1.16207,1.16997,1.17818,1.1867,1.19555,1.20471,1.21421,1.22403,1.23418,1.24466,1.25547,1.2666,1.27806,1.28983,1.30192,1.31431,1.327,1.33997,1.35322,1.36672,1.38047,1.39445,1.40864,1.42302,1.43756,1.45225,1.46706,1.48196,1.49693,1.51193,1.52694,1.54192,1.55684,1.57167,1.58637,1.6009,1.61524,1.62935,1.64319,1.65672,1.66991,1.68274,1.69515,1.70713,1.71864,1.72966,1.74016,1.75011,1.75949,1.76829,1.7765,1.78409,1.79106,1.7974,1.80311,1.8082,1.81266,1.81651,1.81975,1.8224,1.82449,1.82601,1.82701,1.82751,1.82753,1.82709,1.82624,1.82499,1.82339,1.82146,1.81923,1.81673,1.814,1.81105,1.80792,1.80463,1.80121,1.79767,1.79403,1.79031,1.78652,1.78267,1.77877,1.77483,1.77085,1.76683,1.76276,1.75865,1.75448,1.75026,1.74595,1.74156,1.73707,1.73246,1.72771,1.7228,1.71771,1.71241,1.70688,1.70109,1.69502,1.68862,1.68187,1.67473,1.66717,1.65915,1.65063,1.64157,1.63193,1.62166,1.61071,1.59905,1.58662,1.57336,1.55924,1.5442,1.52818,1.51114,1.49302,1.47377,1.45333,1.43167,1.40874,1.38449,1.35888,1.33188,1.30348,1.27365,1.24238,1.20968,1.17556,1.14007,1.10325,1.06516,1.0259,0.985579,0.944324,0.902293,0.859666,0.816644,0.773447,0.730315,0.687498,0.645254,0.603837,0.563497,0.524462,0.486934,0.451081,0.417028,0.384854,0.35459,0.326226,0.299713},
	{1.01365,1.01448,1.01535,1.01627,1.01723,1.01825,1.01933,1.02046,1.02165,1.02291,1.02423,1.02563,1.0271,1.02865,1.03028,1.032,1.03382,1.03573,1.03774,1.03986,1.04209,1.04444,1.04692,1.04952,1.05226,1.05515,1.05818,1.06137,1.06472,1.06824,1.07194,1.07582,1.0799,1.08418,1.08867,1.09337,1.0983,1.10346,1.10887,1.11452,1.12044,1.12662,1.13307,1.13981,1.14683,1.15416,1.16178,1.16972,1.17798,1.18656,1.19546,1.2047,1.21427,1.22418,1.23443,1.24502,1.25595,1.26722,1.27882,1.29075,1.30301,1.31559,1.32848,1.34167,1.35515,1.36891,1.38293,1.39719,1.41169,1.42639,1.44127,1.45632,1.47151,1.48681,1.50219,1.51763,1.53309,1.54854,1.56394,1.57928,1.5945,1.60957,1.62446,1.63913,1.65354,1.66767,1.68146,1.69489,1.70792,1.72052,1.73266,1.7443,1.75543,1.76601,1.77603,1.78545,1.79427,1.80248,1.81005,1.81699,1.82329,1.82895,1.83397,1.83836,1.84213,1.8453,1.84787,1.84988,1.85133,1.85226,1.85269,1.85265,1.85217,1.85128,1.85001,1.84838,1.84644,1.84421,1.84172,1.839,1.83608,1.83298,1.82972,1.82632,1.82281,1.8192,1.8155,1.81172,1.80788,1.80398,1.80002,1.796,1.79192,1.78778,1.78356,1.77927,1.77489,1.7704,1.76579,1.76105,1.75615,1.75107,1.74579,1.74029,1.73453,1.7285,1.72216,1.71547,1.70841,1.70095,1.69303,1.68464,1.67572,1.66623,1.65613,1.64538,1.63394,1.62174,1.60875,1.59491,1.58017,1.56448,1.54779,1.53004,1.51119,1.49117,1.46996,1.44748,1.42371,1.39859,1.3721,1.3442,1.31488,1.28411,1.2519,1.21825,1.18319,1.14676,1.10901,1.07001,1.02986,0.988684,0.946615,0.903818,0.86048,0.81681,0.773034,0.729397,0.686154,0.643562,0.601878,0.561347,0.522192,0.48461,0.448756,0.414744,0.382642,0.35247,0.324208,0.297798},
	{1.01352,1.01434,1.01521,1.01612,1.01708,1.01809,1.01915,1.02028,1.02146,1.02271,1.02402,1.02541,1.02687,1.02841,1.03003,1.03174,1.03355,1.03545,1.03745,1.03956,1.04178,1.04411,1.04658,1.04917,1.0519,1.05477,1.05779,1.06096,1.0643,1.06781,1.0715,1.07537,1.07944,1.08371,1.08819,1.09289,1.09781,1.10297,1.10838,1.11403,1.11995,1.12614,1.13261,1.13936,1.1464,1.15375,1.16141,1.16939,1.17768,1.18631,1.19527,1.20458,1.21422,1.22421,1.23456,1.24525,1.25629,1.26769,1.27943,1.29152,1.30394,1.3167,1.32979,1.3432,1.35691,1.37091,1.38519,1.39974,1.41453,1.42955,1.44478,1.46019,1.47575,1.49144,1.50724,1.52311,1.53903,1.55495,1.57085,1.58669,1.60244,1.61805,1.6335,1.64874,1.66374,1.67846,1.69287,1.70692,1.72058,1.73381,1.74659,1.75888,1.77066,1.78189,1.79255,1.80262,1.81209,1.82093,1.82913,1.83669,1.8436,1.84985,1.85546,1.86042,1.86474,1.86844,1.87154,1.87404,1.87598,1.87737,1.87824,1.87862,1.87854,1.87802,1.8771,1.87582,1.87419,1.87225,1.87003,1.86756,1.86487,1.86197,1.85891,1.85568,1.85232,1.84885,1.84527,1.84159,1.83783,1.834,1.83008,1.8261,1.82204,1.8179,1.81367,1.80935,1.80493,1.80038,1.7957,1.79087,1.78587,1.78067,1.77526,1.76961,1.7637,1.75749,1.75095,1.74406,1.73677,1.72906,1.72089,1.71222,1.703,1.6932,1.68278,1.67168,1.65987,1.64729,1.63389,1.61962,1.60444,1.58829,1.57112,1.55287,1.5335,1.51295,1.49118,1.46813,1.44376,1.41804,1.39093,1.36239,1.33242,1.30099,1.26811,1.23379,1.19804,1.16092,1.12248,1.0828,1.04197,1.00012,0.957387,0.913942,0.869974,0.825694,0.781331,0.737133,0.693356,0.650261,0.608103,0.567129,0.527561,0.489594,0.453381,0.419035,0.386621,0.356157,0.327619,0.300948},
	{1.01339,1.0142,1.01506,1.01596,1.01692,1.01792,1.01898,1.02009,1.02127,1.02251,1.02381,1.02519,1.02664,1.02817,1.02978,1.03148,1.03327,1.03516,1.03714,1.03924,1.04145,1.04377,1.04622,1.0488,1.05151,1.05437,1.05737,1.06053,1.06386,1.06735,1.07103,1.07489,1.07894,1.0832,1.08767,1.09236,1.09728,1.10243,1.10783,1.11349,1.11941,1.1256,1.13207,1.13884,1.1459,1.15327,1.16095,1.16896,1.17729,1.18596,1.19498,1.20434,1.21405,1.22412,1.23455,1.24533,1.25649,1.268,1.27987,1.29211,1.30469,1.31763,1.33091,1.34452,1.35845,1.37269,1.38723,1.40205,1.41714,1.43247,1.44803,1.46379,1.47972,1.49581,1.51202,1.52832,1.54468,1.56108,1.57746,1.59381,1.61008,1.62624,1.64225,1.65806,1.67365,1.68897,1.70399,1.71866,1.73296,1.74683,1.76026,1.7732,1.78564,1.79752,1.80884,1.81957,1.82969,1.83917,1.84802,1.8562,1.86373,1.8706,1.8768,1.88234,1.88723,1.89148,1.8951,1.89812,1.90054,1.9024,1.90372,1.90453,1.90485,1.90472,1.90416,1.90321,1.9019,1.90025,1.89831,1.89609,1.89362,1.89093,1.88805,1.88499,1.88177,1.87842,1.87494,1.87134,1.86765,1.86385,1.85997,1.85599,1.85192,1.84775,1.84347,1.83908,1.83457,1.82992,1.82512,1.82015,1.81498,1.80961,1.804,1.79813,1.79197,1.7855,1.77868,1.77148,1.76387,1.75581,1.74726,1.73819,1.72855,1.7183,1.7074,1.69581,1.68346,1.67032,1.65634,1.64146,1.62564,1.60882,1.59095,1.57197,1.55184,1.53051,1.50792,1.48404,1.45882,1.43221,1.4042,1.37474,1.34383,1.31146,1.27763,1.24234,1.20565,1.16758,1.1282,1.08761,1.0459,1.0032,0.959658,0.915456,0.870789,0.825875,0.780948,0.736261,0.692074,0.648647,0.606237,0.565086,0.525409,0.487394,0.451184,0.41688,0.384537,0.354158,0.325713,0.299137},
	{1.01325,1.01406,1.01491,1.0158,1.01675,1.01774,1.01879,1.0199,1.02107,1.02229,1.02359,1.02495,1.02639,1.02791,1.02951,1.0312,1.03298,1.03485,1.03682,1.03891,1.0411,1.04341,1.04584,1.0484,1.0511,1.05394,1.05693,1.06008,1.06339,1.06687,1.07052,1.07437,1.07841,1.08266,1.08711,1.09179,1.0967,1.10184,1.10724,1.11289,1.1188,1.125,1.13147,1.13824,1.14531,1.1527,1.1604,1.16843,1.1768,1.18551,1.19457,1.20398,1.21375,1.22389,1.2344,1.24528,1.25653,1.26816,1.28016,1.29253,1.30527,1.31837,1.33183,1.34564,1.35979,1.37427,1.38906,1.40415,1.41953,1.43517,1.45105,1.46716,1.48347,1.49994,1.51656,1.53329,1.55011,1.56697,1.58385,1.60071,1.61751,1.63421,1.65078,1.66718,1.68336,1.6993,1.71494,1.73025,1.74519,1.75972,1.77381,1.78742,1.80053,1.81309,1.82509,1.83649,1.84728,1.85744,1.86694,1.87579,1.88396,1.89146,1.89828,1.90443,1.90991,1.91473,1.91891,1.92246,1.92541,1.92777,1.92956,1.93082,1.93158,1.93186,1.93169,1.9311,1.93013,1.92881,1.92716,1.92521,1.923,1.92054,1.91787,1.915,1.91195,1.90875,1.9054,1.90192,1.89832,1.8946,1.89077,1.88683,1.88278,1.87861,1.87433,1.86991,1.86535,1.86064,1.85577,1.8507,1.84543,1.83993,1.83418,1.82815,1.82182,1.81516,1.80813,1.80071,1.79286,1.78454,1.77572,1.76636,1.75641,1.74583,1.73458,1.72262,1.70989,1.69635,1.68194,1.66662,1.65033,1.63303,1.61466,1.59517,1.5745,1.55261,1.52945,1.50498,1.47914,1.45192,1.42326,1.39316,1.36158,1.32853,1.29401,1.25804,1.22065,1.18188,1.14181,1.10052,1.05812,1.01473,0.970527,0.925671,0.880368,0.834839,0.789321,0.744071,0.699348,0.655415,0.612529,0.570932,0.53084,0.492437,0.455866,0.421225,0.388565,0.357889,0.329164,0.302322},
	{1.0131,1.01391,1.01475,1.01564,1.01658,1.01756,1.0186,1.0197,1.02086,1.02208,1.02336,1.02471,1.02614,1.02765,1.02923,1.03091,1.03267,1.03453,1.03649,1.03856,1.04073,1.04303,1.04545,1.04799,1.05067,1.0535,1.05647,1.05959,1.06289,1.06635,1.06999,1.07382,1.07784,1.08207,1.08651,1.09117,1.09607,1.1012,1.10658,1.11223,1.11814,1.12432,1.1308,1.13757,1.14465,1.15204,1.15976,1.16781,1.17621,1.18495,1.19404,1.2035,1.21333,1.22353,1.23411,1.24508,1.25642,1.26815,1.28027,1.29277,1.30566,1.31892,1.33256,1.34656,1.36091,1.37561,1.39065,1.406,1.42166,1.4376,1.45381,1.47026,1.48692,1.50378,1.5208,1.53796,1.55522,1.57255,1.58992,1.60729,1.62461,1.64186,1.659,1.67598,1.69276,1.70931,1.72558,1.74153,1.75712,1.77231,1.78708,1.80137,1.81515,1.8284,1.84109,1.85318,1.86465,1.87549,1.88567,1.89518,1.90401,1.91215,1.91961,1.92638,1.93247,1.93788,1.94263,1.94673,1.95021,1.95307,1.95536,1.95709,1.95829,1.95898,1.95921,1.959,1.95838,1.95738,1.95603,1.95436,1.95241,1.95018,1.94772,1.94504,1.94216,1.9391,1.93587,1.93249,1.92897,1.92532,1.92153,1.91761,1.91356,1.90938,1.90506,1.90059,1.89596,1.89116,1.88617,1.88098,1.87556,1.86989,1.86395,1.85772,1.85117,1.84426,1.83696,1.82925,1.82109,1.81245,1.80327,1.79353,1.78319,1.77219,1.7605,1.74806,1.73484,1.72078,1.70584,1.68995,1.67307,1.65515,1.63614,1.61598,1.59462,1.57202,1.54813,1.52289,1.49628,1.46825,1.43878,1.40785,1.37543,1.34153,1.30615,1.26931,1.23105,1.19143,1.1505,1.10838,1.06517,1.021,0.976038,0.930468,0.884495,0.838345,0.79226,0.746499,0.701326,0.657004,0.613789,0.57192,0.531608,0.493033,0.456329,0.421586,0.388848,0.35811,0.329333,0.302444},
	{1.01296,1.01375,1.01459,1.01547,1.0164,1.01738,1.01841,1.0195,1.02064,1.02185,1.02312,1.02446,1.02588,1.02737,1.02895,1.03061,1.03236,1.0342,1.03615,1.03819,1.04036,1.04263,1.04503,1.04756,1.05022,1.05303,1.05598,1.05909,1.06236,1.0658,1.06942,1.07323,1.07724,1.08145,1.08587,1.09051,1.09539,1.10051,1.10588,1.11151,1.11741,1.12359,1.13006,1.13683,1.14391,1.1513,1.15903,1.1671,1.17551,1.18427,1.1934,1.2029,1.21278,1.22304,1.23368,1.24472,1.25615,1.26798,1.28021,1.29284,1.30586,1.31927,1.33307,1.34726,1.36181,1.37673,1.392,1.40761,1.42355,1.43978,1.4563,1.47309,1.49011,1.50735,1.52477,1.54235,1.56006,1.57785,1.59571,1.61358,1.63144,1.64924,1.66694,1.68451,1.70189,1.71906,1.73596,1.75256,1.76882,1.78468,1.80013,1.81511,1.82959,1.84354,1.85693,1.86973,1.8819,1.89344,1.90432,1.91451,1.92403,1.93284,1.94095,1.94837,1.95508,1.9611,1.96645,1.97113,1.97516,1.97856,1.98136,1.98358,1.98524,1.98638,1.98703,1.98721,1.98696,1.98631,1.98529,1.98392,1.98224,1.98027,1.97804,1.97556,1.97287,1.96997,1.96689,1.96363,1.96021,1.95664,1.95291,1.94904,1.94501,1.94083,1.9365,1.932,1.92732,1.92245,1.91737,1.91207,1.90653,1.90073,1.89463,1.88822,1.88148,1.87436,1.86683,1.85888,1.85045,1.84152,1.83204,1.82198,1.81129,1.79993,1.78786,1.77503,1.76138,1.74688,1.73147,1.7151,1.69772,1.67928,1.65973,1.639,1.61706,1.59386,1.56934,1.54347,1.5162,1.48751,1.45735,1.42571,1.39258,1.35796,1.32185,1.28427,1.24528,1.20491,1.16326,1.1204,1.07646,1.03159,0.985935,0.939694,0.893073,0.846304,0.799629,0.753313,0.707619,0.662812,0.619147,0.576865,0.536174,0.497251,0.460228,0.425191,0.39218,0.361188,0.332171,0.305056},
	{1.01281,1.01359,1.01442,1.0153,1.01622,1.01719,1.01821,1.01928,1.02042,1.02161,1.02288,1.02421,1.02561,1.02709,1.02865,1.03029,1.03203,1.03386,1.03578,1.03782,1.03996,1.04222,1.0446,1.04711,1.04975,1.05253,1.05546,1.05855,1.0618,1.06522,1.06882,1.07261,1.07659,1.08078,1.08518,1.08981,1.09466,1.09976,1.10511,1.11073,1.11661,1.12278,1.12924,1.136,1.14308,1.15048,1.15821,1.16628,1.17471,1.18349,1.19265,1.20218,1.21209,1.2224,1.2331,1.24421,1.25572,1.26764,1.27997,1.29271,1.30585,1.31941,1.33337,1.34773,1.36248,1.3776,1.3931,1.40896,1.42515,1.44168,1.4585,1.47562,1.49299,1.5106,1.52842,1.54641,1.56455,1.58281,1.60115,1.61953,1.63791,1.65625,1.67452,1.69267,1.71067,1.72845,1.74599,1.76325,1.78017,1.79671,1.81285,1.82853,1.84371,1.85838,1.87248,1.88599,1.89888,1.91113,1.92271,1.93361,1.94381,1.95331,1.9621,1.97017,1.97752,1.98418,1.99013,1.9954,2,2.00395,2.00727,2.00999,2.01213,2.01373,2.01481,2.0154,2.01553,2.01523,2.01454,2.01347,2.01207,2.01036,2.00836,2.00609,2.00358,2.00084,1.9979,1.99476,1.99143,1.98793,1.98425,1.9804,1.97638,1.97219,1.96782,1.96327,1.95851,1.95355,1.94836,1.94292,1.93723,1.93125,1.92495,1.91833,1.91134,1.90396,1.89616,1.8879,1.87915,1.86988,1.86003,1.84958,1.83847,1.82668,1.81414,1.80082,1.78667,1.77163,1.75566,1.73871,1.72072,1.70164,1.68143,1.66002,1.63738,1.61344,1.58817,1.56153,1.53346,1.50396,1.47297,1.44049,1.4065,1.37102,1.33403,1.29559,1.25572,1.21449,1.17198,1.12829,1.08354,1.03788,0.991475,0.944522,0.897235,0.849848,0.802613,0.755793,0.709656,0.664465,0.620478,0.577928,0.537021,0.497927,0.46077,0.42563,0.392536,0.361477,0.332402,0.305235},
	{1.01265,1.01343,1.01425,1.01512,1.01603,1.01699,1.018,1.01906,1.02019,1.02137,1.02262,1.02394,1.02533,1.02679,1.02834,1.02997,1.03169,1.0335,1.03541,1.03742,1.03955,1.04179,1.04415,1.04663,1.04926,1.05202,1.05493,1.05799,1.06122,1.06461,1.06819,1.07195,1.07591,1.08008,1.08445,1.08905,1.09389,1.09896,1.10429,1.10989,1.11575,1.1219,1.12835,1.1351,1.14216,1.14956,1.15729,1.16536,1.17379,1.18259,1.19177,1.20132,1.21127,1.22162,1.23237,1.24354,1.25512,1.26711,1.27954,1.29238,1.30565,1.31934,1.33345,1.34797,1.3629,1.37823,1.39394,1.41004,1.42649,1.44329,1.46042,1.47785,1.49557,1.51354,1.53175,1.55015,1.56873,1.58744,1.60626,1.62514,1.64405,1.66294,1.68178,1.70052,1.71911,1.73753,1.75571,1.77362,1.79122,1.80845,1.82529,1.84167,1.85758,1.87296,1.88779,1.90203,1.91566,1.92863,1.94094,1.95257,1.96348,1.97369,1.98316,1.99192,1.99994,2.00724,2.01382,2.0197,2.0249,2.02942,2.03329,2.03654,2.03918,2.04126,2.04279,2.0438,2.04433,2.04441,2.04407,2.04333,2.04223,2.04079,2.03904,2.037,2.03469,2.03214,2.02935,2.02634,2.02312,2.0197,2.01609,2.01229,2.0083,2.00411,1.99973,1.99513,1.99032,1.98528,1.97999,1.97444,1.96861,1.96247,1.95601,1.94919,1.94199,1.93438,1.92633,1.9178,1.90876,1.89917,1.88899,1.87819,1.86671,1.85452,1.84157,1.82782,1.8132,1.79769,1.78122,1.76375,1.74522,1.72558,1.70478,1.68276,1.65949,1.63491,1.60898,1.58165,1.55289,1.52266,1.49094,1.45772,1.42298,1.38672,1.34897,1.30974,1.2691,1.22709,1.18381,1.13936,1.09386,1.04746,1.00035,0.952712,0.904772,0.856765,0.808947,0.761584,0.714945,0.669296,0.624891,0.581964,0.540718,0.501318,0.463887,0.428497,0.395175,0.363904,0.33463,0.307275},
	{1.01249,1.01326,1.01408,1.01493,1.01583,1.01678,1.01778,1.01884,1.01995,1.02112,1.02236,1.02366,1.02504,1.02649,1.02802,1.02963,1.03133,1.03313,1.03502,1.03702,1.03912,1.04134,1.04368,1.04614,1.04874,1.05148,1.05436,1.0574,1.0606,1.06398,1.06753,1.07126,1.07519,1.07933,1.08368,1.08825,1.09306,1.09811,1.10342,1.10898,1.11483,1.12096,1.12738,1.13412,1.14117,1.14855,1.15627,1.16434,1.17277,1.18158,1.19076,1.20034,1.21031,1.22069,1.23148,1.2427,1.25434,1.26641,1.27891,1.29185,1.30523,1.31904,1.33329,1.34797,1.36307,1.37858,1.39451,1.41083,1.42753,1.4446,1.46201,1.47975,1.4978,1.51613,1.53472,1.55353,1.57253,1.59169,1.61098,1.63036,1.64979,1.66922,1.68862,1.70795,1.72715,1.74619,1.76502,1.78359,1.80186,1.81979,1.83732,1.85443,1.87105,1.88717,1.90274,1.91772,1.93208,1.9458,1.95885,1.9712,1.98284,1.99376,2.00395,2.01339,2.0221,2.03006,2.03729,2.0438,2.0496,2.0547,2.05914,2.06292,2.06608,2.06864,2.07063,2.07209,2.07303,2.07349,2.07351,2.0731,2.0723,2.07114,2.06964,2.06783,2.06572,2.06334,2.0607,2.05781,2.0547,2.05136,2.0478,2.04403,2.04004,2.03584,2.03142,2.02676,2.02187,2.01672,2.01131,2.00561,1.99961,1.99328,1.9866,1.97955,1.97209,1.9642,1.95584,1.94698,1.93759,1.92763,1.91705,1.90583,1.8939,1.88124,1.8678,1.85352,1.83836,1.82227,1.80521,1.78711,1.76793,1.74761,1.72611,1.70337,1.67935,1.654,1.62727,1.59912,1.56952,1.53844,1.50585,1.47173,1.4361,1.39893,1.36027,1.32014,1.27859,1.23568,1.19152,1.14621,1.09988,1.05268,1.00481,0.956457,0.907853,0.859241,0.810879,0.763036,0.715984,0.669986,0.625298,0.582146,0.540729,0.501205,0.463686,0.428238,0.394877,0.36358,0.334287,0.306915},
	{1.01233,1.01309,1.01389,1.01474,1.01563,1.01657,1.01756,1.01861,1.01971,1.02087,1.02209,1.02338,1.02474,1.02617,1.02769,1.02928,1.03097,1.03274,1.03462,1.03659,1.03867,1.04087,1.04319,1.04563,1.0482,1.05092,1.05377,1.05679,1.05996,1.06331,1.06683,1.07053,1.07444,1.07854,1.08286,1.08741,1.09218,1.0972,1.10248,1.10802,1.11383,1.11994,1.12634,1.13305,1.14008,1.14745,1.15515,1.16321,1.17164,1.18044,1.18963,1.19922,1.20921,1.21961,1.23044,1.24169,1.25339,1.26552,1.2781,1.29112,1.3046,1.31852,1.3329,1.34772,1.36298,1.37868,1.3948,1.41133,1.42827,1.4456,1.46329,1.48134,1.49971,1.51839,1.53734,1.55655,1.57597,1.59558,1.61534,1.63521,1.65515,1.67513,1.69509,1.71501,1.73482,1.75449,1.77396,1.7932,1.81216,1.83078,1.84903,1.86686,1.88422,1.90108,1.9174,1.93313,1.94826,1.96273,1.97654,1.98964,2.00204,2.01369,2.02461,2.03477,2.04418,2.05283,2.06073,2.06789,2.07432,2.08004,2.08506,2.0894,2.0931,2.09618,2.09866,2.10057,2.10195,2.10283,2.10322,2.10317,2.1027,2.10184,2.10061,2.09905,2.09716,2.09498,2.09251,2.08977,2.08678,2.08354,2.08006,2.07634,2.07239,2.06819,2.06375,2.05907,2.05412,2.04889,2.04338,2.03756,2.03142,2.02493,2.01808,2.01082,2.00314,1.99501,1.98639,1.97725,1.96756,1.95727,1.94635,1.93476,1.92246,1.90939,1.89552,1.88079,1.86516,1.84858,1.831,1.81237,1.79264,1.77175,1.74965,1.7263,1.70164,1.67563,1.64822,1.61938,1.58907,1.55726,1.52393,1.48906,1.45266,1.41473,1.37528,1.33437,1.29203,1.24835,1.20341,1.15733,1.11024,1.06232,1.01373,0.964697,0.915441,0.86621,0.817268,0.768884,0.721332,0.674878,0.629774,0.586246,0.544491,0.504661,0.466865,0.431164,0.39757,0.366056,0.336559,0.308995},
	{1.01216,1.01291,1.01371,1.01455,1.01543,1.01636,1.01734,1.01837,1.01946,1.0206,1.02181,1.02309,1.02443,1.02585,1.02735,1.02892,1.03059,1.03235,1.0342,1.03615,1.03821,1.04039,1.04268,1.04509,1.04764,1.05033,1.05316,1.05614,1.05929,1.0626,1.06609,1.06977,1.07364,1.07771,1.082,1.08651,1.09125,1.09624,1.10148,1.10699,1.11277,1.11885,1.12522,1.1319,1.13891,1.14625,1.15393,1.16198,1.17039,1.17919,1.18837,1.19796,1.20796,1.21838,1.22923,1.24052,1.25225,1.26443,1.27707,1.29018,1.30374,1.31777,1.33226,1.34721,1.36262,1.37849,1.39479,1.41153,1.4287,1.44627,1.46423,1.48257,1.50126,1.52027,1.53959,1.55919,1.57902,1.59907,1.61929,1.63964,1.6601,1.68061,1.70114,1.72163,1.74205,1.76235,1.78247,1.80238,1.82202,1.84135,1.86031,1.87887,1.89698,1.9146,1.93167,1.94818,1.96407,1.97932,1.99389,2.00777,2.02093,2.03334,2.045,2.0559,2.06603,2.07539,2.08398,2.09181,2.09889,2.10524,2.11086,2.1158,2.12005,2.12366,2.12665,2.12904,2.13087,2.13217,2.13297,2.13328,2.13316,2.13261,2.13167,2.13037,2.12872,2.12674,2.12445,2.12187,2.11901,2.11588,2.11248,2.10882,2.1049,2.10072,2.09627,2.09155,2.08654,2.08124,2.07563,2.0697,2.06341,2.05677,2.04973,2.04227,2.03437,2.02599,2.01711,2.00768,1.99768,1.98707,1.97581,1.96385,1.95115,1.93767,1.92337,1.90819,1.89209,1.87502,1.85692,1.83775,1.81746,1.79599,1.77329,1.74932,1.72402,1.69735,1.66926,1.63973,1.6087,1.57616,1.54209,1.50647,1.4693,1.43058,1.39036,1.34866,1.30553,1.26106,1.21535,1.1685,1.12066,1.072,1.0227,0.972985,0.923075,0.873225,0.823701,0.774776,0.726725,0.679813,0.634292,0.590388,0.548292,0.508155,0.47008,0.434124,0.400295,0.368561,0.338858,0.311099},
	{1.01199,1.01273,1.01352,1.01435,1.01522,1.01614,1.0171,1.01812,1.0192,1.02033,1.02152,1.02278,1.02411,1.02552,1.02699,1.02855,1.0302,1.03193,1.03377,1.0357,1.03773,1.03988,1.04215,1.04454,1.04706,1.04972,1.05252,1.05548,1.05859,1.06187,1.06533,1.06897,1.0728,1.07684,1.08109,1.08557,1.09027,1.09522,1.10043,1.1059,1.11165,1.11768,1.12402,1.13067,1.13764,1.14496,1.15261,1.16064,1.16903,1.17781,1.18698,1.19656,1.20656,1.21699,1.22785,1.23916,1.25092,1.26315,1.27584,1.28901,1.30265,1.31677,1.33136,1.34644,1.36199,1.378,1.39448,1.41142,1.42879,1.4466,1.46482,1.48343,1.50242,1.52177,1.54144,1.5614,1.58164,1.60211,1.62279,1.64362,1.66458,1.68562,1.7067,1.72778,1.7488,1.76972,1.79049,1.81106,1.83139,1.85143,1.87111,1.89041,1.90927,1.92764,1.94549,1.96277,1.97944,1.99547,2.01083,2.02548,2.03942,2.0526,2.06502,2.07667,2.08754,2.09762,2.10692,2.11543,2.12318,2.13018,2.13643,2.14196,2.14679,2.15094,2.15445,2.15734,2.15964,2.16137,2.16257,2.16327,2.1635,2.16328,2.16263,2.16159,2.16018,2.15841,2.15631,2.15389,2.15115,2.14813,2.14481,2.1412,2.13731,2.13313,2.12866,2.1239,2.11883,2.11344,2.10771,2.10164,2.0952,2.08837,2.08113,2.07345,2.0653,2.05665,2.04747,2.03774,2.0274,2.01643,2.00478,1.99241,1.97929,1.96536,1.95058,1.93491,1.91829,1.90067,1.88201,1.86225,1.84134,1.81923,1.79588,1.77122,1.74522,1.71782,1.689,1.65869,1.62689,1.59355,1.55866,1.52221,1.48421,1.44465,1.40357,1.36102,1.31705,1.27174,1.22519,1.17753,1.1289,1.07948,1.02945,0.97903,0.928467,0.87801,0.82793,0.778501,0.729999,0.68269,0.636825,0.592624,0.550274,0.509921,0.471662,0.435546,0.401575,0.369713,0.339891,0.31202},
	{1.01181,1.01255,1.01332,1.01414,1.015,1.01591,1.01686,1.01787,1.01893,1.02005,1.02123,1.02247,1.02379,1.02517,1.02663,1.02817,1.0298,1.03151,1.03332,1.03523,1.03724,1.03936,1.0416,1.04396,1.04646,1.04908,1.05186,1.05478,1.05786,1.0611,1.06452,1.06813,1.07193,1.07592,1.08014,1.08457,1.08924,1.09415,1.09931,1.10474,1.11045,1.11644,1.12274,1.12935,1.13629,1.14356,1.15119,1.15918,1.16754,1.1763,1.18545,1.19502,1.20501,1.21543,1.2263,1.23762,1.24941,1.26166,1.2744,1.28762,1.30132,1.31552,1.33021,1.34539,1.36106,1.37722,1.39386,1.41098,1.42855,1.44658,1.46504,1.48393,1.50321,1.52286,1.54287,1.56321,1.58384,1.60473,1.62585,1.64715,1.66861,1.69018,1.71181,1.73346,1.75508,1.77663,1.79805,1.81929,1.84031,1.86106,1.88147,1.90151,1.92113,1.94027,1.9589,1.97697,1.99444,2.01127,2.02743,2.04288,2.05761,2.07159,2.08479,2.09722,2.10884,2.11967,2.1297,2.13893,2.14738,2.15504,2.16194,2.1681,2.17353,2.17826,2.18232,2.18573,2.18852,2.19072,2.19236,2.19346,2.19407,2.19419,2.19387,2.19312,2.19198,2.19045,2.18855,2.18631,2.18374,2.18084,2.17763,2.1741,2.17027,2.16612,2.16166,2.15688,2.15178,2.14633,2.14053,2.13435,2.12779,2.12082,2.11341,2.10555,2.09719,2.08832,2.07891,2.06891,2.05829,2.04701,2.03505,2.02234,2.00886,1.99455,1.97938,1.96329,1.94623,1.92816,1.90903,1.88877,1.86736,1.84472,1.82082,1.79559,1.76901,1.74101,1.71157,1.68063,1.64817,1.61417,1.5786,1.54146,1.50275,1.46248,1.42069,1.3774,1.3327,1.28666,1.23938,1.191,1.14165,1.09151,1.04079,0.989695,0.938475,0.887385,0.836697,0.786688,0.737636,0.689807,0.643452,0.598791,0.556011,0.515254,0.476615,0.440142,0.405835,0.373654,0.343529,0.315373},
	{1.01163,1.01236,1.01312,1.01393,1.01478,1.01567,1.01662,1.01761,1.01866,1.01976,1.02093,1.02215,1.02345,1.02481,1.02626,1.02778,1.02938,1.03107,1.03286,1.03474,1.03673,1.03882,1.04104,1.04337,1.04583,1.04843,1.05116,1.05405,1.0571,1.0603,1.06369,1.06725,1.07101,1.07497,1.07914,1.08353,1.08815,1.09301,1.09813,1.10352,1.10918,1.11513,1.12138,1.12795,1.13484,1.14207,1.14966,1.15761,1.16594,1.17466,1.18379,1.19333,1.2033,1.21371,1.22457,1.2359,1.24769,1.25997,1.27273,1.28599,1.29975,1.31401,1.32878,1.34405,1.35984,1.37612,1.39291,1.41019,1.42795,1.44619,1.46488,1.48401,1.50357,1.52353,1.54387,1.56456,1.58556,1.60686,1.62841,1.65018,1.67212,1.69421,1.71638,1.7386,1.76081,1.78298,1.80504,1.82695,1.84866,1.87011,1.89126,1.91205,1.93242,1.95234,1.97176,1.99062,2.00889,2.02653,2.0435,2.05977,2.0753,2.09008,2.10409,2.1173,2.1297,2.14129,2.15207,2.16203,2.17118,2.17954,2.1871,2.1939,2.19995,2.20527,2.2099,2.21384,2.21714,2.21981,2.2219,2.22342,2.22441,2.2249,2.22491,2.22446,2.22358,2.22229,2.22061,2.21856,2.21614,2.21337,2.21026,2.20681,2.20303,2.19891,2.19445,2.18965,2.1845,2.17898,2.17308,2.1668,2.1601,2.15296,2.14537,2.1373,2.12873,2.11961,2.10992,2.09963,2.0887,2.0771,2.06478,2.0517,2.03782,2.0231,2.00749,1.99094,1.9734,1.95483,1.93517,1.91438,1.8924,1.86918,1.84467,1.81883,1.7916,1.76294,1.73281,1.70117,1.668,1.63326,1.59695,1.55905,1.51956,1.47851,1.43593,1.39186,1.34636,1.29953,1.25147,1.20231,1.15221,1.10134,1.0499,0.998117,0.946241,0.89453,0.843259,0.792708,0.743155,0.694867,0.648094,0.603053,0.559929,0.518859,0.479935,0.443201,0.40865,0.376241,0.345902,0.317543},
	{1.01145,1.01216,1.01292,1.01372,1.01455,1.01544,1.01637,1.01735,1.01838,1.01947,1.02062,1.02183,1.0231,1.02445,1.02587,1.02737,1.02895,1.03062,1.03238,1.03424,1.0362,1.03827,1.04045,1.04275,1.04518,1.04775,1.05045,1.0533,1.05631,1.05948,1.06282,1.06634,1.07006,1.07397,1.07809,1.08244,1.08701,1.09183,1.0969,1.10223,1.10784,1.11374,1.11994,1.12646,1.1333,1.14048,1.14802,1.15593,1.16421,1.1729,1.18198,1.19149,1.20144,1.21182,1.22267,1.23398,1.24578,1.25806,1.27084,1.28412,1.29792,1.31223,1.32707,1.34242,1.3583,1.3747,1.39162,1.40905,1.42698,1.4454,1.46431,1.48368,1.5035,1.52374,1.54439,1.56542,1.58679,1.60848,1.63045,1.65266,1.67508,1.69767,1.72037,1.74315,1.76595,1.78872,1.81142,1.834,1.85639,1.87854,1.90041,1.92194,1.94308,1.96378,1.98398,2.00364,2.02272,2.04117,2.05896,2.07604,2.0924,2.10799,2.1228,2.13681,2.15,2.16237,2.17391,2.18461,2.19449,2.20354,2.21179,2.21924,2.22592,2.23185,2.23704,2.24153,2.24535,2.24851,2.25105,2.253,2.25439,2.25524,2.25558,2.25543,2.25482,2.25378,2.25231,2.25044,2.24817,2.24553,2.24251,2.23913,2.23539,2.23128,2.22681,2.22197,2.21675,2.21115,2.20513,2.19871,2.19184,2.18452,2.17672,2.16841,2.15957,2.15017,2.14017,2.12955,2.11827,2.10628,2.09356,2.08006,2.06573,2.05053,2.03442,2.01735,1.99927,1.98013,1.95987,1.93846,1.91584,1.89195,1.86675,1.84019,1.81223,1.78281,1.7519,1.71947,1.68547,1.6499,1.61274,1.57397,1.53362,1.49169,1.44822,1.40327,1.35689,1.30919,1.26028,1.21028,1.15937,1.10772,1.05554,1.00307,0.950546,0.898241,0.846433,0.795403,0.745431,0.696782,0.649704,0.604409,0.561076,0.519837,0.480774,0.443923,0.409272,0.376775,0.346354,0.317918},
	{1.01126,1.01197,1.01271,1.0135,1.01432,1.01519,1.01611,1.01707,1.01809,1.01916,1.0203,1.02149,1.02275,1.02407,1.02547,1.02695,1.02851,1.03016,1.03189,1.03372,1.03565,1.03769,1.03985,1.04212,1.04451,1.04704,1.0497,1.05252,1.05548,1.05861,1.06191,1.06539,1.06906,1.07292,1.077,1.08129,1.08581,1.09058,1.09559,1.10087,1.10642,1.11227,1.11841,1.12487,1.13166,1.13879,1.14627,1.15412,1.16236,1.17099,1.18003,1.1895,1.19941,1.20976,1.22058,1.23187,1.24365,1.25593,1.26871,1.28201,1.29583,1.31018,1.32507,1.34049,1.35645,1.37294,1.38998,1.40754,1.42563,1.44423,1.46333,1.48293,1.50299,1.52351,1.54445,1.5658,1.58752,1.60959,1.63197,1.65462,1.67751,1.70059,1.72382,1.74714,1.77053,1.79391,1.81725,1.84048,1.86356,1.88642,1.90902,1.9313,1.95321,1.97469,1.9957,2.01617,2.03607,2.05535,2.07398,2.0919,2.10909,2.12552,2.14117,2.156,2.17001,2.18318,2.1955,2.20698,2.21761,2.2274,2.23636,2.2445,2.25184,2.2584,2.26421,2.26928,2.27365,2.27733,2.28037,2.28278,2.28459,2.28584,2.28655,2.28675,2.28645,2.28568,2.28446,2.28281,2.28074,2.27826,2.27538,2.27211,2.26845,2.26441,2.25997,2.25514,2.24991,2.24426,2.23819,2.23168,2.22472,2.21727,2.20932,2.20085,2.19183,2.18222,2.172,2.16113,2.14959,2.13732,2.1243,2.11048,2.09581,2.08026,2.06378,2.04632,2.02783,2.00826,1.98756,1.96569,1.94259,1.91821,1.89249,1.8654,1.83689,1.8069,1.77541,1.74238,1.70778,1.67158,1.63377,1.59435,1.55332,1.51071,1.46656,1.4209,1.37383,1.32542,1.27579,1.22509,1.17347,1.12112,1.06825,1.01509,0.961903,0.90895,0.856513,0.804876,0.75432,0.705112,0.6575,0.611697,0.56788,0.526181,0.486681,0.449415,0.41437,0.381498,0.350721,0.321949},
	{1.01107,1.01176,1.0125,1.01327,1.01409,1.01494,1.01585,1.0168,1.0178,1.01886,1.01997,1.02115,1.02238,1.02369,1.02507,1.02653,1.02806,1.02968,1.03139,1.03319,1.0351,1.03711,1.03922,1.04146,1.04382,1.04631,1.04894,1.05171,1.05464,1.05772,1.06098,1.06441,1.06803,1.07184,1.07586,1.0801,1.08457,1.08928,1.09423,1.09945,1.10495,1.11073,1.11681,1.12321,1.12993,1.137,1.14442,1.15221,1.16039,1.16896,1.17795,1.18736,1.19722,1.20753,1.21831,1.22957,1.24132,1.25358,1.26635,1.27965,1.29348,1.30785,1.32277,1.33824,1.35426,1.37084,1.38797,1.40565,1.42387,1.44263,1.46191,1.48171,1.502,1.52277,1.54399,1.56564,1.5877,1.61012,1.63289,1.65596,1.6793,1.70285,1.72658,1.75045,1.7744,1.79837,1.82233,1.84621,1.86996,1.89352,1.91684,1.93986,1.96253,1.98479,2.00659,2.02787,2.04859,2.0687,2.08816,2.10692,2.12495,2.14222,2.15869,2.17435,2.18918,2.20315,2.21627,2.22852,2.23991,2.25043,2.26011,2.26894,2.27695,2.28415,2.29056,2.29622,2.30113,2.30534,2.30886,2.31173,2.31397,2.31561,2.31668,2.3172,2.3172,2.3167,2.31571,2.31426,2.31235,2.31001,2.30724,2.30405,2.30044,2.29642,2.29198,2.28711,2.28182,2.27608,2.2699,2.26325,2.25611,2.24847,2.24031,2.23159,2.2223,2.2124,2.20186,2.19065,2.17874,2.16609,2.15265,2.13838,2.12325,2.10721,2.09022,2.07222,2.05316,2.033,2.01169,1.98918,1.96541,1.94034,1.91391,1.88608,1.8568,1.82603,1.79374,1.75988,1.72442,1.68736,1.64867,1.60835,1.56642,1.5229,1.47783,1.43126,1.38328,1.33398,1.28348,1.23192,1.17948,1.12635,1.07273,1.01888,0.965058,0.911526,0.858573,0.806486,0.755544,0.706013,0.658139,0.612127,0.568149,0.526327,0.486735,0.449399,0.414298,0.38138,0.350562,0.321751}
};

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo....

//LS X coefficient for A=atomic weight * 1.05
const G4double G4AtimaFluctuations::ls_X_coefficients_ahi[110][200]=
{
	{0.999182,0.999124,0.999062,0.998995,0.998925,0.998851,0.998772,0.998688,0.998598,0.998504,0.998404,0.998297,0.998184,0.998064,0.997937,0.997802,0.997659,0.997507,0.997346,0.997175,0.996994,0.996802,0.996598,0.996382,0.996153,0.99591,0.995653,0.995381,0.995092,0.994785,0.994461,0.994117,0.993753,0.993367,0.992958,0.992526,0.992067,0.991582,0.991068,0.990524,0.989948,0.989338,0.988693,0.988011,0.987289,0.986525,0.985717,0.984863,0.98396,0.983005,0.981996,0.98093,0.979804,0.978614,0.977358,0.976032,0.974632,0.973155,0.971597,0.969955,0.968223,0.966397,0.964474,0.962449,0.960317,0.958074,0.955714,0.953233,0.950626,0.947887,0.945012,0.941995,0.938832,0.935516,0.932044,0.928409,0.924608,0.920635,0.916485,0.912155,0.90764,0.902937,0.898042,0.892952,0.887666,0.88218,0.876495,0.870608,0.864522,0.858236,0.851752,0.845074,0.838206,0.831151,0.823917,0.81651,0.808938,0.801211,0.79334,0.785335,0.77721,0.768979,0.760656,0.752258,0.7438,0.735302,0.72678,0.718255,0.709743,0.701267,0.692844,0.684494,0.676236,0.668089,0.660071,0.6522,0.644492,0.636962,0.629625,0.622494,0.61558,0.608893,0.602443,0.596236,0.590278,0.584572,0.579122,0.573929,0.568991,0.564308,0.559876,0.555692,0.55175,0.548044,0.544568,0.541314,0.538273,0.535438,0.5328,0.53035,0.528077,0.525973,0.524029,0.522235,0.520583,0.519062,0.517665,0.516384,0.515209,0.514135,0.513152,0.512254,0.511434,0.510686,0.510005,0.509383,0.508817,0.508302,0.507831,0.507403,0.507011,0.506653,0.506325,0.506024,0.505746,0.505489,0.50525,0.505026,0.504815,0.504615,0.504423,0.504237,0.504055,0.503874,0.503693,0.503509,0.50332,0.503124,0.502918,0.502699,0.502465,0.502213,0.501939,0.501641,0.501313,0.500953,0.500554,0.500113,0.499624,0.49908,0.498474,0.4978,0.497047,0.496207,0.495269,0.494222,0.493052,0.491747,0.490288,0.48866},
	{0.999599,0.999546,0.999489,0.999428,0.999363,0.999294,0.999221,0.999143,0.99906,0.998971,0.998877,0.998778,0.998671,0.998558,0.998438,0.998311,0.998175,0.998031,0.997878,0.997716,0.997543,0.99736,0.997165,0.996959,0.99674,0.996507,0.99626,0.995998,0.99572,0.995425,0.995112,0.99478,0.994428,0.994054,0.993659,0.993239,0.992795,0.992323,0.991824,0.991295,0.990734,0.990141,0.989512,0.988846,0.988141,0.987395,0.986605,0.98577,0.984886,0.983951,0.982962,0.981917,0.980812,0.979644,0.978411,0.977108,0.975732,0.974279,0.972746,0.971129,0.969423,0.967625,0.965729,0.963732,0.961629,0.959416,0.957086,0.954636,0.95206,0.949354,0.946512,0.943529,0.9404,0.93712,0.933684,0.930086,0.926322,0.922387,0.918276,0.913985,0.90951,0.904848,0.899994,0.894946,0.889702,0.884259,0.878617,0.872775,0.866732,0.860491,0.854053,0.847421,0.840598,0.833589,0.826401,0.81904,0.811514,0.803834,0.796008,0.78805,0.77997,0.771785,0.763507,0.755153,0.74674,0.738286,0.729807,0.721324,0.712855,0.704419,0.696036,0.687725,0.679505,0.671395,0.663414,0.655577,0.647903,0.640406,0.6331,0.625999,0.619114,0.612455,0.606031,0.599848,0.593914,0.588231,0.582802,0.577628,0.572709,0.568043,0.563628,0.559458,0.55553,0.551837,0.548372,0.545128,0.542097,0.539271,0.536639,0.534195,0.531927,0.529827,0.527886,0.526094,0.524442,0.522921,0.521522,0.520238,0.519059,0.517979,0.516989,0.516083,0.515253,0.514493,0.513798,0.513161,0.512576,0.51204,0.511546,0.51109,0.510669,0.510277,0.50991,0.509566,0.50924,0.508929,0.508629,0.508337,0.50805,0.507765,0.507477,0.507184,0.506881,0.506566,0.506235,0.505883,0.505505,0.505098,0.504656,0.504173,0.503643,0.50306,0.502417,0.501704,0.500915,0.500037,0.499061,0.497975,0.496766,0.495418,0.493916,0.492242,0.490376,0.488297,0.485981,0.483402,0.480533,0.477343,0.4738,0.469868},
	{1.00017,1.00012,1.00007,1.00002,0.999959,0.999897,0.99983,0.999759,0.999682,0.999601,0.999514,0.999421,0.999322,0.999217,0.999104,0.998984,0.998857,0.998721,0.998576,0.998422,0.998258,0.998083,0.997898,0.997701,0.997491,0.997268,0.997031,0.996779,0.996512,0.996228,0.995926,0.995606,0.995266,0.994905,0.994522,0.994115,0.993684,0.993226,0.992741,0.992226,0.991681,0.991103,0.99049,0.98984,0.989152,0.988423,0.987652,0.986834,0.985969,0.985054,0.984085,0.98306,0.981976,0.98083,0.979618,0.978338,0.976985,0.975557,0.974048,0.972456,0.970776,0.969004,0.967136,0.965167,0.963092,0.960907,0.958608,0.956188,0.953644,0.95097,0.94816,0.945211,0.942116,0.938871,0.935469,0.931908,0.92818,0.924283,0.92021,0.915958,0.911523,0.9069,0.902087,0.897081,0.891878,0.886478,0.880878,0.875079,0.869081,0.862884,0.85649,0.849902,0.843124,0.836161,0.829018,0.821702,0.814222,0.806587,0.798807,0.790893,0.782859,0.774718,0.766485,0.758175,0.749806,0.741394,0.732958,0.724516,0.716087,0.707692,0.699348,0.691075,0.682893,0.67482,0.666873,0.659071,0.65143,0.643965,0.63669,0.629618,0.622761,0.616129,0.609731,0.603573,0.597662,0.592001,0.586593,0.581439,0.576538,0.571889,0.56749,0.563335,0.559421,0.55574,0.552287,0.549053,0.546032,0.543214,0.54059,0.538152,0.53589,0.533794,0.531857,0.530067,0.528417,0.526896,0.525498,0.524212,0.523031,0.521947,0.520952,0.520039,0.519202,0.518434,0.517728,0.517079,0.51648,0.515928,0.515415,0.514939,0.514494,0.514075,0.513679,0.513301,0.512937,0.512583,0.512236,0.511891,0.511544,0.511191,0.510828,0.510451,0.510054,0.509634,0.509185,0.508701,0.508176,0.507604,0.506978,0.506289,0.50553,0.50469,0.50376,0.502728,0.501582,0.500307,0.498889,0.49731,0.495552,0.493595,0.491416,0.488992,0.486294,0.483295,0.479964,0.476266,0.472165,0.467623,0.4626,0.457054},
	{1.00086,1.00082,1.00078,1.00073,1.00069,1.00063,1.00058,1.00051,1.00045,1.00038,1.0003,1.00021,1.00012,1.00003,0.999924,0.999813,0.999695,0.999568,0.999433,0.999288,0.999134,0.998969,0.998793,0.998606,0.998406,0.998193,0.997967,0.997726,0.997469,0.997196,0.996906,0.996598,0.996269,0.995921,0.99555,0.995157,0.994739,0.994295,0.993823,0.993323,0.992793,0.99223,0.991632,0.990999,0.990327,0.989615,0.988861,0.988062,0.987215,0.986318,0.985369,0.984364,0.983301,0.982176,0.980986,0.979728,0.978398,0.976993,0.975508,0.973941,0.972286,0.97054,0.968699,0.966757,0.96471,0.962554,0.960284,0.957895,0.955381,0.952738,0.949961,0.947044,0.943983,0.940771,0.937405,0.933879,0.930188,0.926327,0.922292,0.918078,0.913682,0.909099,0.904325,0.899359,0.894198,0.888839,0.883282,0.877525,0.87157,0.865416,0.859066,0.852522,0.845788,0.838869,0.831771,0.8245,0.817065,0.809474,0.801738,0.793869,0.785879,0.777782,0.769592,0.761326,0.752999,0.744629,0.736235,0.727834,0.719445,0.711089,0.702784,0.694549,0.686403,0.678365,0.670453,0.662685,0.655076,0.647642,0.640397,0.633354,0.626525,0.61992,0.613547,0.607413,0.601525,0.595886,0.590498,0.585363,0.580481,0.575849,0.571465,0.567325,0.563425,0.559757,0.556315,0.553093,0.550081,0.547272,0.544656,0.542225,0.539969,0.537879,0.535946,0.53416,0.532512,0.530994,0.529596,0.528311,0.527129,0.526044,0.525047,0.524131,0.52329,0.522516,0.521804,0.521147,0.520539,0.519976,0.519452,0.518962,0.518501,0.518064,0.517648,0.517247,0.516857,0.516475,0.516095,0.515713,0.515325,0.514927,0.514513,0.514078,0.513618,0.513126,0.512597,0.512023,0.511399,0.510715,0.509964,0.509136,0.508222,0.507209,0.506086,0.504839,0.503454,0.501913,0.500199,0.498292,0.49617,0.493809,0.491184,0.488265,0.485024,0.481427,0.477438,0.473019,0.468131,0.462732,0.456779,0.450227},
	{1.00162,1.0016,1.00157,1.00154,1.00151,1.00147,1.00143,1.00138,1.00132,1.00127,1.0012,1.00113,1.00105,1.00097,1.00088,1.00078,1.00067,1.00056,1.00043,1.0003,1.00016,1,0.99984,0.999664,0.999476,0.999275,0.99906,0.998831,0.998587,0.998326,0.998048,0.997752,0.997436,0.997101,0.996743,0.996363,0.995959,0.995529,0.995072,0.994586,0.99407,0.993523,0.992941,0.992324,0.991668,0.990973,0.990236,0.989455,0.988626,0.987748,0.986818,0.985833,0.98479,0.983685,0.982517,0.981281,0.979973,0.978591,0.977131,0.975587,0.973958,0.972237,0.970422,0.968507,0.966488,0.96436,0.962118,0.959758,0.957275,0.954663,0.951917,0.949033,0.946004,0.942827,0.939495,0.936003,0.932348,0.928523,0.924525,0.920349,0.91599,0.911446,0.906712,0.901786,0.896665,0.891347,0.885831,0.880116,0.874203,0.868092,0.861784,0.855284,0.848593,0.841718,0.834663,0.827436,0.820045,0.812498,0.804806,0.79698,0.789033,0.780979,0.772832,0.764608,0.756323,0.747994,0.73964,0.731279,0.72293,0.714612,0.706344,0.698146,0.690036,0.682033,0.674155,0.66642,0.658842,0.651439,0.644223,0.637209,0.630407,0.623827,0.617479,0.611369,0.605503,0.599885,0.594517,0.589401,0.584536,0.579921,0.575553,0.571428,0.567541,0.563885,0.560455,0.557243,0.554241,0.551441,0.548833,0.546409,0.544159,0.542074,0.540146,0.538364,0.536719,0.535203,0.533807,0.532522,0.531341,0.530255,0.529256,0.528337,0.527493,0.526714,0.525996,0.525333,0.524718,0.524145,0.52361,0.523108,0.522633,0.522181,0.521747,0.521326,0.520914,0.520506,0.520098,0.519685,0.519261,0.518823,0.518364,0.517879,0.517363,0.516808,0.516209,0.515557,0.514845,0.514064,0.513204,0.512255,0.511206,0.510043,0.508752,0.507319,0.505727,0.503956,0.501987,0.499797,0.497363,0.494656,0.49165,0.488312,0.484609,0.480505,0.475962,0.47094,0.465395,0.459285,0.452567,0.445192},
	{1.00243,1.00243,1.00242,1.00241,1.00239,1.00237,1.00235,1.00232,1.00228,1.00224,1.00219,1.00214,1.00208,1.00201,1.00194,1.00185,1.00176,1.00166,1.00155,1.00144,1.00131,1.00117,1.00102,1.00086,1.00069,1.0005,1.0003,1.00008,0.999853,0.999606,0.999342,0.99906,0.998759,0.998438,0.998095,0.997729,0.99734,0.996925,0.996483,0.996012,0.995512,0.99498,0.994415,0.993814,0.993175,0.992498,0.991778,0.991014,0.990204,0.989345,0.988434,0.987468,0.986445,0.985362,0.984214,0.983,0.981715,0.980355,0.978918,0.977399,0.975794,0.974098,0.972309,0.97042,0.968428,0.966328,0.964115,0.961784,0.95933,0.956748,0.954033,0.951181,0.948184,0.94504,0.941741,0.938284,0.934664,0.930875,0.926913,0.922774,0.918453,0.913946,0.909251,0.904364,0.899282,0.894005,0.888529,0.882856,0.876984,0.870914,0.864649,0.858191,0.851543,0.84471,0.837698,0.830514,0.823165,0.815661,0.808012,0.800229,0.792325,0.784312,0.776207,0.768024,0.75978,0.751492,0.743177,0.734855,0.726545,0.718264,0.710033,0.701871,0.693796,0.685827,0.677982,0.670278,0.662732,0.655358,0.648171,0.641184,0.634409,0.627855,0.62153,0.615443,0.609599,0.604002,0.598654,0.593556,0.588709,0.58411,0.579757,0.575646,0.571772,0.56813,0.564711,0.561509,0.558517,0.555725,0.553125,0.550708,0.548465,0.546386,0.544462,0.542685,0.541044,0.539531,0.538137,0.536853,0.535673,0.534587,0.533587,0.532668,0.531821,0.53104,0.530318,0.52965,0.52903,0.528451,0.527909,0.527398,0.526913,0.526449,0.526002,0.525567,0.525138,0.524712,0.524282,0.523845,0.523395,0.522927,0.522435,0.521913,0.521355,0.520755,0.520103,0.519394,0.518617,0.517764,0.516825,0.515787,0.514638,0.513364,0.511951,0.510382,0.508638,0.5067,0.504545,0.50215,0.499488,0.496531,0.49325,0.489609,0.485575,0.481109,0.476171,0.47072,0.464711,0.458102,0.450846,0.442898},
	{1.00324,1.00327,1.00328,1.0033,1.0033,1.00331,1.00331,1.0033,1.00329,1.00327,1.00324,1.00321,1.00317,1.00313,1.00307,1.00301,1.00294,1.00286,1.00277,1.00267,1.00256,1.00245,1.00231,1.00217,1.00202,1.00185,1.00166,1.00147,1.00125,1.00102,1.00078,1.00051,1.00023,0.999922,0.999595,0.999246,0.998873,0.998475,0.998049,0.997596,0.997113,0.996598,0.996049,0.995466,0.994845,0.994185,0.993484,0.992739,0.991947,0.991107,0.990216,0.98927,0.988268,0.987205,0.986079,0.984886,0.983623,0.982286,0.980872,0.979377,0.977796,0.976126,0.974361,0.972499,0.970533,0.968461,0.966275,0.963973,0.961548,0.958996,0.956312,0.95349,0.950526,0.947414,0.944148,0.940725,0.937139,0.933385,0.929458,0.925355,0.921071,0.916602,0.911944,0.907096,0.902053,0.896815,0.891379,0.885746,0.879914,0.873886,0.867661,0.861245,0.854638,0.847848,0.840878,0.833735,0.826429,0.818966,0.811359,0.803618,0.795755,0.787784,0.779719,0.771576,0.763372,0.755123,0.746848,0.738564,0.73029,0.722046,0.713851,0.705724,0.697683,0.689747,0.681934,0.674261,0.666745,0.6594,0.652241,0.645281,0.638531,0.632001,0.625701,0.619636,0.613813,0.608235,0.602906,0.597826,0.592996,0.588413,0.584074,0.579977,0.576116,0.572484,0.569076,0.565884,0.562901,0.560117,0.557524,0.555114,0.552876,0.550802,0.548882,0.547108,0.545469,0.543958,0.542565,0.541281,0.5401,0.539012,0.53801,0.537087,0.536236,0.53545,0.534722,0.534046,0.533416,0.532827,0.532273,0.531748,0.531248,0.530767,0.5303,0.529842,0.529388,0.528933,0.528472,0.528,0.52751,0.526998,0.526456,0.525879,0.525259,0.524589,0.523861,0.523065,0.522193,0.521233,0.520174,0.519004,0.517707,0.51627,0.514675,0.512904,0.510936,0.50875,0.506321,0.503623,0.500627,0.497303,0.493617,0.489534,0.485015,0.480021,0.47451,0.468437,0.46176,0.454431,0.446408,0.437646},
	{1.00405,1.0041,1.00415,1.00419,1.00422,1.00425,1.00428,1.0043,1.00431,1.00432,1.00432,1.00432,1.00431,1.00429,1.00426,1.00422,1.00418,1.00412,1.00406,1.00399,1.0039,1.00381,1.0037,1.00358,1.00345,1.0033,1.00314,1.00296,1.00277,1.00256,1.00233,1.00209,1.00182,1.00154,1.00123,1.0009,1.00055,1.00017,0.999763,0.999328,0.998864,0.998368,0.997839,0.997274,0.996673,0.996032,0.99535,0.994624,0.993853,0.993033,0.992162,0.991237,0.990255,0.989214,0.988109,0.986938,0.985698,0.984384,0.982994,0.981522,0.979966,0.97832,0.976581,0.974745,0.972806,0.97076,0.968602,0.966328,0.963932,0.96141,0.958756,0.955965,0.953031,0.949951,0.946718,0.943328,0.939776,0.936056,0.932165,0.928097,0.923849,0.919416,0.914796,0.909985,0.90498,0.899781,0.894384,0.88879,0.882998,0.877009,0.870826,0.864449,0.857884,0.851134,0.844205,0.837104,0.829838,0.822417,0.814851,0.80715,0.799327,0.791396,0.783371,0.775268,0.767103,0.758892,0.750654,0.742408,0.734171,0.725963,0.717802,0.709709,0.701701,0.693797,0.686016,0.678373,0.670886,0.663569,0.656438,0.649503,0.642778,0.636272,0.629994,0.623951,0.618148,0.61259,0.607279,0.602217,0.597402,0.592834,0.58851,0.584426,0.580577,0.576956,0.573559,0.570376,0.567401,0.564625,0.562039,0.559635,0.557402,0.555333,0.553417,0.551645,0.550009,0.548499,0.547106,0.545823,0.544641,0.543551,0.542547,0.54162,0.540764,0.539972,0.539237,0.538553,0.537914,0.537315,0.536748,0.536209,0.535693,0.535194,0.534707,0.534227,0.533748,0.533264,0.532771,0.532263,0.531733,0.531176,0.530584,0.529951,0.529269,0.528529,0.527723,0.526841,0.525872,0.524805,0.523627,0.522323,0.520879,0.519278,0.517501,0.515528,0.513337,0.510904,0.508203,0.505204,0.501878,0.498191,0.494108,0.489591,0.484598,0.47909,0.473022,0.466349,0.459028,0.451012,0.442259,0.432731},
	{1.00483,1.00491,1.00499,1.00506,1.00513,1.00519,1.00525,1.0053,1.00534,1.00538,1.00542,1.00544,1.00546,1.00547,1.00548,1.00547,1.00546,1.00543,1.0054,1.00536,1.0053,1.00523,1.00515,1.00506,1.00495,1.00483,1.0047,1.00455,1.00438,1.0042,1.004,1.00378,1.00354,1.00327,1.00299,1.00268,1.00235,1.002,1.00161,1.0012,1.00076,1.00028,0.999774,0.999231,0.99865,0.998031,0.99737,0.996666,0.995916,0.995117,0.994268,0.993365,0.992405,0.991386,0.990304,0.989156,0.987939,0.986648,0.985281,0.983834,0.982302,0.980682,0.978969,0.977158,0.975246,0.973227,0.971097,0.968851,0.966484,0.963991,0.961366,0.958605,0.955703,0.952654,0.949454,0.946096,0.942577,0.938891,0.935034,0.931001,0.926788,0.922392,0.917808,0.913034,0.908067,0.902905,0.897546,0.891991,0.886238,0.880288,0.874144,0.867808,0.861282,0.854572,0.847683,0.840622,0.833397,0.826015,0.818489,0.810828,0.803045,0.795153,0.787167,0.779102,0.770974,0.762801,0.7546,0.746389,0.738188,0.730014,0.721888,0.713828,0.705852,0.69798,0.690228,0.682615,0.675156,0.667867,0.660761,0.653852,0.647151,0.640668,0.634411,0.628389,0.622606,0.617066,0.611773,0.606727,0.601927,0.597374,0.593063,0.588991,0.585153,0.581544,0.578156,0.574982,0.572014,0.569245,0.566665,0.564266,0.562037,0.559971,0.558058,0.556288,0.554652,0.553142,0.551749,0.550463,0.549278,0.548185,0.547175,0.546242,0.545379,0.544578,0.543833,0.543137,0.542485,0.541869,0.541285,0.540727,0.540188,0.539664,0.539149,0.538636,0.538121,0.537598,0.53706,0.536502,0.535917,0.535297,0.534636,0.533925,0.533156,0.53232,0.531406,0.530404,0.529302,0.528086,0.526743,0.525256,0.523608,0.521781,0.519754,0.517505,0.515008,0.512237,0.509163,0.505756,0.50198,0.497801,0.493179,0.488075,0.482446,0.476249,0.469438,0.461969,0.453798,0.444881,0.435181,0.424662},
	{1.00559,1.0057,1.0058,1.00591,1.00601,1.0061,1.0062,1.00628,1.00636,1.00644,1.00651,1.00657,1.00662,1.00667,1.00671,1.00674,1.00676,1.00677,1.00677,1.00676,1.00674,1.0067,1.00666,1.0066,1.00652,1.00643,1.00633,1.00621,1.00608,1.00592,1.00575,1.00556,1.00535,1.00511,1.00486,1.00458,1.00427,1.00394,1.00358,1.0032,1.00278,1.00233,1.00184,1.00133,1.00077,1.00017,0.999537,0.998857,0.99813,0.997355,0.996529,0.995649,0.994713,0.993718,0.99266,0.991536,0.990343,0.989077,0.987735,0.986312,0.984806,0.983211,0.981524,0.97974,0.977854,0.975863,0.97376,0.971542,0.969204,0.96674,0.964145,0.961415,0.958543,0.955526,0.952357,0.949031,0.945545,0.941893,0.938069,0.934071,0.929893,0.925532,0.920984,0.916246,0.911316,0.906191,0.90087,0.895352,0.889637,0.883727,0.877621,0.871323,0.864837,0.858166,0.851316,0.844295,0.837108,0.829766,0.822278,0.814656,0.806912,0.799058,0.79111,0.783083,0.774992,0.766855,0.75869,0.750515,0.742348,0.734208,0.726115,0.718086,0.710142,0.7023,0.694578,0.686994,0.679562,0.6723,0.665219,0.658335,0.651657,0.645197,0.638962,0.632959,0.627196,0.621674,0.616398,0.611368,0.606585,0.602045,0.597748,0.593688,0.589862,0.586263,0.582885,0.57972,0.576761,0.573999,0.571426,0.569032,0.566809,0.564748,0.562838,0.561071,0.559438,0.55793,0.556538,0.555254,0.554068,0.552974,0.551963,0.551028,0.550161,0.549357,0.548607,0.547905,0.547246,0.546623,0.54603,0.545462,0.544911,0.544374,0.543844,0.543314,0.542781,0.542236,0.541675,0.54109,0.540475,0.539822,0.539124,0.538372,0.537557,0.536669,0.535699,0.534633,0.53346,0.532166,0.530736,0.529153,0.527399,0.525454,0.523296,0.520902,0.518246,0.5153,0.512034,0.508415,0.504409,0.499977,0.495081,0.489679,0.483727,0.477183,0.47,0.462133,0.453539,0.444177,0.434012,0.423011},
	{1.0063,1.00644,1.00658,1.00672,1.00686,1.00699,1.00712,1.00724,1.00736,1.00747,1.00758,1.00768,1.00777,1.00785,1.00793,1.008,1.00806,1.00811,1.00815,1.00818,1.00819,1.0082,1.00819,1.00817,1.00813,1.00808,1.00801,1.00793,1.00783,1.00771,1.00757,1.00742,1.00724,1.00704,1.00681,1.00657,1.00629,1.00599,1.00566,1.00531,1.00492,1.0045,1.00404,1.00355,1.00302,1.00245,1.00184,1.00119,1.00049,0.999738,0.998938,0.998085,0.997174,0.996204,0.995172,0.994073,0.992906,0.991666,0.99035,0.988954,0.987473,0.985905,0.984245,0.982488,0.98063,0.978666,0.976592,0.974402,0.972093,0.969658,0.967093,0.964393,0.961552,0.958565,0.955428,0.952135,0.948681,0.945062,0.941272,0.937307,0.933164,0.928838,0.924325,0.919623,0.914729,0.90964,0.904356,0.898876,0.893198,0.887325,0.881257,0.874998,0.868549,0.861917,0.855106,0.848122,0.840974,0.83367,0.82622,0.818636,0.810929,0.803113,0.795201,0.78721,0.779156,0.771054,0.762924,0.754783,0.746649,0.738542,0.730481,0.722484,0.71457,0.706758,0.699064,0.691507,0.684102,0.676865,0.66981,0.662949,0.656293,0.649854,0.64364,0.637657,0.631912,0.626408,0.621148,0.616133,0.611364,0.606838,0.602553,0.598505,0.594689,0.5911,0.58773,0.584574,0.581621,0.578866,0.576298,0.573909,0.57169,0.569631,0.567724,0.565958,0.564326,0.562817,0.561424,0.560137,0.558949,0.55785,0.556834,0.555893,0.555019,0.554205,0.553445,0.552732,0.552059,0.55142,0.55081,0.550222,0.549649,0.549087,0.548528,0.547968,0.547399,0.546815,0.54621,0.545576,0.544906,0.544192,0.543425,0.542597,0.541697,0.540716,0.53964,0.538458,0.537156,0.535718,0.534128,0.532368,0.530418,0.528256,0.525859,0.5232,0.520253,0.516986,0.513368,0.509364,0.504935,0.500044,0.494648,0.488705,0.482169,0.474997,0.467144,0.458564,0.449218,0.439068,0.428082,0.416236},
	{1.00697,1.00715,1.00733,1.0075,1.00767,1.00784,1.00801,1.00817,1.00832,1.00847,1.00862,1.00876,1.0089,1.00902,1.00914,1.00925,1.00935,1.00945,1.00953,1.0096,1.00966,1.00971,1.00974,1.00976,1.00977,1.00976,1.00973,1.00969,1.00963,1.00955,1.00945,1.00934,1.0092,1.00903,1.00885,1.00863,1.0084,1.00813,1.00784,1.00752,1.00716,1.00677,1.00635,1.00589,1.00539,1.00485,1.00427,1.00365,1.00298,1.00226,1.00149,1.00066,0.999781,0.998839,0.997835,0.996764,0.995625,0.994412,0.993124,0.991755,0.990303,0.988763,0.98713,0.985401,0.983572,0.981636,0.979591,0.977431,0.975151,0.972746,0.970211,0.967541,0.964731,0.961776,0.95867,0.955409,0.951988,0.948401,0.944644,0.940714,0.936604,0.932312,0.927835,0.923168,0.918309,0.913257,0.908009,0.902565,0.896924,0.891088,0.885058,0.878835,0.872424,0.865829,0.859055,0.852109,0.844999,0.837732,0.83032,0.822772,0.815102,0.807322,0.799446,0.791491,0.783471,0.775404,0.767308,0.7592,0.751099,0.743024,0.734994,0.727027,0.719143,0.711359,0.703693,0.696163,0.688784,0.681572,0.67454,0.667702,0.661069,0.654651,0.648456,0.642492,0.636765,0.631278,0.626034,0.621035,0.616279,0.611767,0.607494,0.603457,0.599652,0.596073,0.592712,0.589564,0.586619,0.58387,0.581308,0.578924,0.57671,0.574655,0.572751,0.570988,0.569357,0.56785,0.566457,0.56517,0.56398,0.56288,0.561862,0.560917,0.560039,0.559221,0.558455,0.557734,0.557054,0.556406,0.555785,0.555184,0.554598,0.55402,0.553444,0.552864,0.552274,0.551665,0.551032,0.550368,0.549664,0.548912,0.548103,0.547227,0.546275,0.545235,0.544095,0.542841,0.541458,0.539932,0.538243,0.536375,0.534304,0.532009,0.529466,0.526646,0.523521,0.52006,0.516229,0.511992,0.50731,0.502143,0.496448,0.490183,0.483301,0.475758,0.46751,0.458513,0.448727,0.43812,0.426661,0.414332},
	{1.00761,1.00782,1.00803,1.00824,1.00845,1.00865,1.00886,1.00906,1.00925,1.00945,1.00963,1.00982,1.00999,1.01016,1.01033,1.01048,1.01063,1.01077,1.0109,1.01101,1.01112,1.01121,1.01129,1.01136,1.01141,1.01145,1.01147,1.01148,1.01146,1.01143,1.01137,1.0113,1.0112,1.01108,1.01094,1.01077,1.01057,1.01035,1.0101,1.00981,1.00949,1.00914,1.00876,1.00834,1.00787,1.00737,1.00683,1.00624,1.0056,1.00491,1.00417,1.00338,1.00253,1.00162,1.00064,0.999603,0.998493,0.997311,0.996052,0.994713,0.99329,0.99178,0.990177,0.988477,0.986677,0.984772,0.982756,0.980626,0.978376,0.976002,0.973497,0.970858,0.96808,0.965156,0.962082,0.958853,0.955464,0.951911,0.948187,0.94429,0.940215,0.935957,0.931514,0.926882,0.922058,0.917041,0.911829,0.906421,0.900816,0.895016,0.889022,0.882837,0.876462,0.869904,0.863167,0.856257,0.849183,0.841953,0.834577,0.827065,0.819431,0.811686,0.803845,0.795924,0.787939,0.779906,0.771842,0.763766,0.755697,0.747653,0.739653,0.731715,0.72386,0.716103,0.708464,0.70096,0.693606,0.686418,0.679409,0.672593,0.665981,0.659582,0.653407,0.647461,0.641751,0.63628,0.631051,0.626066,0.621324,0.616824,0.612562,0.608536,0.604741,0.60117,0.597818,0.594676,0.591738,0.588994,0.586437,0.584058,0.581846,0.579794,0.577891,0.576129,0.574498,0.57299,0.571596,0.570306,0.569113,0.568008,0.566984,0.566033,0.565147,0.564319,0.563542,0.56281,0.562115,0.561452,0.560813,0.560192,0.559583,0.55898,0.558375,0.557763,0.557137,0.556488,0.555811,0.555097,0.554337,0.553523,0.552646,0.551694,0.550657,0.549522,0.548276,0.546906,0.545394,0.543724,0.541877,0.539832,0.537568,0.535059,0.532279,0.529199,0.525789,0.522015,0.517841,0.51323,0.508141,0.502533,0.496361,0.489582,0.482149,0.474019,0.465148,0.455494,0.445023,0.433704,0.421515,0.408446},
	{1.0082,1.00845,1.00869,1.00894,1.00918,1.00943,1.00967,1.00991,1.01015,1.01038,1.01061,1.01084,1.01106,1.01128,1.01149,1.01169,1.01188,1.01207,1.01224,1.01241,1.01257,1.01271,1.01284,1.01296,1.01306,1.01315,1.01322,1.01327,1.01331,1.01333,1.01332,1.0133,1.01325,1.01318,1.01308,1.01296,1.01281,1.01263,1.01242,1.01218,1.01191,1.0116,1.01126,1.01087,1.01045,1.00999,1.00948,1.00893,1.00833,1.00768,1.00698,1.00622,1.0054,1.00453,1.00359,1.00258,1.00151,1.00036,0.999129,0.997823,0.996432,0.994953,0.993381,0.991713,0.989944,0.98807,0.986086,0.983987,0.981769,0.979426,0.976953,0.974346,0.971599,0.968707,0.965666,0.962469,0.959113,0.955593,0.951903,0.948039,0.943997,0.939774,0.935365,0.930767,0.925979,0.920997,0.91582,0.910447,0.904879,0.899114,0.893156,0.887006,0.880668,0.874146,0.867445,0.860571,0.853533,0.846338,0.838997,0.831521,0.823921,0.816211,0.808404,0.800517,0.792565,0.784564,0.776532,0.768488,0.76045,0.752435,0.744464,0.736556,0.728727,0.720998,0.713385,0.705906,0.698576,0.691411,0.684424,0.67763,0.671038,0.664659,0.658502,0.652574,0.64688,0.641425,0.636211,0.63124,0.626511,0.622023,0.617773,0.613758,0.609972,0.60641,0.603066,0.599932,0.597,0.594263,0.591711,0.589336,0.587129,0.58508,0.58318,0.58142,0.579791,0.578283,0.576889,0.575599,0.574404,0.573298,0.572271,0.571316,0.570426,0.569593,0.56881,0.568071,0.567368,0.566695,0.566045,0.565412,0.564789,0.56417,0.563549,0.562917,0.562269,0.561596,0.560891,0.560147,0.559353,0.558501,0.557582,0.556583,0.555494,0.554302,0.552992,0.551551,0.54996,0.548204,0.546261,0.54411,0.541729,0.539091,0.53617,0.532935,0.529354,0.525394,0.521018,0.516186,0.510858,0.50499,0.49854,0.491461,0.483708,0.475238,0.466007,0.455976,0.445113,0.433388,0.420785,0.407297},
	{1.00876,1.00904,1.00932,1.0096,1.00988,1.01016,1.01044,1.01072,1.011,1.01127,1.01155,1.01182,1.01209,1.01235,1.01261,1.01286,1.0131,1.01334,1.01357,1.01378,1.01399,1.01419,1.01437,1.01454,1.0147,1.01484,1.01497,1.01507,1.01516,1.01524,1.01528,1.01531,1.01532,1.0153,1.01526,1.01518,1.01509,1.01496,1.0148,1.01461,1.01438,1.01413,1.01383,1.01349,1.01312,1.0127,1.01224,1.01173,1.01117,1.01056,1.0099,1.00918,1.0084,1.00757,1.00666,1.00569,1.00465,1.00354,1.00235,1.00108,0.999722,0.998277,0.996739,0.995105,0.99337,0.991528,0.989577,0.987511,0.985325,0.983015,0.980575,0.978001,0.975287,0.972428,0.96942,0.966256,0.962934,0.959446,0.95579,0.95196,0.947952,0.943763,0.939388,0.934825,0.930071,0.925124,0.919982,0.914645,0.909111,0.903382,0.897459,0.891345,0.885042,0.878555,0.871889,0.865051,0.858047,0.850887,0.843581,0.836139,0.828573,0.820896,0.813123,0.805268,0.797348,0.789379,0.781378,0.773364,0.765356,0.75737,0.749428,0.741546,0.733745,0.726041,0.718453,0.710998,0.703691,0.696548,0.689583,0.682809,0.676236,0.669876,0.663737,0.657825,0.652147,0.646707,0.641507,0.636549,0.631832,0.627355,0.623115,0.619109,0.615332,0.611778,0.608441,0.605313,0.602387,0.599654,0.597106,0.594734,0.592529,0.590482,0.588583,0.586823,0.585193,0.583684,0.582287,0.580993,0.579794,0.578682,0.577648,0.576685,0.575786,0.574942,0.574147,0.573393,0.572674,0.571983,0.571313,0.570657,0.570009,0.569361,0.568708,0.56804,0.567352,0.566634,0.56588,0.56508,0.564225,0.563305,0.562309,0.561225,0.560042,0.558745,0.55732,0.55575,0.554017,0.552103,0.549986,0.547643,0.54505,0.542179,0.539001,0.535484,0.531596,0.527298,0.522555,0.517323,0.511562,0.505227,0.498274,0.490656,0.482331,0.473253,0.463383,0.452685,0.44113,0.428697,0.415376,0.401172},
	{1.00928,1.00959,1.0099,1.01021,1.01053,1.01085,1.01117,1.01149,1.01181,1.01213,1.01245,1.01276,1.01308,1.01339,1.01369,1.01399,1.01429,1.01458,1.01486,1.01513,1.01539,1.01564,1.01588,1.01611,1.01632,1.01652,1.0167,1.01687,1.01702,1.01715,1.01725,1.01734,1.0174,1.01744,1.01745,1.01744,1.0174,1.01733,1.01722,1.01709,1.01692,1.01671,1.01647,1.01618,1.01586,1.01549,1.01508,1.01461,1.0141,1.01354,1.01292,1.01225,1.01152,1.01072,1.00986,1.00893,1.00793,1.00686,1.00571,1.00447,1.00316,1.00175,1.00025,0.998649,0.996949,0.995144,0.993228,0.991196,0.989045,0.986769,0.984364,0.981824,0.979144,0.976319,0.973344,0.970215,0.966926,0.963473,0.959851,0.956055,0.952082,0.947927,0.943587,0.939058,0.934339,0.929427,0.924319,0.919017,0.913518,0.907824,0.901936,0.895857,0.889589,0.883136,0.876505,0.869701,0.862732,0.855606,0.848333,0.840925,0.833392,0.825748,0.818007,0.810184,0.802295,0.794357,0.786387,0.778402,0.770422,0.762465,0.75455,0.746695,0.73892,0.731241,0.723677,0.716245,0.708961,0.70184,0.694895,0.688141,0.681587,0.675245,0.669123,0.663228,0.657565,0.652139,0.646953,0.642007,0.637302,0.632836,0.628607,0.62461,0.620842,0.617296,0.613966,0.610845,0.607925,0.605198,0.602654,0.600286,0.598085,0.59604,0.594143,0.592384,0.590755,0.589246,0.587849,0.586554,0.585353,0.584238,0.583201,0.582234,0.581329,0.58048,0.579677,0.578916,0.578187,0.577485,0.576803,0.576134,0.575471,0.574806,0.574133,0.573444,0.572731,0.571987,0.571203,0.57037,0.569478,0.568516,0.567475,0.566341,0.565101,0.563742,0.562247,0.560601,0.558784,0.556777,0.554557,0.552101,0.549383,0.546375,0.543048,0.539367,0.535299,0.530807,0.52585,0.520389,0.514379,0.507777,0.500536,0.492613,0.483962,0.474541,0.464311,0.453239,0.441298,0.42847,0.41475,0.400148},
	{1.00976,1.0101,1.01045,1.0108,1.01115,1.0115,1.01186,1.01222,1.01258,1.01294,1.01331,1.01367,1.01403,1.01439,1.01474,1.01509,1.01544,1.01578,1.01611,1.01644,1.01676,1.01707,1.01736,1.01765,1.01792,1.01818,1.01842,1.01865,1.01886,1.01905,1.01922,1.01937,1.01949,1.01959,1.01967,1.01972,1.01973,1.01972,1.01968,1.0196,1.01949,1.01934,1.01916,1.01893,1.01866,1.01835,1.01799,1.01758,1.01712,1.01661,1.01604,1.01542,1.01473,1.01399,1.01317,1.01229,1.01133,1.0103,1.0092,1.008,1.00673,1.00536,1.0039,1.00234,1.00068,0.998911,0.997032,0.995038,0.992924,0.990685,0.988315,0.985811,0.983167,0.980377,0.977438,0.974344,0.97109,0.967672,0.964084,0.960324,0.956385,0.952265,0.947959,0.943466,0.938781,0.933904,0.928831,0.923563,0.918099,0.91244,0.906586,0.900541,0.894308,0.887889,0.881292,0.874522,0.867587,0.860494,0.853255,0.845879,0.838378,0.830766,0.823057,0.815265,0.807406,0.799497,0.791556,0.783601,0.775648,0.767718,0.75983,0.752,0.744249,0.736595,0.729054,0.721645,0.714382,0.707281,0.700356,0.69362,0.687085,0.68076,0.674653,0.668773,0.663125,0.657712,0.652538,0.647604,0.64291,0.638454,0.634234,0.630246,0.626485,0.622946,0.619622,0.616506,0.61359,0.610866,0.608326,0.60596,0.603759,0.601715,0.599818,0.598058,0.596426,0.594914,0.593513,0.592212,0.591005,0.589883,0.588837,0.58786,0.586944,0.586081,0.585263,0.584484,0.583737,0.583013,0.582307,0.58161,0.580916,0.580217,0.579506,0.578774,0.578015,0.577218,0.576375,0.575476,0.574511,0.573469,0.572338,0.571104,0.569754,0.568273,0.566643,0.564847,0.562864,0.560673,0.558251,0.555573,0.55261,0.549333,0.54571,0.541707,0.537286,0.53241,0.527037,0.521124,0.514627,0.507501,0.4997,0.491181,0.481899,0.471815,0.460894,0.449106,0.436431,0.422862,0.408402,0.393075},
	{1.01022,1.01059,1.01096,1.01134,1.01173,1.01212,1.01251,1.01291,1.01331,1.01372,1.01412,1.01453,1.01494,1.01534,1.01575,1.01615,1.01655,1.01694,1.01733,1.01771,1.01809,1.01846,1.01881,1.01916,1.01949,1.01981,1.02012,1.02041,1.02068,1.02094,1.02117,1.02139,1.02158,1.02175,1.02189,1.022,1.02208,1.02214,1.02216,1.02215,1.0221,1.02202,1.02189,1.02173,1.02152,1.02127,1.02097,1.02062,1.02022,1.01976,1.01925,1.01868,1.01805,1.01735,1.01659,1.01576,1.01485,1.01387,1.01281,1.01166,1.01043,1.00911,1.00769,1.00617,1.00455,1.00283,1.00099,0.999034,0.996959,0.994759,0.992427,0.989961,0.987354,0.984602,0.9817,0.978642,0.975424,0.972042,0.968491,0.964765,0.960862,0.956778,0.952507,0.948049,0.9434,0.938557,0.933519,0.928286,0.922857,0.917232,0.911413,0.905402,0.899202,0.892818,0.886254,0.879517,0.872615,0.865555,0.858348,0.851004,0.843535,0.835954,0.828275,0.820513,0.812684,0.804804,0.796891,0.788963,0.781038,0.773134,0.765271,0.757466,0.749739,0.742108,0.734589,0.727201,0.719958,0.712877,0.705971,0.699253,0.692735,0.686425,0.680334,0.674468,0.668833,0.663433,0.658271,0.653348,0.648664,0.644217,0.640005,0.636024,0.63227,0.628737,0.625418,0.622306,0.619393,0.616672,0.614133,0.611768,0.609568,0.607522,0.605623,0.60386,0.602225,0.600708,0.5993,0.597993,0.596777,0.595645,0.594587,0.593597,0.592665,0.591785,0.590948,0.590147,0.589375,0.588624,0.587887,0.587156,0.586423,0.585682,0.584923,0.584138,0.583319,0.582456,0.58154,0.580559,0.579504,0.578361,0.577118,0.575761,0.574274,0.57264,0.570842,0.56886,0.566671,0.564254,0.561583,0.558629,0.555364,0.551756,0.547771,0.543372,0.53852,0.533175,0.527294,0.520833,0.513746,0.505989,0.497517,0.488286,0.478254,0.467387,0.455655,0.443035,0.429517,0.415103,0.399813,0.383683},
	{1.01064,1.01104,1.01144,1.01185,1.01227,1.0127,1.01313,1.01357,1.01401,1.01445,1.0149,1.01535,1.01581,1.01626,1.01671,1.01717,1.01762,1.01807,1.01851,1.01895,1.01938,1.01981,1.02023,1.02064,1.02103,1.02142,1.02179,1.02215,1.02249,1.02281,1.02311,1.02339,1.02365,1.02389,1.0241,1.02428,1.02444,1.02456,1.02465,1.02471,1.02473,1.02472,1.02466,1.02456,1.02442,1.02424,1.024,1.02371,1.02338,1.02298,1.02253,1.02202,1.02145,1.02081,1.0201,1.01933,1.01847,1.01754,1.01653,1.01544,1.01426,1.01298,1.01161,1.01014,1.00857,1.00689,1.00509,1.00318,1.00115,0.99899,0.996699,0.994273,0.991706,0.988993,0.986129,0.98311,0.979931,0.976586,0.973072,0.969383,0.965517,0.961468,0.957234,0.952812,0.948198,0.943391,0.938388,0.93319,0.927796,0.922205,0.916421,0.910444,0.904278,0.897927,0.891397,0.884693,0.877824,0.870796,0.863621,0.856308,0.84887,0.84132,0.833671,0.825939,0.818139,0.810288,0.802402,0.794501,0.786602,0.778724,0.770886,0.763105,0.755402,0.747793,0.740296,0.732928,0.725706,0.718645,0.711757,0.705057,0.698555,0.692262,0.686186,0.680334,0.674713,0.669326,0.664176,0.659264,0.65459,0.650153,0.645951,0.641979,0.638233,0.634707,0.631394,0.628289,0.625382,0.622666,0.620132,0.617771,0.615574,0.613532,0.611636,0.609875,0.608242,0.606726,0.60532,0.604013,0.602797,0.601665,0.600607,0.599615,0.598682,0.5978,0.596961,0.596157,0.595381,0.594626,0.593884,0.593148,0.592409,0.591661,0.590894,0.590101,0.589272,0.588399,0.58747,0.586477,0.585407,0.584248,0.582988,0.581611,0.580102,0.578445,0.576621,0.57461,0.572391,0.569939,0.567231,0.564237,0.560928,0.557273,0.553236,0.548781,0.543869,0.538459,0.532508,0.525972,0.518807,0.510965,0.502403,0.493076,0.482945,0.471972,0.46013,0.447395,0.433759,0.419222,0.403805,0.387545},
	{1.01103,1.01146,1.01189,1.01233,1.01279,1.01325,1.01371,1.01419,1.01467,1.01515,1.01564,1.01614,1.01664,1.01714,1.01764,1.01814,1.01865,1.01915,1.01965,1.02015,1.02064,1.02112,1.0216,1.02208,1.02254,1.02299,1.02343,1.02385,1.02426,1.02465,1.02503,1.02538,1.02571,1.02602,1.02631,1.02656,1.02679,1.02699,1.02716,1.02729,1.02738,1.02744,1.02746,1.02743,1.02736,1.02724,1.02708,1.02686,1.02659,1.02627,1.02588,1.02544,1.02493,1.02435,1.02371,1.02299,1.0222,1.02132,1.02037,1.01933,1.0182,1.01698,1.01566,1.01424,1.01272,1.01109,1.00934,1.00747,1.00549,1.00337,1.00113,0.998743,0.996218,0.993546,0.990724,0.987745,0.984605,0.981299,0.977824,0.974174,0.970345,0.966334,0.962137,0.957751,0.953174,0.948403,0.943436,0.938273,0.932914,0.927358,0.921608,0.915665,0.909533,0.903216,0.896719,0.890048,0.88321,0.876215,0.869071,0.861789,0.854382,0.846861,0.839242,0.831538,0.823766,0.815943,0.808084,0.800209,0.792336,0.784482,0.776668,0.768911,0.761229,0.753642,0.746166,0.738819,0.731616,0.724572,0.717702,0.711019,0.704533,0.698255,0.692193,0.686355,0.680746,0.675371,0.670232,0.66533,0.660666,0.656239,0.652044,0.64808,0.644341,0.640821,0.637514,0.634414,0.631511,0.628799,0.626268,0.62391,0.621715,0.619674,0.617778,0.616018,0.614384,0.612867,0.611459,0.610149,0.608931,0.607794,0.606731,0.605734,0.604794,0.603904,0.603055,0.602242,0.601454,0.600686,0.59993,0.599177,0.59842,0.597651,0.596862,0.596043,0.595186,0.594281,0.593318,0.592286,0.591173,0.589967,0.588653,0.587218,0.585645,0.583917,0.582014,0.579917,0.577603,0.575047,0.572223,0.569104,0.565657,0.561851,0.55765,0.553016,0.54791,0.54229,0.536113,0.529335,0.521908,0.513789,0.504932,0.495294,0.484836,0.473524,0.461331,0.448237,0.434236,0.419334,0.403558,0.386947},
	{1.01139,1.01185,1.01231,1.01278,1.01327,1.01376,1.01426,1.01477,1.01529,1.01581,1.01634,1.01688,1.01743,1.01797,1.01853,1.01908,1.01963,1.02019,1.02075,1.0213,1.02185,1.0224,1.02294,1.02348,1.024,1.02452,1.02503,1.02552,1.026,1.02647,1.02692,1.02734,1.02775,1.02814,1.0285,1.02883,1.02914,1.02941,1.02966,1.02986,1.03004,1.03017,1.03027,1.03032,1.03032,1.03028,1.03019,1.03005,1.02986,1.0296,1.02929,1.02891,1.02847,1.02797,1.02739,1.02674,1.02601,1.0252,1.02431,1.02333,1.02226,1.02109,1.01983,1.01847,1.017,1.01542,1.01372,1.01191,1.00997,1.0079,1.0057,1.00337,1.00089,0.998259,0.995479,0.992543,0.989445,0.986181,0.982745,0.979135,0.975346,0.971374,0.967215,0.962867,0.958327,0.953592,0.948662,0.943535,0.938211,0.932691,0.926975,0.921067,0.914968,0.908684,0.90222,0.895581,0.888776,0.881812,0.874699,0.867447,0.86007,0.852578,0.844987,0.837312,0.829567,0.82177,0.813938,0.806088,0.798239,0.790409,0.782617,0.774882,0.767222,0.759656,0.752199,0.744871,0.737686,0.730659,0.723806,0.717137,0.710666,0.704402,0.698353,0.692527,0.68693,0.681565,0.676436,0.671543,0.666887,0.662467,0.658279,0.65432,0.650586,0.64707,0.643767,0.640669,0.637768,0.635057,0.632526,0.630167,0.62797,0.625927,0.624028,0.622263,0.620623,0.6191,0.617683,0.616364,0.615135,0.613985,0.612908,0.611895,0.610936,0.610026,0.609154,0.608315,0.607499,0.606698,0.605905,0.605112,0.604311,0.603492,0.602647,0.601766,0.600841,0.59986,0.598812,0.597686,0.596469,0.595147,0.593706,0.592129,0.590399,0.588498,0.586404,0.584096,0.581549,0.578738,0.575633,0.572205,0.568421,0.564246,0.559642,0.554571,0.54899,0.542857,0.536126,0.528753,0.520691,0.511896,0.502324,0.491936,0.480695,0.468573,0.45555,0.441616,0.426775,0.411049,0.394478,0.377119},
	{1.01173,1.01221,1.0127,1.0132,1.01372,1.01424,1.01478,1.01532,1.01588,1.01644,1.01701,1.01759,1.01818,1.01877,1.01937,1.01998,1.02058,1.02119,1.0218,1.02242,1.02303,1.02363,1.02424,1.02484,1.02543,1.02602,1.0266,1.02716,1.02771,1.02825,1.02878,1.02928,1.02976,1.03023,1.03067,1.03108,1.03147,1.03182,1.03215,1.03244,1.03269,1.03291,1.03309,1.03322,1.03331,1.03335,1.03334,1.03328,1.03316,1.03298,1.03275,1.03245,1.03208,1.03165,1.03114,1.03056,1.0299,1.02916,1.02834,1.02742,1.02642,1.02532,1.02412,1.02281,1.0214,1.01988,1.01824,1.01648,1.01459,1.01258,1.01043,1.00814,1.00571,1.00313,1.0004,0.997505,0.994451,0.99123,0.987838,0.98427,0.980521,0.976589,0.972471,0.968162,0.96366,0.958963,0.95407,0.94898,0.943692,0.938207,0.932527,0.926653,0.920589,0.914338,0.907907,0.9013,0.894527,0.887594,0.880511,0.87329,0.865942,0.858479,0.850917,0.843268,0.83555,0.827779,0.819972,0.812147,0.804322,0.796515,0.788746,0.781032,0.773393,0.765846,0.758409,0.751099,0.743931,0.736921,0.730083,0.72343,0.716973,0.710722,0.704686,0.698872,0.693285,0.687931,0.682811,0.677927,0.673279,0.668866,0.664684,0.660732,0.657002,0.653491,0.650192,0.647097,0.644199,0.641489,0.638959,0.6366,0.634403,0.632359,0.630457,0.628689,0.627046,0.625518,0.624095,0.62277,0.621532,0.620374,0.619286,0.618261,0.617289,0.616363,0.615475,0.614616,0.613778,0.612953,0.612134,0.611311,0.610476,0.60962,0.608733,0.607807,0.60683,0.605792,0.604682,0.603486,0.602192,0.600785,0.599249,0.597568,0.595723,0.593695,0.591461,0.588998,0.586282,0.583284,0.579975,0.576324,0.572295,0.567854,0.56296,0.557575,0.551654,0.545154,0.53803,0.530235,0.521725,0.512455,0.502383,0.491471,0.479686,0.467006,0.453413,0.438906,0.423496,0.407215,0.39011,0.372255},
	{1.01205,1.01255,1.01307,1.0136,1.01414,1.01469,1.01526,1.01584,1.01643,1.01703,1.01764,1.01826,1.0189,1.01953,1.02018,1.02083,1.02149,1.02215,1.02282,1.02349,1.02416,1.02483,1.0255,1.02616,1.02682,1.02748,1.02812,1.02876,1.02939,1.03,1.0306,1.03118,1.03175,1.03229,1.03281,1.03331,1.03378,1.03422,1.03463,1.03501,1.03535,1.03565,1.03591,1.03613,1.0363,1.03643,1.03651,1.03653,1.03649,1.0364,1.03625,1.03603,1.03575,1.03539,1.03496,1.03446,1.03388,1.03321,1.03246,1.03161,1.03068,1.02964,1.02851,1.02727,1.02592,1.02446,1.02288,1.02118,1.01935,1.01739,1.0153,1.01306,1.01068,1.00815,1.00547,1.00263,0.999621,0.996446,0.993099,0.989574,0.985869,0.98198,0.977902,0.973634,0.969172,0.964515,0.95966,0.954607,0.949357,0.943908,0.938263,0.932424,0.926394,0.920178,0.913779,0.907206,0.900463,0.893562,0.88651,0.879318,0.871999,0.864565,0.85703,0.849409,0.841717,0.833972,0.826189,0.818388,0.810586,0.802802,0.795054,0.787361,0.779742,0.772214,0.764795,0.757502,0.750351,0.743357,0.736534,0.729895,0.723451,0.717213,0.711189,0.705386,0.699809,0.694464,0.689353,0.684478,0.679837,0.67543,0.671255,0.667308,0.663583,0.660076,0.656779,0.653687,0.650791,0.648082,0.645553,0.643193,0.640995,0.638949,0.637045,0.635273,0.633626,0.632092,0.630663,0.62933,0.628084,0.626915,0.625816,0.624777,0.623791,0.622849,0.621942,0.621062,0.620201,0.61935,0.618502,0.617646,0.616775,0.615879,0.614948,0.613973,0.612942,0.611843,0.610666,0.609396,0.608019,0.606521,0.604885,0.603092,0.601125,0.598961,0.596578,0.593952,0.591056,0.587861,0.584336,0.580449,0.576162,0.57144,0.566243,0.560527,0.55425,0.547368,0.539833,0.531601,0.522626,0.512865,0.502278,0.490829,0.478491,0.465242,0.451074,0.435992,0.420016,0.403186,0.385562,0.367229},
	{1.01235,1.01287,1.01341,1.01397,1.01454,1.01512,1.01572,1.01633,1.01695,1.01759,1.01824,1.0189,1.01958,1.02026,1.02095,1.02165,1.02236,1.02308,1.0238,1.02452,1.02525,1.02598,1.02671,1.02744,1.02817,1.02889,1.02961,1.03032,1.03102,1.03171,1.03239,1.03305,1.0337,1.03432,1.03493,1.03551,1.03607,1.03659,1.03709,1.03756,1.03799,1.03838,1.03873,1.03904,1.03931,1.03952,1.03969,1.0398,1.03985,1.03985,1.03978,1.03965,1.03946,1.03919,1.03884,1.03842,1.03792,1.03733,1.03665,1.03589,1.03503,1.03407,1.033,1.03183,1.03055,1.02916,1.02764,1.02601,1.02424,1.02234,1.02031,1.01813,1.0158,1.01333,1.0107,1.00791,1.00495,1.00183,0.998527,0.995049,0.99139,0.987545,0.983511,0.979285,0.974865,0.970248,0.965433,0.96042,0.955207,0.949796,0.944187,0.938384,0.932389,0.926207,0.919842,0.913301,0.906591,0.89972,0.892699,0.885537,0.878247,0.870841,0.863333,0.855739,0.848073,0.840352,0.832594,0.824816,0.817037,0.809275,0.801548,0.793875,0.786275,0.778766,0.771365,0.764089,0.756954,0.749975,0.743167,0.736542,0.730111,0.723885,0.717872,0.712079,0.706513,0.701178,0.696075,0.691208,0.686574,0.682174,0.678005,0.674063,0.670343,0.66684,0.663548,0.660458,0.657564,0.654858,0.65233,0.649971,0.647773,0.645726,0.643821,0.642048,0.640398,0.638861,0.637428,0.63609,0.634839,0.633664,0.632557,0.63151,0.630514,0.62956,0.628641,0.627747,0.626871,0.626003,0.625135,0.624258,0.623363,0.62244,0.62148,0.620472,0.619404,0.618266,0.617043,0.615724,0.614293,0.612735,0.611032,0.609167,0.607119,0.604867,0.602387,0.599655,0.596642,0.593319,0.589655,0.585615,0.581163,0.576262,0.570869,0.564944,0.558441,0.551315,0.543521,0.535013,0.525745,0.515676,0.504767,0.492982,0.480297,0.466694,0.452167,0.436724,0.420392,0.403215,0.385259,0.366615},
	{1.01262,1.01317,1.01373,1.01431,1.01491,1.01552,1.01615,1.01679,1.01745,1.01812,1.01881,1.01951,1.02022,1.02095,1.02169,1.02244,1.02319,1.02396,1.02474,1.02552,1.0263,1.02709,1.02789,1.02868,1.02948,1.03027,1.03106,1.03184,1.03262,1.03339,1.03414,1.03489,1.03561,1.03632,1.03701,1.03768,1.03833,1.03895,1.03953,1.04009,1.04061,1.0411,1.04154,1.04195,1.04231,1.04262,1.04288,1.04308,1.04323,1.04332,1.04335,1.04331,1.0432,1.04302,1.04277,1.04243,1.04202,1.04152,1.04092,1.04024,1.03946,1.03858,1.03759,1.0365,1.03529,1.03397,1.03252,1.03096,1.02926,1.02742,1.02545,1.02333,1.02107,1.01865,1.01608,1.01334,1.01044,1.00737,1.00412,1.00069,0.99708,0.993282,0.989294,0.985113,0.980736,0.976161,0.971388,0.966415,0.961241,0.955868,0.950298,0.944531,0.938572,0.932424,0.926093,0.919585,0.912907,0.906068,0.899077,0.891945,0.884683,0.877305,0.869825,0.862256,0.854616,0.84692,0.839185,0.83143,0.823673,0.815932,0.808225,0.800572,0.792991,0.785499,0.778115,0.770855,0.763735,0.756771,0.749976,0.743364,0.736945,0.73073,0.724728,0.718945,0.713388,0.708061,0.702966,0.698105,0.693478,0.689084,0.68492,0.680982,0.677265,0.673765,0.670475,0.667387,0.664494,0.661788,0.65926,0.6569,0.6547,0.652651,0.650742,0.648964,0.647309,0.645766,0.644326,0.642979,0.641718,0.640532,0.639413,0.638351,0.637339,0.636368,0.635429,0.634513,0.633611,0.632716,0.631817,0.630906,0.629973,0.629008,0.628001,0.626941,0.625816,0.624613,0.623321,0.621924,0.620407,0.618753,0.616946,0.614965,0.612789,0.610397,0.607763,0.604861,0.601663,0.598137,0.594251,0.589969,0.585254,0.580067,0.574365,0.568106,0.561244,0.553733,0.545528,0.536583,0.526855,0.516301,0.504886,0.492577,0.479355,0.465205,0.45013,0.434143,0.417281,0.399598,0.38117,0.3621},
	{1.01288,1.01345,1.01404,1.01464,1.01526,1.0159,1.01655,1.01723,1.01792,1.01862,1.01935,1.02009,1.02084,1.02161,1.02239,1.02318,1.02399,1.02481,1.02564,1.02647,1.02732,1.02817,1.02902,1.02988,1.03074,1.0316,1.03247,1.03332,1.03417,1.03502,1.03586,1.03668,1.03749,1.03829,1.03907,1.03982,1.04056,1.04127,1.04195,1.0426,1.04322,1.0438,1.04434,1.04484,1.0453,1.04571,1.04607,1.04637,1.04662,1.04681,1.04694,1.04699,1.04698,1.0469,1.04674,1.0465,1.04617,1.04576,1.04526,1.04466,1.04397,1.04317,1.04227,1.04126,1.04013,1.03889,1.03752,1.03602,1.03439,1.03263,1.03073,1.02868,1.02648,1.02412,1.02161,1.01893,1.01609,1.01307,1.00988,1.0065,1.00294,0.999192,0.995252,0.991118,0.986787,0.982257,0.977526,0.972595,0.967463,0.96213,0.956598,0.950869,0.944946,0.938834,0.932537,0.926063,0.919418,0.91261,0.90565,0.898547,0.891315,0.883965,0.876511,0.868969,0.861354,0.853682,0.845971,0.838239,0.830503,0.822782,0.815096,0.807462,0.799898,0.792424,0.785056,0.777812,0.770707,0.763757,0.756975,0.750375,0.743968,0.737764,0.731772,0.725999,0.72045,0.715131,0.710044,0.70519,0.700569,0.696181,0.692021,0.688088,0.684376,0.680879,0.677591,0.674506,0.671615,0.668909,0.666382,0.664022,0.661822,0.659771,0.65786,0.65608,0.654422,0.652875,0.65143,0.650078,0.64881,0.647617,0.646489,0.645419,0.644396,0.643413,0.64246,0.641529,0.640611,0.639697,0.638777,0.637843,0.636884,0.635891,0.634852,0.633757,0.632594,0.631349,0.630009,0.62856,0.626986,0.625269,0.623392,0.621334,0.619075,0.61659,0.613855,0.610842,0.607522,0.603864,0.599834,0.595395,0.59051,0.585139,0.579238,0.572764,0.565673,0.557916,0.54945,0.540228,0.530207,0.519346,0.507611,0.494972,0.48141,0.466915,0.451491,0.435159,0.417957,0.399945,0.381207,0.361848},
	{1.01312,1.01371,1.01432,1.01494,1.01559,1.01625,1.01694,1.01764,1.01836,1.0191,1.01986,1.02063,1.02142,1.02223,1.02306,1.0239,1.02475,1.02562,1.0265,1.02739,1.02829,1.0292,1.03012,1.03104,1.03197,1.0329,1.03383,1.03476,1.03569,1.03661,1.03753,1.03844,1.03933,1.04021,1.04108,1.04193,1.04276,1.04356,1.04434,1.04509,1.0458,1.04648,1.04713,1.04773,1.04829,1.0488,1.04926,1.04967,1.05002,1.05031,1.05054,1.0507,1.05079,1.0508,1.05074,1.0506,1.05038,1.05006,1.04966,1.04915,1.04855,1.04785,1.04703,1.04611,1.04506,1.0439,1.04261,1.0412,1.03965,1.03796,1.03613,1.03415,1.03202,1.02973,1.02728,1.02467,1.02189,1.01893,1.01579,1.01247,1.00897,1.00527,1.00138,0.997298,0.993015,0.988531,0.983846,0.978959,0.973869,0.968577,0.963085,0.957395,0.95151,0.945435,0.939173,0.932733,0.926121,0.919346,0.912416,0.905344,0.89814,0.890818,0.883392,0.875875,0.868285,0.860637,0.852949,0.845239,0.837524,0.829823,0.822156,0.81454,0.806994,0.799536,0.792184,0.784954,0.777863,0.770926,0.764157,0.757568,0.751172,0.744978,0.738995,0.73323,0.72769,0.722378,0.717297,0.712449,0.707833,0.703449,0.699293,0.695363,0.691653,0.688159,0.684872,0.681787,0.678896,0.67619,0.673661,0.671299,0.669095,0.667041,0.665126,0.66334,0.661675,0.66012,0.658667,0.657305,0.656026,0.65482,0.653678,0.652591,0.65155,0.650547,0.649572,0.648616,0.64767,0.646724,0.645771,0.644798,0.643797,0.642757,0.641667,0.640514,0.639287,0.637972,0.636554,0.635019,0.63335,0.631529,0.629537,0.627352,0.624953,0.622315,0.619412,0.616215,0.612694,0.608816,0.604545,0.599846,0.594678,0.589,0.582768,0.575938,0.568465,0.560301,0.551401,0.54172,0.531217,0.519852,0.507593,0.494417,0.480305,0.465257,0.449282,0.432409,0.414688,0.396186,0.376998,0.357242},
	{1.01335,1.01395,1.01458,1.01523,1.0159,1.01658,1.01729,1.01802,1.01878,1.01955,1.02034,1.02115,1.02198,1.02283,1.02369,1.02458,1.02548,1.02639,1.02732,1.02827,1.02923,1.03019,1.03117,1.03216,1.03315,1.03415,1.03515,1.03616,1.03716,1.03816,1.03916,1.04015,1.04113,1.0421,1.04306,1.044,1.04492,1.04582,1.04669,1.04754,1.04836,1.04914,1.04989,1.05059,1.05126,1.05187,1.05244,1.05296,1.05341,1.05381,1.05415,1.05442,1.05461,1.05474,1.05478,1.05474,1.05462,1.05441,1.0541,1.0537,1.0532,1.05259,1.05187,1.05103,1.05008,1.04901,1.04781,1.04648,1.04501,1.0434,1.04165,1.03975,1.03769,1.03548,1.0331,1.03055,1.02783,1.02494,1.02187,1.01861,1.01516,1.01152,1.00769,1.00365,0.999422,0.994988,0.990351,0.98551,0.980465,0.975216,0.969766,0.964116,0.95827,0.952232,0.946008,0.939603,0.933024,0.926282,0.919384,0.912342,0.905168,0.897874,0.890475,0.882984,0.875419,0.867795,0.86013,0.852441,0.844748,0.837067,0.829419,0.821821,0.814292,0.80685,0.799513,0.792298,0.785221,0.778297,0.77154,0.764962,0.758577,0.752392,0.746419,0.740663,0.73513,0.729825,0.724751,0.719909,0.715299,0.710919,0.706768,0.702842,0.699136,0.695644,0.69236,0.689277,0.686387,0.683682,0.681153,0.678792,0.676588,0.674532,0.672616,0.670828,0.669161,0.667603,0.666146,0.664779,0.663495,0.662283,0.661134,0.66004,0.658991,0.657978,0.656992,0.656024,0.655064,0.654104,0.653134,0.652143,0.651122,0.650059,0.648944,0.647764,0.646507,0.645158,0.643704,0.642129,0.640415,0.638546,0.6365,0.634257,0.631794,0.629086,0.626106,0.622825,0.619213,0.615236,0.610858,0.606043,0.600749,0.594935,0.588558,0.581572,0.573931,0.565589,0.556501,0.54662,0.535908,0.524324,0.511837,0.498425,0.484072,0.468777,0.452553,0.435431,0.417463,0.398719,0.379297,0.359318},
	{1.01356,1.01418,1.01483,1.01549,1.01618,1.0169,1.01763,1.01839,1.01917,1.01997,1.02079,1.02164,1.0225,1.02339,1.0243,1.02522,1.02617,1.02713,1.02811,1.02911,1.03012,1.03115,1.03219,1.03324,1.03429,1.03536,1.03643,1.03751,1.03859,1.03967,1.04075,1.04182,1.04289,1.04395,1.045,1.04603,1.04705,1.04804,1.04902,1.04996,1.05088,1.05177,1.05262,1.05344,1.05421,1.05493,1.05561,1.05624,1.05681,1.05732,1.05776,1.05814,1.05845,1.05869,1.05884,1.05891,1.0589,1.0588,1.0586,1.0583,1.0579,1.05739,1.05677,1.05604,1.05519,1.05421,1.0531,1.05186,1.05048,1.04896,1.04729,1.04547,1.04349,1.04135,1.03905,1.03658,1.03393,1.0311,1.02809,1.0249,1.02151,1.01793,1.01416,1.01018,1.006,1.00162,0.997034,0.992242,0.987243,0.98204,0.976633,0.971025,0.96522,0.959221,0.953034,0.946665,0.940121,0.933412,0.926547,0.919536,0.912391,0.905125,0.897753,0.890288,0.882748,0.875148,0.867505,0.859838,0.852164,0.844503,0.836873,0.829292,0.821779,0.814353,0.80703,0.799829,0.792764,0.785852,0.779105,0.772538,0.766161,0.759986,0.75402,0.748271,0.742744,0.737445,0.732376,0.727538,0.722931,0.718554,0.714406,0.710481,0.706775,0.703283,0.699999,0.696914,0.694022,0.691314,0.688782,0.686415,0.684206,0.682143,0.680219,0.678423,0.676745,0.675175,0.673705,0.672324,0.671023,0.669793,0.668624,0.667507,0.666432,0.66539,0.664373,0.66337,0.662371,0.661368,0.66035,0.659306,0.658226,0.657099,0.655912,0.654652,0.653307,0.651861,0.6503,0.648606,0.646762,0.644748,0.642544,0.640127,0.637472,0.634554,0.631344,0.627812,0.623924,0.619647,0.614943,0.609772,0.604094,0.597866,0.591042,0.583576,0.575423,0.566536,0.55687,0.546382,0.535033,0.522787,0.509618,0.495509,0.480452,0.464455,0.44754,0.429751,0.411152,0.391828,0.371892,0.351478},
	{1.01375,1.01439,1.01506,1.01574,1.01646,1.01719,1.01795,1.01873,1.01954,1.02037,1.02122,1.0221,1.023,1.02393,1.02487,1.02584,1.02683,1.02784,1.02887,1.02992,1.03098,1.03207,1.03316,1.03428,1.0354,1.03653,1.03768,1.03883,1.03998,1.04114,1.0423,1.04346,1.04461,1.04576,1.04689,1.04802,1.04913,1.05023,1.0513,1.05235,1.05338,1.05437,1.05533,1.05625,1.05714,1.05798,1.05877,1.05951,1.06019,1.06082,1.06138,1.06187,1.0623,1.06265,1.06292,1.06311,1.06321,1.06322,1.06313,1.06295,1.06266,1.06226,1.06174,1.06112,1.06036,1.05949,1.05848,1.05733,1.05605,1.05462,1.05304,1.05131,1.04941,1.04736,1.04514,1.04274,1.04017,1.03741,1.03448,1.03135,1.02803,1.02451,1.0208,1.01688,1.01276,1.00844,1.0039,0.99916,0.994211,0.989055,0.983694,0.97813,0.972367,0.966409,0.960261,0.953929,0.947422,0.940747,0.933915,0.926935,0.919821,0.912584,0.905239,0.897801,0.890285,0.882709,0.875089,0.867443,0.85979,0.852148,0.844536,0.836972,0.829476,0.822064,0.814756,0.807568,0.800515,0.793614,0.786879,0.780321,0.773954,0.767787,0.761828,0.756086,0.750566,0.745273,0.740208,0.735375,0.730772,0.726399,0.722252,0.718329,0.714625,0.711135,0.70785,0.704766,0.701873,0.699164,0.696629,0.69426,0.692047,0.689981,0.688051,0.686249,0.684565,0.682988,0.681509,0.680118,0.678807,0.677564,0.676381,0.675248,0.674157,0.673096,0.672057,0.671031,0.670007,0.668975,0.667925,0.666846,0.665727,0.664556,0.663321,0.662009,0.660606,0.659096,0.657464,0.655693,0.653763,0.651656,0.649349,0.646818,0.64404,0.640986,0.637627,0.633933,0.629869,0.6254,0.620488,0.615092,0.60917,0.602679,0.595574,0.587807,0.579333,0.570105,0.560079,0.549213,0.537468,0.524813,0.511223,0.496683,0.481191,0.46476,0.447417,0.429213,0.410217,0.390525,0.370254,0.349547},
	{1.01393,1.01459,1.01527,1.01598,1.01671,1.01747,1.01825,1.01906,1.01989,1.02075,1.02163,1.02254,1.02348,1.02444,1.02542,1.02643,1.02746,1.02852,1.02959,1.03069,1.03181,1.03295,1.0341,1.03528,1.03646,1.03766,1.03888,1.0401,1.04133,1.04256,1.0438,1.04504,1.04628,1.04752,1.04875,1.04997,1.05118,1.05237,1.05355,1.0547,1.05583,1.05694,1.05801,1.05904,1.06004,1.06099,1.0619,1.06276,1.06356,1.06431,1.06499,1.06561,1.06615,1.06662,1.06701,1.06732,1.06754,1.06767,1.0677,1.06763,1.06746,1.06717,1.06677,1.06625,1.06561,1.06484,1.06394,1.0629,1.06171,1.06038,1.0589,1.05726,1.05545,1.05349,1.05135,1.04904,1.04655,1.04387,1.04101,1.03795,1.0347,1.03126,1.02761,1.02375,1.01969,1.01543,1.01095,1.00626,1.00136,0.996258,0.990945,0.985427,0.979708,0.973792,0.967685,0.961392,0.954922,0.948283,0.941484,0.934537,0.927453,0.920246,0.912929,0.905517,0.898027,0.890474,0.882877,0.875252,0.867619,0.859996,0.852402,0.844855,0.837374,0.829977,0.822682,0.815506,0.808466,0.801575,0.79485,0.788301,0.781942,0.775782,0.769831,0.764095,0.75858,0.753291,0.74823,0.7434,0.738799,0.734428,0.730283,0.726361,0.722657,0.719165,0.715879,0.712792,0.709897,0.707184,0.704644,0.70227,0.700051,0.697977,0.696039,0.694228,0.692532,0.690943,0.689451,0.688045,0.686716,0.685455,0.684251,0.683094,0.681976,0.680887,0.679816,0.678754,0.67769,0.676614,0.675516,0.674383,0.673205,0.671969,0.670662,0.66927,0.667778,0.666171,0.664432,0.662543,0.660483,0.658233,0.655768,0.653066,0.650098,0.646838,0.643253,0.639312,0.634979,0.630218,0.624988,0.619249,0.612958,0.606068,0.598535,0.590312,0.581352,0.57161,0.561042,0.549608,0.537273,0.524009,0.509796,0.494625,0.478501,0.461445,0.443497,0.424716,0.405184,0.385008,0.364319,0.34327},
	{1.0141,1.01478,1.01547,1.0162,1.01695,1.01773,1.01853,1.01936,1.02022,1.02111,1.02202,1.02296,1.02393,1.02492,1.02594,1.02699,1.02806,1.02916,1.03029,1.03143,1.0326,1.0338,1.03501,1.03624,1.03749,1.03876,1.04004,1.04133,1.04263,1.04395,1.04527,1.04659,1.04792,1.04924,1.05056,1.05188,1.05319,1.05448,1.05576,1.05702,1.05826,1.05947,1.06065,1.0618,1.06291,1.06399,1.06501,1.06599,1.06692,1.06778,1.06859,1.06933,1.07,1.0706,1.07112,1.07155,1.07189,1.07215,1.0723,1.07235,1.0723,1.07214,1.07186,1.07145,1.07093,1.07027,1.06948,1.06854,1.06746,1.06624,1.06486,1.06331,1.06161,1.05974,1.05769,1.05547,1.05306,1.05047,1.04769,1.04471,1.04154,1.03816,1.03458,1.03079,1.0268,1.02259,1.01818,1.01355,1.0087,1.00365,0.998389,0.99292,0.987247,0.981376,0.975311,0.969059,0.962627,0.956025,0.949261,0.942348,0.935296,0.928118,0.92083,0.913446,0.905981,0.898452,0.890878,0.883275,0.875662,0.868057,0.860481,0.85295,0.845485,0.838103,0.830821,0.823658,0.816629,0.809749,0.803033,0.796493,0.790142,0.783989,0.778044,0.772313,0.766804,0.761519,0.756462,0.751635,0.747037,0.742667,0.738523,0.734602,0.730898,0.727405,0.724118,0.72103,0.718132,0.715415,0.712872,0.710493,0.708268,0.706188,0.704243,0.702423,0.700718,0.699119,0.697614,0.696195,0.694851,0.693573,0.69235,0.691174,0.690033,0.688918,0.687819,0.686726,0.685628,0.684514,0.683374,0.682195,0.680966,0.679673,0.678304,0.676843,0.675275,0.673585,0.671754,0.669763,0.667593,0.665221,0.662622,0.659773,0.656645,0.65321,0.649434,0.645284,0.640725,0.635718,0.630223,0.624196,0.617595,0.610374,0.602485,0.593883,0.584522,0.574355,0.563341,0.551442,0.538624,0.524862,0.510142,0.494459,0.477824,0.460265,0.441828,0.422583,0.402619,0.382053,0.361026,0.339697},
	{1.01426,1.01495,1.01566,1.0164,1.01717,1.01797,1.01879,1.01965,1.02053,1.02144,1.02238,1.02335,1.02435,1.02538,1.02644,1.02752,1.02864,1.02978,1.03095,1.03214,1.03336,1.03461,1.03588,1.03717,1.03848,1.03981,1.04116,1.04252,1.0439,1.04529,1.04669,1.0481,1.04951,1.05092,1.05234,1.05375,1.05515,1.05655,1.05793,1.0593,1.06064,1.06196,1.06326,1.06453,1.06576,1.06695,1.0681,1.0692,1.07025,1.07125,1.07218,1.07305,1.07385,1.07457,1.07522,1.07578,1.07626,1.07664,1.07693,1.07711,1.07718,1.07714,1.07698,1.07671,1.0763,1.07576,1.07509,1.07427,1.0733,1.07219,1.07091,1.06948,1.06788,1.0661,1.06415,1.06202,1.05971,1.05721,1.05451,1.05162,1.04852,1.04522,1.04172,1.038,1.03408,1.02994,1.02558,1.02101,1.01623,1.01123,1.00603,1.00061,0.994984,0.98916,0.983139,0.97693,0.970539,0.963975,0.957248,0.950369,0.94335,0.936203,0.928945,0.921588,0.914149,0.906645,0.899094,0.891513,0.88392,0.876335,0.868776,0.861262,0.853812,0.846444,0.839176,0.832024,0.825006,0.818136,0.81143,0.804898,0.798555,0.792409,0.78647,0.780745,0.77524,0.769959,0.764905,0.760081,0.755485,0.751117,0.746974,0.743053,0.739349,0.735855,0.732567,0.729477,0.726576,0.723856,0.721309,0.718925,0.716695,0.714608,0.712656,0.710828,0.709114,0.707504,0.705988,0.704556,0.703198,0.701904,0.700663,0.699467,0.698304,0.697165,0.69604,0.694917,0.693786,0.692636,0.691456,0.690233,0.688955,0.687609,0.68618,0.684654,0.683015,0.681246,0.679328,0.677242,0.674967,0.67248,0.669757,0.66677,0.663492,0.659892,0.655938,0.651594,0.646824,0.641588,0.635846,0.629553,0.622666,0.615138,0.606922,0.597972,0.588241,0.577685,0.566264,0.55394,0.540683,0.52647,0.511291,0.495145,0.478049,0.460036,0.441161,0.421499,0.401148,0.380233,0.358901,0.33732},
	{1.01441,1.01511,1.01584,1.0166,1.01738,1.0182,1.01904,1.01992,1.02082,1.02176,1.02273,1.02373,1.02476,1.02582,1.02691,1.02803,1.02918,1.03037,1.03158,1.03282,1.03409,1.03539,1.03671,1.03806,1.03943,1.04082,1.04224,1.04367,1.04512,1.04659,1.04807,1.04956,1.05106,1.05256,1.05407,1.05557,1.05707,1.05857,1.06006,1.06153,1.06299,1.06442,1.06583,1.06722,1.06857,1.06988,1.07116,1.07238,1.07356,1.07469,1.07575,1.07675,1.07768,1.07854,1.07933,1.08002,1.08063,1.08115,1.08157,1.08188,1.08209,1.08218,1.08216,1.08201,1.08173,1.08131,1.08076,1.08007,1.07922,1.07822,1.07706,1.07574,1.07425,1.07258,1.07073,1.0687,1.06649,1.06408,1.06147,1.05867,1.05566,1.05244,1.04902,1.04538,1.04152,1.03745,1.03317,1.02866,1.02394,1.01901,1.01385,1.00849,1.00292,0.997142,0.991169,0.985004,0.978655,0.972131,0.965442,0.958599,0.951614,0.9445,0.937271,0.929943,0.922531,0.915052,0.907524,0.899965,0.892393,0.884827,0.877285,0.869788,0.862353,0.854999,0.847743,0.840604,0.833596,0.826736,0.820038,0.813514,0.807178,0.801038,0.795104,0.789383,0.783882,0.778604,0.773553,0.76873,0.764136,0.759768,0.755625,0.751703,0.747997,0.744502,0.741211,0.738116,0.735211,0.732486,0.729933,0.727542,0.725304,0.723209,0.721247,0.719407,0.717681,0.716056,0.714525,0.713075,0.711698,0.710382,0.709118,0.707895,0.706703,0.705532,0.704371,0.703209,0.702034,0.700836,0.699603,0.698321,0.696979,0.695561,0.694054,0.692441,0.690707,0.688833,0.6868,0.684587,0.682173,0.679533,0.676642,0.673473,0.669995,0.666178,0.661986,0.657384,0.652335,0.646796,0.640726,0.634081,0.626816,0.618882,0.610235,0.600826,0.590611,0.579547,0.567593,0.554717,0.540891,0.526097,0.51033,0.493596,0.47592,0.457343,0.43793,0.417767,0.396963,0.375652,0.353993,0.332161},
	{1.01455,1.01526,1.016,1.01678,1.01758,1.01841,1.01928,1.02017,1.0211,1.02206,1.02305,1.02408,1.02514,1.02623,1.02736,1.02851,1.0297,1.03093,1.03218,1.03347,1.03479,1.03613,1.03751,1.03892,1.04035,1.0418,1.04328,1.04479,1.04631,1.04785,1.04941,1.05098,1.05256,1.05415,1.05575,1.05735,1.05895,1.06055,1.06214,1.06372,1.06529,1.06684,1.06837,1.06987,1.07134,1.07278,1.07418,1.07554,1.07685,1.0781,1.0793,1.08044,1.08151,1.08251,1.08343,1.08426,1.08501,1.08567,1.08623,1.08668,1.08702,1.08725,1.08737,1.08735,1.08721,1.08692,1.0865,1.08594,1.08522,1.08434,1.0833,1.0821,1.08072,1.07916,1.07743,1.0755,1.07339,1.07108,1.06857,1.06586,1.06294,1.05982,1.05647,1.05292,1.04914,1.04515,1.04093,1.0365,1.03184,1.02697,1.02188,1.01657,1.01105,1.00533,0.999405,0.993287,0.986983,0.980502,0.973853,0.967047,0.960098,0.953017,0.94582,0.938522,0.931138,0.923685,0.916181,0.908644,0.901093,0.893547,0.886023,0.878543,0.871123,0.863783,0.85654,0.849412,0.842416,0.835565,0.828876,0.822361,0.816031,0.809897,0.803969,0.798253,0.792756,0.787482,0.782434,0.777613,0.77302,0.768653,0.764511,0.760589,0.756882,0.753386,0.750093,0.746997,0.744089,0.741361,0.738804,0.736408,0.734165,0.732064,0.730095,0.728248,0.726513,0.724879,0.723337,0.721875,0.720485,0.719154,0.717874,0.716634,0.715422,0.714229,0.713044,0.711856,0.710652,0.709422,0.708153,0.706833,0.705447,0.703982,0.702423,0.700754,0.698957,0.697014,0.694905,0.69261,0.690106,0.687367,0.684369,0.681081,0.677476,0.673518,0.669175,0.66441,0.659182,0.653452,0.647175,0.640309,0.632806,0.624619,0.615702,0.606008,0.595492,0.584111,0.571828,0.558608,0.544429,0.529274,0.51314,0.496037,0.477994,0.459057,0.439295,0.418799,0.397683,0.376088,0.354174,0.332123},
	{1.01467,1.0154,1.01616,1.01695,1.01776,1.01861,1.0195,1.02041,1.02136,1.02234,1.02336,1.02441,1.0255,1.02662,1.02778,1.02897,1.0302,1.03146,1.03276,1.03409,1.03545,1.03685,1.03828,1.03974,1.04123,1.04275,1.04429,1.04586,1.04746,1.04907,1.05071,1.05236,1.05403,1.0557,1.05739,1.05909,1.06079,1.06249,1.06418,1.06587,1.06755,1.06922,1.07086,1.07248,1.07408,1.07565,1.07717,1.07866,1.0801,1.0815,1.08283,1.08411,1.08532,1.08646,1.08752,1.0885,1.08939,1.09019,1.09089,1.09149,1.09198,1.09235,1.09261,1.09273,1.09273,1.09258,1.0923,1.09187,1.09128,1.09053,1.08962,1.08854,1.08728,1.08585,1.08423,1.08242,1.08042,1.07821,1.07581,1.0732,1.07037,1.06734,1.06409,1.06062,1.05692,1.05301,1.04887,1.04451,1.03992,1.03512,1.03009,1.02484,1.01938,1.01372,1.00784,1.00177,0.995518,0.98908,0.982473,0.975707,0.968795,0.961749,0.954585,0.947317,0.939961,0.932535,0.925056,0.917542,0.910013,0.902486,0.89498,0.887516,0.880112,0.872786,0.865556,0.858439,0.851453,0.844612,0.83793,0.831422,0.825098,0.81897,0.813046,0.807334,0.80184,0.796568,0.791521,0.786702,0.782109,0.777742,0.773598,0.769674,0.765965,0.762466,0.759169,0.756068,0.753155,0.75042,0.747856,0.745453,0.7432,0.741089,0.739109,0.737249,0.7355,0.733851,0.732291,0.730811,0.729399,0.728045,0.726739,0.72547,0.724227,0.723,0.721776,0.720545,0.719295,0.718013,0.716687,0.715304,0.71385,0.712309,0.710666,0.708905,0.707006,0.704952,0.702722,0.700293,0.697642,0.694743,0.69157,0.688091,0.684276,0.680092,0.675502,0.670468,0.664951,0.658908,0.652295,0.645066,0.637175,0.628576,0.61922,0.609061,0.598057,0.586164,0.573349,0.559581,0.544838,0.529111,0.512403,0.49473,0.476129,0.456655,0.436386,0.415423,0.393891,0.371941,0.34974,0.327478},
	{1.01479,1.01553,1.0163,1.0171,1.01794,1.0188,1.0197,1.02064,1.02161,1.02261,1.02365,1.02473,1.02584,1.02699,1.02818,1.02941,1.03067,1.03197,1.03331,1.03468,1.03609,1.03754,1.03902,1.04053,1.04208,1.04365,1.04526,1.0469,1.04856,1.05025,1.05196,1.0537,1.05545,1.05721,1.05899,1.06078,1.06258,1.06438,1.06618,1.06798,1.06977,1.07155,1.07331,1.07506,1.07678,1.07847,1.08013,1.08175,1.08333,1.08486,1.08634,1.08775,1.08911,1.09039,1.0916,1.09273,1.09377,1.09472,1.09557,1.09631,1.09695,1.09747,1.09787,1.09815,1.09829,1.09829,1.09815,1.09785,1.09741,1.0968,1.09602,1.09507,1.09394,1.09263,1.09114,1.08945,1.08756,1.08547,1.08317,1.08067,1.07795,1.07501,1.07186,1.06848,1.06487,1.06104,1.05699,1.0527,1.04819,1.04345,1.03849,1.03331,1.02792,1.02231,1.01649,1.01047,1.00427,0.997876,0.991313,0.984588,0.977715,0.970706,0.963576,0.95634,0.949014,0.941615,0.934162,0.926672,0.919165,0.911658,0.904172,0.896725,0.889336,0.882024,0.874807,0.867702,0.860726,0.853894,0.847221,0.84072,0.834403,0.82828,0.822361,0.816653,0.811162,0.805892,0.800847,0.796029,0.791436,0.787069,0.782925,0.779,0.775289,0.771787,0.768487,0.765383,0.762465,0.759726,0.757156,0.754746,0.752487,0.750367,0.748378,0.746508,0.744748,0.743086,0.741513,0.740017,0.738589,0.737217,0.73589,0.734599,0.733331,0.732076,0.730822,0.729558,0.728271,0.726949,0.725579,0.724147,0.722639,0.721039,0.719331,0.717498,0.715522,0.713383,0.711059,0.708527,0.705764,0.702743,0.699435,0.695811,0.691838,0.687482,0.682705,0.67747,0.671734,0.665456,0.658591,0.651092,0.642913,0.634007,0.624325,0.613823,0.602457,0.590188,0.57698,0.562808,0.547651,0.531503,0.514371,0.496276,0.47726,0.457384,0.436732,0.41541,0.393551,0.37131,0.348861,0.326397},
	{1.01491,1.01566,1.01644,1.01725,1.0181,1.01898,1.0199,1.02085,1.02184,1.02286,1.02393,1.02503,1.02617,1.02735,1.02856,1.02982,1.03112,1.03246,1.03383,1.03525,1.0367,1.03819,1.03972,1.04129,1.04289,1.04453,1.0462,1.0479,1.04963,1.05139,1.05318,1.05499,1.05683,1.05868,1.06055,1.06243,1.06433,1.06623,1.06814,1.07004,1.07195,1.07384,1.07572,1.07759,1.07944,1.08126,1.08305,1.08481,1.08652,1.08819,1.08981,1.09137,1.09287,1.0943,1.09566,1.09694,1.09814,1.09924,1.10024,1.10114,1.10194,1.10261,1.10317,1.10359,1.10388,1.10404,1.10404,1.1039,1.10359,1.10313,1.10249,1.10168,1.10069,1.09951,1.09814,1.09658,1.09482,1.09285,1.09067,1.08827,1.08566,1.08283,1.07978,1.07649,1.07298,1.06924,1.06527,1.06107,1.05664,1.05198,1.04709,1.04198,1.03665,1.0311,1.02534,1.01938,1.01323,1.00689,1.00037,0.993688,0.986856,0.979885,0.972791,0.965588,0.958294,0.950924,0.943498,0.936033,0.928548,0.921062,0.913595,0.906165,0.898792,0.891495,0.884291,0.877197,0.870232,0.863409,0.856744,0.85025,0.843939,0.837822,0.831907,0.826202,0.820714,0.815446,0.810402,0.805584,0.800992,0.796624,0.792479,0.788551,0.784838,0.781333,0.77803,0.774921,0.771998,0.769253,0.766676,0.764259,0.761991,0.759862,0.757862,0.755981,0.754208,0.752532,0.750944,0.749431,0.747983,0.74659,0.745241,0.743924,0.742628,0.741342,0.740054,0.738752,0.737424,0.736057,0.734637,0.73315,0.731581,0.729915,0.728135,0.726222,0.724159,0.721923,0.719494,0.716848,0.713959,0.710801,0.707343,0.703557,0.699407,0.694858,0.689874,0.684413,0.678436,0.671896,0.664751,0.656952,0.648453,0.639206,0.629165,0.618284,0.60652,0.593836,0.580199,0.565583,0.549975,0.53337,0.515779,0.49723,0.47777,0.457466,0.43641,0.414714,0.392519,0.369984,0.34729,0.324635},
	{1.01501,1.01577,1.01656,1.01739,1.01825,1.01915,1.02008,1.02105,1.02206,1.0231,1.02419,1.02531,1.02647,1.02768,1.02893,1.03022,1.03155,1.03292,1.03433,1.03579,1.03728,1.03882,1.0404,1.04202,1.04367,1.04537,1.0471,1.04886,1.05066,1.0525,1.05436,1.05625,1.05816,1.0601,1.06206,1.06404,1.06603,1.06804,1.07005,1.07206,1.07408,1.07609,1.07809,1.08008,1.08206,1.08401,1.08593,1.08782,1.08968,1.09149,1.09325,1.09496,1.09661,1.0982,1.09971,1.10114,1.10249,1.10375,1.10492,1.10598,1.10693,1.10776,1.10848,1.10906,1.10951,1.10982,1.10998,1.10999,1.10984,1.10952,1.10903,1.10837,1.10752,1.10648,1.10525,1.10382,1.10218,1.10034,1.09828,1.09601,1.09351,1.09079,1.08785,1.08467,1.08126,1.07761,1.07374,1.06962,1.06527,1.06069,1.05588,1.05084,1.04558,1.0401,1.0344,1.0285,1.0224,1.01611,1.00965,1.00301,0.996222,0.989292,0.982235,0.975068,0.967806,0.960467,0.953068,0.945629,0.938168,0.930704,0.923257,0.915845,0.908489,0.901206,0.894015,0.886934,0.879979,0.873166,0.866509,0.860022,0.853717,0.847605,0.841694,0.835993,0.830507,0.825241,0.820199,0.815381,0.810789,0.806421,0.802274,0.798345,0.794629,0.791121,0.787814,0.7847,0.781772,0.779022,0.776439,0.774014,0.771738,0.769601,0.767591,0.765699,0.763914,0.762225,0.760622,0.759093,0.757627,0.756215,0.754844,0.753503,0.752181,0.750866,0.749547,0.74821,0.746844,0.745435,0.743968,0.742431,0.740807,0.73908,0.737232,0.735247,0.733103,0.730779,0.728254,0.725503,0.722499,0.719216,0.715623,0.711689,0.707378,0.702656,0.697484,0.691821,0.685624,0.67885,0.671453,0.663386,0.654601,0.64505,0.634688,0.623469,0.611352,0.598299,0.584281,0.569273,0.553264,0.536254,0.518257,0.499306,0.47945,0.458765,0.437346,0.415312,0.392807,0.369999,0.34707,0.324221},
	{1.0151,1.01587,1.01668,1.01752,1.01839,1.0193,1.02025,1.02123,1.02226,1.02332,1.02443,1.02558,1.02676,1.028,1.02927,1.03059,1.03195,1.03336,1.03481,1.0363,1.03784,1.03942,1.04104,1.04271,1.04442,1.04617,1.04796,1.04979,1.05166,1.05356,1.0555,1.05747,1.05946,1.06149,1.06353,1.0656,1.06769,1.0698,1.07191,1.07404,1.07616,1.07829,1.08041,1.08253,1.08463,1.08671,1.08877,1.0908,1.0928,1.09475,1.09666,1.09852,1.10033,1.10206,1.10373,1.10533,1.10684,1.10826,1.10958,1.11081,1.11192,1.11292,1.1138,1.11455,1.11516,1.11563,1.11596,1.11613,1.11613,1.11597,1.11564,1.11513,1.11443,1.11353,1.11245,1.11116,1.10966,1.10795,1.10602,1.10387,1.1015,1.0989,1.09606,1.09299,1.08969,1.08615,1.08237,1.07835,1.07409,1.06959,1.06486,1.0599,1.05471,1.0493,1.04367,1.03783,1.03179,1.02556,1.01914,1.01256,1.00581,0.998925,0.991908,0.984778,0.97755,0.970243,0.962874,0.955461,0.948025,0.940583,0.933157,0.925764,0.918424,0.911157,0.903979,0.89691,0.889966,0.883162,0.876513,0.870033,0.863734,0.857626,0.85172,0.846021,0.840538,0.835273,0.830231,0.825414,0.82082,0.81645,0.812301,0.80837,0.80465,0.801138,0.797826,0.794707,0.791773,0.789015,0.786424,0.783991,0.781705,0.779556,0.777535,0.775629,0.773829,0.772124,0.770503,0.768954,0.767468,0.766031,0.764634,0.763264,0.761911,0.760561,0.759203,0.757825,0.756412,0.754952,0.75343,0.751831,0.750139,0.748338,0.74641,0.744335,0.742094,0.739664,0.737022,0.734144,0.731002,0.727567,0.72381,0.719696,0.715192,0.710259,0.704859,0.698951,0.69249,0.685432,0.677731,0.66934,0.66021,0.650294,0.639546,0.627922,0.615381,0.601889,0.587417,0.571944,0.555463,0.537976,0.519505,0.500087,0.479778,0.45866,0.436835,0.414431,0.391597,0.368507,0.345348,0.322327},
	{1.01519,1.01597,1.01679,1.01764,1.01852,1.01945,1.02041,1.02141,1.02245,1.02353,1.02466,1.02583,1.02704,1.02829,1.0296,1.03094,1.03233,1.03377,1.03526,1.03679,1.03837,1.03999,1.04166,1.04338,1.04514,1.04695,1.0488,1.05069,1.05262,1.05459,1.0566,1.05864,1.06072,1.06283,1.06496,1.06713,1.06931,1.07151,1.07373,1.07597,1.0782,1.08045,1.08269,1.08493,1.08716,1.08937,1.09157,1.09374,1.09588,1.09798,1.10004,1.10205,1.10401,1.1059,1.10773,1.10949,1.11116,1.11275,1.11424,1.11563,1.11692,1.11809,1.11913,1.12005,1.12084,1.12148,1.12197,1.1223,1.12247,1.12248,1.1223,1.12195,1.12141,1.12067,1.11973,1.11859,1.11723,1.11566,1.11387,1.11186,1.10961,1.10713,1.10442,1.10147,1.09828,1.09484,1.09116,1.08724,1.08308,1.07868,1.07403,1.06915,1.06404,1.0587,1.05314,1.04737,1.04139,1.03522,1.02886,1.02233,1.01563,1.00879,1.00181,0.994723,0.987532,0.980258,0.97292,0.965536,0.958125,0.950708,0.943302,0.935929,0.928607,0.921355,0.914192,0.907135,0.900202,0.893407,0.886767,0.880294,0.874001,0.867898,0.861995,0.8563,0.850818,0.845555,0.840514,0.835696,0.831102,0.826731,0.822579,0.818644,0.814922,0.811405,0.808088,0.804964,0.802023,0.799258,0.796659,0.794217,0.791922,0.789762,0.787729,0.78581,0.783996,0.782275,0.780636,0.779069,0.777561,0.776101,0.774678,0.773281,0.771897,0.770513,0.769118,0.767699,0.766242,0.764732,0.763157,0.761499,0.759742,0.75787,0.755864,0.753704,0.75137,0.748838,0.746086,0.743086,0.739812,0.736233,0.73232,0.728036,0.723348,0.718217,0.712602,0.706462,0.699753,0.692429,0.684443,0.675748,0.666296,0.656039,0.644931,0.63293,0.619996,0.606097,0.591204,0.575301,0.558382,0.540457,0.521547,0.501698,0.480971,0.459452,0.437251,0.414501,0.391358,0.367999,0.344617,0.321419},
	{1.01527,1.01606,1.01689,1.01775,1.01864,1.01958,1.02056,1.02157,1.02263,1.02373,1.02487,1.02606,1.0273,1.02858,1.0299,1.03128,1.0327,1.03417,1.03569,1.03726,1.03887,1.04054,1.04226,1.04402,1.04583,1.04769,1.0496,1.05155,1.05354,1.05558,1.05766,1.05978,1.06194,1.06413,1.06635,1.0686,1.07088,1.07319,1.07551,1.07785,1.0802,1.08256,1.08492,1.08729,1.08965,1.09199,1.09432,1.09663,1.09892,1.10117,1.10338,1.10554,1.10766,1.10971,1.1117,1.11362,1.11547,1.11722,1.11889,1.12045,1.12191,1.12325,1.12447,1.12557,1.12653,1.12734,1.12801,1.12851,1.12886,1.12903,1.12903,1.12884,1.12846,1.12788,1.1271,1.12611,1.12491,1.12348,1.12184,1.11996,1.11785,1.11551,1.11292,1.11009,1.10702,1.1037,1.10013,1.09631,1.09225,1.08794,1.08339,1.0786,1.07357,1.06831,1.06283,1.05712,1.05121,1.0451,1.0388,1.03232,1.02568,1.01888,1.01195,1.0049,0.99775,0.990512,0.983206,0.975852,0.968468,0.961075,0.953693,0.94634,0.939036,0.9318,0.924651,0.917607,0.910685,0.9039,0.897268,0.890802,0.884515,0.878416,0.872517,0.866824,0.861344,0.856082,0.85104,0.846221,0.841625,0.837251,0.833097,0.829158,0.82543,0.821908,0.818585,0.815453,0.812505,0.809731,0.807122,0.804669,0.802361,0.800188,0.79814,0.796205,0.794373,0.792632,0.790972,0.78938,0.787846,0.786358,0.784904,0.783471,0.782049,0.780624,0.779183,0.777713,0.7762,0.77463,0.772987,0.771256,0.769419,0.767459,0.765357,0.763091,0.760641,0.757983,0.755093,0.751942,0.748504,0.744748,0.74064,0.736146,0.73123,0.725852,0.719971,0.713545,0.706528,0.698874,0.690536,0.681466,0.671616,0.660939,0.64939,0.636927,0.623513,0.609116,0.593712,0.577289,0.559844,0.541393,0.521965,0.501609,0.480396,0.45842,0.435798,0.41267,0.389201,0.365574,0.341986,0.318647},
	{1.01535,1.01615,1.01698,1.01785,1.01876,1.01971,1.0207,1.02173,1.0228,1.02392,1.02508,1.02628,1.02754,1.02884,1.03019,1.03159,1.03304,1.03454,1.0361,1.0377,1.03936,1.04106,1.04282,1.04463,1.04649,1.0484,1.05037,1.05238,1.05443,1.05654,1.05869,1.06088,1.06311,1.06539,1.0677,1.07004,1.07241,1.07481,1.07724,1.07969,1.08215,1.08463,1.08711,1.0896,1.09208,1.09456,1.09703,1.09948,1.10191,1.10431,1.10667,1.109,1.11127,1.11349,1.11565,1.11774,1.11975,1.12168,1.12352,1.12526,1.12689,1.12841,1.12982,1.13109,1.13223,1.13323,1.13407,1.13476,1.13528,1.13563,1.1358,1.13578,1.13557,1.13516,1.13455,1.13372,1.13268,1.13141,1.12991,1.12818,1.12622,1.12401,1.12156,1.11886,1.11591,1.11271,1.10926,1.10556,1.1016,1.09739,1.09294,1.08824,1.0833,1.07812,1.07272,1.06709,1.06125,1.0552,1.04896,1.04254,1.03596,1.02921,1.02233,1.01532,1.00821,1.00101,0.99374,0.986418,0.979063,0.971697,0.964337,0.957006,0.949721,0.942503,0.935369,0.928339,0.921428,0.914654,0.90803,0.901571,0.89529,0.889197,0.883301,0.877611,0.872133,0.866871,0.86183,0.85701,0.852412,0.848036,0.843878,0.839936,0.836203,0.832676,0.829347,0.826208,0.823252,0.820469,0.817851,0.815387,0.813068,0.810883,0.80882,0.80687,0.805021,0.803262,0.801581,0.799968,0.798409,0.796894,0.79541,0.793946,0.792488,0.791024,0.78954,0.788024,0.78646,0.784833,0.783129,0.78133,0.77942,0.777378,0.775187,0.772825,0.770269,0.767496,0.764479,0.761192,0.757605,0.753686,0.749402,0.744718,0.739596,0.733995,0.727875,0.721191,0.713898,0.705949,0.697296,0.687892,0.677688,0.666637,0.654696,0.641823,0.627983,0.613146,0.597291,0.580409,0.562502,0.543587,0.5237,0.502896,0.481253,0.45887,0.435869,0.412399,0.388629,0.364746,0.340953,0.31746},
	{1.01542,1.01622,1.01706,1.01794,1.01886,1.01982,1.02082,1.02187,1.02296,1.02409,1.02527,1.02649,1.02777,1.02909,1.03047,1.03189,1.03337,1.0349,1.03649,1.03812,1.03981,1.04156,1.04336,1.04522,1.04713,1.04909,1.0511,1.05317,1.05529,1.05746,1.05968,1.06194,1.06425,1.06661,1.069,1.07143,1.0739,1.0764,1.07893,1.08148,1.08405,1.08665,1.08925,1.09186,1.09448,1.09709,1.0997,1.10229,1.10486,1.10741,1.10993,1.11241,1.11485,1.11723,1.11956,1.12182,1.124,1.12611,1.12813,1.13005,1.13186,1.13357,1.13516,1.13662,1.13794,1.13912,1.14015,1.14103,1.14173,1.14226,1.14262,1.14278,1.14275,1.14251,1.14207,1.14141,1.14053,1.13943,1.13809,1.13652,1.1347,1.13264,1.13033,1.12777,1.12495,1.12188,1.11855,1.11496,1.11112,1.10702,1.10267,1.09807,1.09322,1.08814,1.08282,1.07727,1.0715,1.06553,1.05936,1.053,1.04647,1.03978,1.03295,1.02599,1.01892,1.01176,1.00452,0.997231,0.989907,0.982568,0.975234,0.967925,0.960661,0.95346,0.946343,0.939326,0.932427,0.925663,0.919049,0.912597,0.906322,0.900233,0.894341,0.888653,0.883176,0.877915,0.872873,0.868052,0.863452,0.859072,0.85491,0.850963,0.847225,0.843691,0.840354,0.837207,0.834242,0.831449,0.828819,0.826343,0.824009,0.821808,0.819729,0.817761,0.815892,0.81411,0.812405,0.810764,0.809177,0.807629,0.80611,0.804607,0.803106,0.801595,0.800061,0.798488,0.796862,0.795168,0.793389,0.791509,0.789509,0.787371,0.785073,0.782594,0.779911,0.776999,0.773831,0.770379,0.766612,0.762499,0.758004,0.753091,0.747721,0.741854,0.735446,0.728454,0.72083,0.712528,0.703499,0.693696,0.683069,0.671575,0.659168,0.645811,0.631468,0.616114,0.59973,0.582312,0.563866,0.544417,0.524006,0.502695,0.480569,0.457736,0.434326,0.410496,0.38642,0.362292,0.338319,0.314711},
	{1.01548,1.01629,1.01714,1.01803,1.01896,1.01993,1.02094,1.022,1.0231,1.02425,1.02544,1.02669,1.02798,1.02933,1.03073,1.03218,1.03368,1.03524,1.03685,1.03852,1.04025,1.04204,1.04388,1.04577,1.04773,1.04974,1.05181,1.05394,1.05611,1.05835,1.06063,1.06297,1.06535,1.06779,1.07026,1.07278,1.07534,1.07794,1.08057,1.08323,1.08591,1.08862,1.09134,1.09408,1.09682,1.09957,1.10232,1.10505,1.10777,1.11047,1.11315,1.11579,1.11838,1.12094,1.12343,1.12586,1.12823,1.13051,1.13271,1.13482,1.13682,1.13872,1.14049,1.14214,1.14366,1.14503,1.14626,1.14732,1.14822,1.14894,1.14948,1.14983,1.14998,1.14993,1.14966,1.14918,1.14848,1.14754,1.14637,1.14496,1.1433,1.14139,1.13923,1.13682,1.13414,1.1312,1.128,1.12454,1.12082,1.11684,1.11259,1.1081,1.10335,1.09836,1.09313,1.08766,1.08198,1.07608,1.06997,1.06368,1.05721,1.05058,1.0438,1.03689,1.02987,1.02275,1.01556,1.0083,1.00101,0.993702,0.986394,0.979109,0.971866,0.964685,0.957585,0.950583,0.943697,0.936944,0.930339,0.923896,0.917627,0.911544,0.905655,0.899971,0.894495,0.889235,0.884193,0.87937,0.874768,0.870386,0.86622,0.862268,0.858525,0.854985,0.851641,0.848487,0.845512,0.84271,0.84007,0.837581,0.835235,0.83302,0.830926,0.82894,0.827052,0.825251,0.823523,0.821858,0.820243,0.818666,0.817115,0.815576,0.814037,0.812484,0.810902,0.809278,0.807597,0.805842,0.803996,0.802043,0.799963,0.797737,0.795344,0.792761,0.789964,0.786928,0.783626,0.780027,0.776101,0.771815,0.767134,0.762019,0.756432,0.75033,0.74367,0.736408,0.728495,0.719885,0.710528,0.700378,0.689385,0.677506,0.664697,0.65092,0.636143,0.620343,0.603504,0.585624,0.566715,0.546804,0.52594,0.504188,0.481641,0.458412,0.434637,0.410479,0.386117,0.361749,0.337584,0.313833},
	{1.01554,1.01636,1.01721,1.01811,1.01905,1.02003,1.02105,1.02212,1.02324,1.0244,1.02561,1.02687,1.02818,1.02955,1.03097,1.03244,1.03397,1.03556,1.0372,1.0389,1.04067,1.04249,1.04437,1.04631,1.04831,1.05037,1.05249,1.05467,1.05691,1.0592,1.06155,1.06396,1.06642,1.06893,1.07149,1.07409,1.07674,1.07943,1.08216,1.08493,1.08772,1.09054,1.09339,1.09625,1.09912,1.102,1.10489,1.10777,1.11063,1.11349,1.11632,1.11912,1.12188,1.1246,1.12727,1.12988,1.13242,1.13489,1.13728,1.13957,1.14176,1.14385,1.14582,1.14767,1.14938,1.15095,1.15237,1.15363,1.15472,1.15564,1.15638,1.15692,1.15726,1.1574,1.15732,1.15703,1.1565,1.15574,1.15475,1.15351,1.15201,1.15027,1.14826,1.146,1.14347,1.14067,1.13761,1.13428,1.13069,1.12682,1.1227,1.11831,1.11367,1.10878,1.10364,1.09827,1.09267,1.08685,1.08082,1.0746,1.06819,1.06162,1.0549,1.04805,1.04107,1.034,1.02684,1.01963,1.01237,1.00509,0.997812,0.990552,0.983332,0.976171,0.969088,0.962102,0.95523,0.948489,0.941893,0.935458,0.929196,0.923117,0.917233,0.91155,0.906077,0.900816,0.895773,0.890948,0.886344,0.881957,0.877787,0.873829,0.870079,0.866531,0.863179,0.860014,0.857029,0.854214,0.851561,0.849058,0.846696,0.844463,0.842349,0.840342,0.838431,0.836603,0.834848,0.833152,0.831503,0.82989,0.828298,0.826714,0.825126,0.82352,0.82188,0.820192,0.81844,0.816608,0.814678,0.812633,0.810453,0.808117,0.805603,0.802889,0.799949,0.796757,0.793285,0.789502,0.785376,0.780873,0.775956,0.770587,0.764725,0.758328,0.751352,0.743749,0.735474,0.726477,0.71671,0.706125,0.694676,0.682316,0.669006,0.65471,0.639397,0.623047,0.605649,0.587206,0.567736,0.547271,0.525868,0.5036,0.480567,0.456891,0.432717,0.408213,0.383566,0.358978,0.334662,0.310828},
	{1.01559,1.01641,1.01728,1.01819,1.01913,1.02012,1.02116,1.02224,1.02336,1.02454,1.02577,1.02704,1.02837,1.02976,1.0312,1.03269,1.03425,1.03586,1.03753,1.03927,1.04106,1.04292,1.04483,1.04682,1.04886,1.05097,1.05314,1.05537,1.05766,1.06002,1.06244,1.06491,1.06744,1.07003,1.07267,1.07536,1.0781,1.08089,1.08372,1.08658,1.08949,1.09242,1.09539,1.09837,1.10137,1.10439,1.10741,1.11043,1.11345,1.11645,1.11944,1.1224,1.12534,1.12823,1.13107,1.13386,1.13658,1.13924,1.14181,1.1443,1.14669,1.14897,1.15114,1.15318,1.1551,1.15687,1.15849,1.15996,1.16125,1.16237,1.16331,1.16405,1.1646,1.16493,1.16505,1.16494,1.16461,1.16404,1.16322,1.16216,1.16084,1.15926,1.15742,1.15532,1.15294,1.1503,1.14738,1.14419,1.14073,1.13699,1.13299,1.12872,1.12419,1.1194,1.11437,1.10909,1.10358,1.09784,1.0919,1.08575,1.07942,1.07291,1.06625,1.05945,1.05253,1.0455,1.03839,1.03122,1.024,1.01675,1.0095,1.00227,0.995075,0.987936,0.980873,0.973903,0.967046,0.960318,0.953733,0.947307,0.941052,0.93498,0.9291,0.923421,0.917949,0.91269,0.907646,0.902821,0.898214,0.893824,0.88965,0.885687,0.881931,0.878377,0.875017,0.871844,0.86885,0.866024,0.863359,0.860844,0.858468,0.85622,0.854089,0.852064,0.850133,0.848283,0.846504,0.844782,0.843104,0.841459,0.839833,0.838212,0.836583,0.834931,0.833242,0.8315,0.829689,0.827793,0.825793,0.82367,0.821406,0.818978,0.816365,0.813542,0.810484,0.807163,0.803551,0.799616,0.795326,0.790645,0.785536,0.77996,0.773876,0.76724,0.760007,0.75213,0.743562,0.734254,0.724158,0.713225,0.701409,0.688666,0.674957,0.660245,0.644505,0.627717,0.609874,0.59098,0.571059,0.550149,0.528308,0.505617,0.482181,0.458127,0.433605,0.408789,0.38387,0.359052,0.334549,0.310572},
	{1.01563,1.01647,1.01734,1.01825,1.01921,1.02021,1.02125,1.02234,1.02348,1.02467,1.02591,1.0272,1.02855,1.02995,1.03141,1.03293,1.03451,1.03614,1.03784,1.03961,1.04143,1.04332,1.04528,1.0473,1.04939,1.05154,1.05376,1.05604,1.05839,1.06081,1.06329,1.06583,1.06843,1.07109,1.07381,1.07659,1.07942,1.0823,1.08522,1.0882,1.09121,1.09426,1.09734,1.10044,1.10357,1.10672,1.10988,1.11305,1.11621,1.11937,1.12252,1.12565,1.12874,1.13181,1.13483,1.1378,1.14071,1.14355,1.14631,1.14899,1.15158,1.15407,1.15644,1.15869,1.16081,1.16279,1.16462,1.16629,1.1678,1.16913,1.17027,1.17122,1.17197,1.17251,1.17283,1.17292,1.17278,1.17241,1.17178,1.1709,1.16976,1.16836,1.1667,1.16476,1.16255,1.16006,1.1573,1.15425,1.15093,1.14733,1.14346,1.13931,1.1349,1.13023,1.1253,1.12012,1.11471,1.10906,1.1032,1.09713,1.09087,1.08444,1.07784,1.0711,1.06423,1.05726,1.05019,1.04306,1.03588,1.02867,1.02145,1.01425,1.00708,0.999964,0.992921,0.98597,0.979128,0.972412,0.965839,0.959422,0.953174,0.947107,0.941231,0.935555,0.930084,0.924824,0.919779,0.914951,0.91034,0.905946,0.901765,0.897795,0.894031,0.890467,0.887097,0.883912,0.880904,0.878064,0.875383,0.87285,0.870454,0.868184,0.86603,0.863979,0.862019,0.860139,0.858326,0.856566,0.854848,0.853158,0.851483,0.849809,0.848121,0.846405,0.844645,0.842826,0.840931,0.838942,0.836841,0.834608,0.832224,0.829664,0.826907,0.823928,0.820699,0.817193,0.813379,0.809226,0.804698,0.799761,0.794374,0.788499,0.782092,0.77511,0.767506,0.759232,0.750242,0.740485,0.729914,0.71848,0.706138,0.692845,0.678564,0.663262,0.646916,0.629511,0.611044,0.591527,0.570989,0.549476,0.527055,0.503816,0.479873,0.455361,0.430441,0.405293,0.380116,0.355116,0.330511,0.306509},
	{1.01567,1.01651,1.01739,1.01831,1.01928,1.02028,1.02134,1.02244,1.02359,1.02479,1.02604,1.02735,1.02871,1.03013,1.03161,1.03315,1.03475,1.03641,1.03814,1.03993,1.04179,1.04371,1.0457,1.04776,1.04989,1.05208,1.05435,1.05669,1.05909,1.06156,1.0641,1.06671,1.06938,1.07212,1.07492,1.07778,1.08069,1.08366,1.08669,1.08976,1.09288,1.09604,1.09924,1.10247,1.10573,1.10901,1.11231,1.11562,1.11893,1.12224,1.12555,1.12884,1.13211,1.13534,1.13854,1.14169,1.14479,1.14782,1.15079,1.15366,1.15645,1.15914,1.16172,1.16418,1.16651,1.1687,1.17075,1.17264,1.17436,1.1759,1.17726,1.17843,1.17939,1.18014,1.18067,1.18097,1.18103,1.18085,1.18043,1.17974,1.1788,1.17758,1.1761,1.17433,1.17229,1.16997,1.16737,1.16448,1.1613,1.15785,1.15411,1.1501,1.14581,1.14126,1.13644,1.13137,1.12606,1.12051,1.11473,1.10875,1.10257,1.09621,1.08968,1.08301,1.0762,1.06928,1.06227,1.05518,1.04804,1.04087,1.03369,1.02651,1.01937,1.01228,1.00526,0.998328,0.991502,0.984801,0.97824,0.971834,0.965594,0.959534,0.953663,0.94799,0.942522,0.937263,0.932217,0.927388,0.922774,0.918376,0.91419,0.910214,0.906443,0.902871,0.899491,0.896296,0.893276,0.890424,0.887728,0.885179,0.882766,0.880478,0.878303,0.876229,0.874245,0.872337,0.870494,0.868702,0.866949,0.86522,0.863502,0.861781,0.860042,0.858271,0.85645,0.854565,0.852597,0.850529,0.848341,0.846014,0.843526,0.840855,0.837976,0.834863,0.83149,0.827827,0.823843,0.819505,0.814778,0.809624,0.804006,0.79788,0.791204,0.783934,0.776022,0.76742,0.758079,0.747952,0.736989,0.725143,0.712369,0.698626,0.683877,0.668093,0.651252,0.633342,0.614366,0.594339,0.573295,0.551285,0.528384,0.504688,0.480316,0.455411,0.43014,0.404688,0.379256,0.354056,0.329304,0.305208},
	{1.01571,1.01655,1.01744,1.01837,1.01934,1.02035,1.02142,1.02253,1.02369,1.0249,1.02617,1.02749,1.02887,1.0303,1.0318,1.03336,1.03498,1.03666,1.03842,1.04023,1.04212,1.04408,1.0461,1.0482,1.05036,1.0526,1.05492,1.0573,1.05976,1.06229,1.06489,1.06756,1.0703,1.07311,1.07598,1.07892,1.08192,1.08499,1.08811,1.09128,1.09451,1.09778,1.10109,1.10444,1.10783,1.11124,1.11468,1.11813,1.1216,1.12507,1.12853,1.13199,1.13542,1.13883,1.14221,1.14555,1.14884,1.15206,1.15522,1.1583,1.16129,1.16419,1.16698,1.16965,1.1722,1.17461,1.17687,1.17898,1.18093,1.18269,1.18427,1.18566,1.18684,1.18781,1.18855,1.18907,1.18935,1.18938,1.18915,1.18867,1.18792,1.1869,1.18561,1.18403,1.18217,1.18002,1.17758,1.17485,1.17184,1.16853,1.16494,1.16107,1.15691,1.15248,1.14779,1.14283,1.13762,1.13218,1.1265,1.12061,1.11451,1.10823,1.10177,1.09517,1.08842,1.08156,1.0746,1.06756,1.06047,1.05334,1.04619,1.03905,1.03194,1.02488,1.01788,1.01097,1.00416,0.997473,0.990925,0.984529,0.978298,0.972245,0.966379,0.96071,0.955243,0.949984,0.944938,0.940106,0.935488,0.931085,0.926894,0.92291,0.919131,0.915549,0.912157,0.908949,0.905916,0.903048,0.900335,0.897767,0.895334,0.893023,0.890823,0.888722,0.886708,0.884768,0.882889,0.881058,0.879262,0.877486,0.875717,0.87394,0.872141,0.870302,0.868408,0.866443,0.864388,0.862224,0.859932,0.857492,0.85488,0.852073,0.849047,0.845774,0.842227,0.838375,0.834186,0.829626,0.824659,0.819247,0.813349,0.806923,0.799925,0.792309,0.784028,0.775033,0.765275,0.754705,0.743276,0.73094,0.717653,0.703377,0.688076,0.671723,0.654301,0.635803,0.616235,0.595619,0.573994,0.551422,0.527982,0.503779,0.478942,0.45362,0.427987,0.402236,0.376572,0.35121,0.326363,0.30224},
	{1.01574,1.01659,1.01748,1.01842,1.01939,1.02042,1.02149,1.02261,1.02378,1.025,1.02628,1.02762,1.02901,1.03046,1.03198,1.03355,1.03519,1.0369,1.03868,1.04052,1.04243,1.04442,1.04648,1.04861,1.05082,1.0531,1.05546,1.05789,1.0604,1.06298,1.06564,1.06837,1.07118,1.07406,1.07701,1.08003,1.08312,1.08627,1.08948,1.09276,1.09609,1.09947,1.1029,1.10637,1.10988,1.11343,1.117,1.1206,1.12421,1.12784,1.13146,1.13508,1.13869,1.14228,1.14584,1.14936,1.15284,1.15626,1.15962,1.1629,1.1661,1.16921,1.17222,1.17511,1.17787,1.18051,1.183,1.18533,1.1875,1.18949,1.1913,1.19291,1.19432,1.19552,1.19649,1.19722,1.19772,1.19797,1.19796,1.19769,1.19715,1.19633,1.19523,1.19384,1.19217,1.1902,1.18794,1.18539,1.18254,1.17939,1.17595,1.17222,1.16821,1.16391,1.15934,1.1545,1.14941,1.14407,1.1385,1.1327,1.1267,1.1205,1.11412,1.10758,1.10091,1.09411,1.0872,1.08022,1.07317,1.06609,1.05898,1.05188,1.0448,1.03776,1.03079,1.0239,1.01711,1.01044,1.00391,0.997525,0.991305,0.985259,0.979399,0.973734,0.96827,0.963012,0.957965,0.953131,0.948511,0.944103,0.939906,0.935915,0.932127,0.928536,0.925134,0.921913,0.918866,0.915983,0.913253,0.910667,0.908213,0.90588,0.903656,0.901528,0.899485,0.897512,0.895598,0.893729,0.891891,0.89007,0.888251,0.886419,0.884559,0.882655,0.88069,0.878647,0.876507,0.87425,0.871857,0.869306,0.866574,0.863637,0.860468,0.857041,0.853326,0.849292,0.844906,0.840133,0.834936,0.829276,0.823111,0.816399,0.809094,0.801149,0.792517,0.78315,0.772997,0.762009,0.75014,0.737343,0.723575,0.708797,0.692979,0.676095,0.658132,0.639085,0.618966,0.597802,0.575639,0.552544,0.528603,0.50393,0.478658,0.452945,0.426971,0.400933,0.375041,0.34951,0.324554,0.300378},
	{1.01577,1.01662,1.01752,1.01846,1.01944,1.02047,1.02155,1.02268,1.02386,1.0251,1.02639,1.02773,1.02914,1.03061,1.03214,1.03373,1.03539,1.03712,1.03892,1.04079,1.04273,1.04475,1.04684,1.049,1.05125,1.05357,1.05597,1.05845,1.06101,1.06364,1.06636,1.06915,1.07202,1.07497,1.078,1.0811,1.08427,1.08751,1.09081,1.09419,1.09762,1.10111,1.10465,1.10825,1.11189,1.11556,1.11927,1.12301,1.12678,1.13055,1.13434,1.13812,1.1419,1.14567,1.14941,1.15312,1.15679,1.16041,1.16397,1.16746,1.17088,1.1742,1.17742,1.18053,1.18353,1.18639,1.1891,1.19167,1.19407,1.1963,1.19834,1.20019,1.20183,1.20326,1.20446,1.20543,1.20615,1.20662,1.20684,1.20678,1.20646,1.20585,1.20496,1.20377,1.20229,1.20052,1.19844,1.19607,1.19339,1.19041,1.18713,1.18355,1.17969,1.17553,1.17109,1.16639,1.16141,1.15619,1.15072,1.14503,1.13912,1.13301,1.12671,1.12025,1.11365,1.10691,1.10007,1.09314,1.08615,1.07911,1.07204,1.06498,1.05793,1.05093,1.04398,1.03711,1.03035,1.0237,1.01718,1.01081,1.0046,0.998558,0.992703,0.987042,0.981579,0.976322,0.971273,0.966436,0.96181,0.957396,0.953191,0.949191,0.945392,0.941788,0.938372,0.935135,0.93207,0.929167,0.926416,0.923806,0.921326,0.918963,0.916707,0.914545,0.912463,0.91045,0.90849,0.906571,0.904678,0.902797,0.900913,0.89901,0.897072,0.895083,0.893024,0.890879,0.888628,0.88625,0.883724,0.881029,0.878139,0.875031,0.871676,0.868047,0.864112,0.85984,0.855197,0.850145,0.844647,0.838662,0.832149,0.825062,0.817355,0.808982,0.799894,0.79004,0.779373,0.767844,0.755404,0.74201,0.727621,0.7122,0.695719,0.678158,0.659507,0.639768,0.61896,0.597119,0.574297,0.550571,0.526037,0.500819,0.475061,0.44893,0.422613,0.396314,0.370247,0.344628,0.31967,0.295569},
	{1.01579,1.01665,1.01755,1.01849,1.01949,1.02052,1.02161,1.02275,1.02394,1.02518,1.02648,1.02784,1.02926,1.03074,1.03229,1.0339,1.03558,1.03733,1.03915,1.04104,1.04301,1.04505,1.04718,1.04938,1.05165,1.05401,1.05646,1.05898,1.06159,1.06428,1.06705,1.0699,1.07284,1.07585,1.07895,1.08213,1.08538,1.0887,1.0921,1.09557,1.09911,1.10271,1.10636,1.11008,1.11384,1.11765,1.1215,1.12538,1.12929,1.13322,1.13716,1.14112,1.14507,1.14901,1.15294,1.15684,1.1607,1.16452,1.16829,1.17199,1.17562,1.17916,1.1826,1.18594,1.18916,1.19225,1.19521,1.19801,1.20065,1.20311,1.2054,1.20748,1.20937,1.21103,1.21248,1.21368,1.21464,1.21535,1.21579,1.21597,1.21586,1.21547,1.2148,1.21382,1.21255,1.21098,1.20909,1.2069,1.20441,1.2016,1.19849,1.19508,1.19137,1.18736,1.18307,1.1785,1.17365,1.16855,1.1632,1.15761,1.1518,1.14578,1.13958,1.1332,1.12667,1.12001,1.11324,1.10637,1.09943,1.09244,1.08542,1.0784,1.07139,1.06442,1.0575,1.05066,1.04392,1.03729,1.03079,1.02443,1.01824,1.01221,1.00636,1.00071,0.995252,0.989998,0.984952,0.980115,0.975489,0.971073,0.966864,0.96286,0.959055,0.955444,0.95202,0.948774,0.945699,0.942784,0.94002,0.937395,0.934899,0.93252,0.930245,0.928061,0.925957,0.923919,0.921932,0.919983,0.918058,0.916142,0.914219,0.912274,0.910291,0.908252,0.90614,0.903936,0.901621,0.899174,0.896573,0.893796,0.890818,0.887613,0.884155,0.880413,0.876357,0.871954,0.867169,0.861966,0.856305,0.850145,0.843444,0.836156,0.828236,0.819635,0.810304,0.800195,0.789257,0.777443,0.764704,0.750997,0.736282,0.720523,0.703693,0.685774,0.666755,0.646643,0.625458,0.603238,0.580038,0.555938,0.531037,0.505461,0.479358,0.452897,0.426269,0.399679,0.37334,0.347473,0.322288,0.297981},
	{1.0158,1.01667,1.01758,1.01853,1.01952,1.02057,1.02166,1.0228,1.024,1.02526,1.02657,1.02794,1.02937,1.03087,1.03243,1.03405,1.03575,1.03752,1.03936,1.04128,1.04327,1.04534,1.04749,1.04972,1.05204,1.05444,1.05692,1.05949,1.06214,1.06488,1.0677,1.07062,1.07361,1.0767,1.07987,1.08312,1.08645,1.08986,1.09335,1.09691,1.10055,1.10425,1.10802,1.11185,1.11574,1.11968,1.12366,1.12769,1.13174,1.13583,1.13994,1.14406,1.14818,1.1523,1.15641,1.1605,1.16456,1.16858,1.17256,1.17647,1.18031,1.18407,1.18774,1.19131,1.19477,1.19809,1.20129,1.20433,1.20721,1.20993,1.21245,1.21479,1.21692,1.21883,1.22052,1.22197,1.22317,1.22412,1.22481,1.22522,1.22534,1.22518,1.22473,1.22398,1.22292,1.22155,1.21987,1.21788,1.21557,1.21295,1.21002,1.20678,1.20323,1.19938,1.19523,1.1908,1.1861,1.18112,1.17589,1.17041,1.16471,1.1588,1.15269,1.1464,1.13995,1.13337,1.12666,1.11986,1.11297,1.10604,1.09907,1.09209,1.08512,1.07819,1.0713,1.06449,1.05777,1.05116,1.04468,1.03834,1.03216,1.02614,1.0203,1.01465,1.0092,1.00395,0.998901,0.994062,0.989432,0.98501,0.980794,0.976781,0.972966,0.969343,0.965905,0.962644,0.959551,0.956617,0.953831,0.951183,0.94866,0.946252,0.943946,0.941728,0.939586,0.937506,0.935474,0.933475,0.931496,0.929521,0.927534,0.925518,0.923458,0.921335,0.919131,0.916827,0.914403,0.911837,0.909107,0.90619,0.903059,0.899688,0.896049,0.892112,0.887845,0.883213,0.878181,0.872711,0.866762,0.860294,0.853261,0.845619,0.837319,0.828315,0.818555,0.807992,0.796574,0.784255,0.770987,0.756729,0.74144,0.72509,0.707653,0.689114,0.66947,0.64873,0.626921,0.604087,0.580293,0.555624,0.530189,0.504123,0.47758,0.450737,0.423791,0.396953,0.370438,0.344465,0.319243,0.294962},
	{1.01582,1.01669,1.0176,1.01855,1.01955,1.0206,1.0217,1.02286,1.02406,1.02532,1.02664,1.02803,1.02947,1.03098,1.03255,1.0342,1.03591,1.0377,1.03956,1.0415,1.04351,1.04561,1.04779,1.05005,1.0524,1.05484,1.05736,1.05997,1.06267,1.06545,1.06833,1.0713,1.07436,1.07751,1.08074,1.08407,1.08748,1.09097,1.09455,1.09821,1.10195,1.10575,1.10964,1.11358,1.11759,1.12166,1.12578,1.12994,1.13415,1.13839,1.14265,1.14694,1.15124,1.15554,1.15983,1.16412,1.16837,1.1726,1.17678,1.18091,1.18497,1.18895,1.19285,1.19665,1.20034,1.20391,1.20735,1.21064,1.21377,1.21673,1.21952,1.22211,1.22449,1.22666,1.2286,1.2303,1.23176,1.23295,1.23388,1.23454,1.23491,1.23499,1.23476,1.23424,1.2334,1.23225,1.23079,1.229,1.22689,1.22447,1.22172,1.21866,1.21528,1.2116,1.20761,1.20333,1.19876,1.19392,1.18882,1.18347,1.17788,1.17208,1.16607,1.15987,1.15351,1.147,1.14037,1.13364,1.12682,1.11994,1.11303,1.1061,1.09917,1.09227,1.08543,1.07865,1.07195,1.06537,1.05891,1.05258,1.04641,1.04041,1.03458,1.02894,1.02349,1.01824,1.0132,1.00836,1.00373,0.999304,0.995083,0.991064,0.98724,0.983608,0.980158,0.976884,0.973777,0.970827,0.968023,0.965354,0.96281,0.960377,0.958044,0.955797,0.953623,0.951508,0.949438,0.947398,0.945373,0.943348,0.941306,0.939232,0.937107,0.934914,0.932634,0.930246,0.927732,0.925067,0.922231,0.919197,0.91594,0.912433,0.908646,0.904549,0.900109,0.89529,0.890057,0.884371,0.87819,0.871472,0.864173,0.856245,0.847642,0.838315,0.828213,0.817287,0.805488,0.792768,0.779081,0.764385,0.748644,0.731826,0.713909,0.69488,0.674739,0.653499,0.631192,0.607865,0.583588,0.558453,0.532573,0.506088,0.479157,0.451964,0.424708,0.397601,0.370863,0.344711,0.319352,0.294973},
	{1.01583,1.0167,1.01761,1.01857,1.01958,1.02064,1.02174,1.0229,1.02411,1.02538,1.02671,1.0281,1.02956,1.03108,1.03267,1.03433,1.03606,1.03786,1.03974,1.0417,1.04374,1.04586,1.04807,1.05036,1.05274,1.05521,1.05777,1.06042,1.06316,1.066,1.06893,1.07195,1.07507,1.07828,1.08158,1.08498,1.08847,1.09205,1.09571,1.09946,1.10329,1.10721,1.1112,1.11526,1.11939,1.12359,1.12784,1.13214,1.1365,1.14089,1.14532,1.14977,1.15424,1.15872,1.1632,1.16767,1.17213,1.17656,1.18095,1.18529,1.18958,1.19379,1.19792,1.20196,1.20589,1.2097,1.21339,1.21693,1.22031,1.22353,1.22657,1.22942,1.23207,1.2345,1.2367,1.23866,1.24038,1.24183,1.24302,1.24392,1.24454,1.24487,1.24489,1.2446,1.244,1.24307,1.24183,1.24026,1.23836,1.23613,1.23358,1.23071,1.22751,1.224,1.22018,1.21606,1.21164,1.20695,1.20198,1.19676,1.19129,1.1856,1.17969,1.1736,1.16733,1.16091,1.15436,1.1477,1.14095,1.13413,1.12727,1.12039,1.11351,1.10666,1.09984,1.0931,1.08643,1.07987,1.07343,1.06713,1.06098,1.05498,1.04917,1.04353,1.03809,1.03284,1.0278,1.02296,1.01832,1.01389,1.00967,1.00564,1.0018,0.998156,0.994692,0.991401,0.988275,0.985303,0.982475,0.979781,0.977208,0.974743,0.972375,0.97009,0.967874,0.965713,0.963593,0.961498,0.959414,0.957323,0.95521,0.953058,0.950848,0.948563,0.946182,0.943685,0.941051,0.938257,0.935279,0.932092,0.928668,0.924981,0.920999,0.91669,0.912021,0.906956,0.901456,0.895483,0.888995,0.881947,0.874294,0.86599,0.856985,0.84723,0.836675,0.825271,0.812969,0.799721,0.785483,0.770214,0.75388,0.736452,0.717912,0.698251,0.677473,0.655598,0.632663,0.608723,0.583856,0.558159,0.531757,0.504797,0.477444,0.44989,0.42234,0.395008,0.368116,0.341879,0.3165,0.29216},
	{1.01583,1.01671,1.01763,1.01859,1.0196,1.02066,1.02177,1.02294,1.02416,1.02544,1.02677,1.02817,1.02964,1.03117,1.03277,1.03444,1.03619,1.03801,1.03991,1.04189,1.04395,1.0461,1.04833,1.05065,1.05306,1.05557,1.05816,1.06085,1.06364,1.06652,1.0695,1.07257,1.07575,1.07902,1.08239,1.08586,1.08942,1.09308,1.09683,1.10067,1.1046,1.10861,1.11271,1.11689,1.12114,1.12546,1.12985,1.13429,1.13879,1.14334,1.14792,1.15254,1.15718,1.16184,1.16651,1.17118,1.17583,1.18047,1.18507,1.18963,1.19414,1.19858,1.20295,1.20722,1.2114,1.21546,1.2194,1.2232,1.22684,1.23032,1.23363,1.23674,1.23965,1.24235,1.24482,1.24705,1.24904,1.25076,1.25221,1.25337,1.25425,1.25483,1.2551,1.25506,1.2547,1.25401,1.253,1.25165,1.24997,1.24796,1.24561,1.24294,1.23993,1.2366,1.23296,1.229,1.22475,1.2202,1.21538,1.21029,1.20495,1.19938,1.19359,1.1876,1.18143,1.17511,1.16864,1.16206,1.15538,1.14863,1.14183,1.13501,1.12818,1.12137,1.11459,1.10788,1.10125,1.09472,1.08831,1.08202,1.07589,1.06991,1.06411,1.05848,1.05305,1.04781,1.04276,1.03792,1.03329,1.02885,1.02462,1.02059,1.01674,1.01309,1.00961,1.00631,1.00316,1.00017,0.997327,0.99461,0.992013,0.989522,0.987124,0.984806,0.982555,0.980355,0.978192,0.976051,0.973916,0.97177,0.969598,0.96738,0.965099,0.962736,0.96027,0.957682,0.954948,0.952046,0.94895,0.945636,0.942075,0.938238,0.934095,0.929612,0.924755,0.919488,0.913771,0.907565,0.900826,0.893511,0.885572,0.876963,0.867634,0.857536,0.846618,0.834831,0.822126,0.808457,0.793779,0.778054,0.761249,0.743336,0.724299,0.704134,0.682847,0.660462,0.637021,0.612582,0.587229,0.561066,0.534221,0.506845,0.479112,0.451215,0.423362,0.395771,0.368665,0.342257,0.316747,0.292313},
	{1.01583,1.01671,1.01763,1.0186,1.01962,1.02068,1.0218,1.02297,1.02419,1.02548,1.02683,1.02824,1.02971,1.03125,1.03287,1.03455,1.03631,1.03815,1.04007,1.04206,1.04415,1.04632,1.04857,1.05092,1.05336,1.0559,1.05853,1.06126,1.06408,1.06701,1.07004,1.07316,1.07639,1.07973,1.08316,1.0867,1.09033,1.09407,1.0979,1.10183,1.10586,1.10997,1.11418,1.11847,1.12284,1.12728,1.1318,1.13638,1.14103,1.14573,1.15047,1.15526,1.16007,1.16491,1.16977,1.17463,1.17948,1.18432,1.18914,1.19392,1.19865,1.20333,1.20793,1.21245,1.21687,1.22119,1.22538,1.22944,1.23335,1.2371,1.24067,1.24406,1.24725,1.25022,1.25296,1.25547,1.25773,1.25972,1.26145,1.26288,1.26403,1.26487,1.2654,1.26562,1.26551,1.26507,1.2643,1.26318,1.26173,1.25994,1.25781,1.25534,1.25253,1.2494,1.24594,1.24216,1.23807,1.23369,1.22901,1.22407,1.21887,1.21342,1.20775,1.20188,1.19581,1.18958,1.18321,1.17671,1.17011,1.16343,1.1567,1.14993,1.14316,1.1364,1.12967,1.123,1.1164,1.1099,1.10351,1.09725,1.09114,1.08518,1.07939,1.07377,1.06835,1.06311,1.05807,1.05324,1.0486,1.04416,1.03993,1.03589,1.03203,1.02837,1.02488,1.02156,1.0184,1.0154,1.01253,1.00979,1.00717,1.00465,1.00222,0.99987,0.997582,0.995343,0.993137,0.990949,0.988762,0.986561,0.984327,0.982042,0.979689,0.977247,0.974696,0.972015,0.96918,0.966168,0.962954,0.959511,0.955811,0.951824,0.947518,0.942861,0.937815,0.932345,0.926411,0.91997,0.912981,0.905398,0.897175,0.888262,0.87861,0.868171,0.856893,0.844726,0.831623,0.817537,0.802426,0.78625,0.76898,0.750589,0.731065,0.710404,0.688618,0.665733,0.641794,0.616866,0.591037,0.564415,0.537134,0.509349,0.481239,0.453,0.424844,0.39699,0.369661,0.343071,0.317417,0.292871},
	{1.01583,1.01671,1.01763,1.01861,1.01962,1.02069,1.02182,1.02299,1.02422,1.02552,1.02687,1.02829,1.02977,1.03132,1.03295,1.03465,1.03642,1.03827,1.04021,1.04222,1.04433,1.04652,1.0488,1.05117,1.05364,1.05621,1.05887,1.06164,1.0645,1.06747,1.07055,1.07372,1.07701,1.0804,1.0839,1.0875,1.09121,1.09502,1.09894,1.10295,1.10707,1.11129,1.11559,1.11999,1.12448,1.12905,1.1337,1.13842,1.14321,1.14806,1.15296,1.15791,1.1629,1.16792,1.17296,1.17801,1.18307,1.18812,1.19315,1.19815,1.20311,1.20802,1.21286,1.21763,1.2223,1.22688,1.23133,1.23565,1.23983,1.24386,1.24771,1.25137,1.25484,1.25809,1.26112,1.26391,1.26645,1.26873,1.27073,1.27245,1.27387,1.27498,1.27579,1.27627,1.27642,1.27623,1.27571,1.27484,1.27363,1.27207,1.27016,1.26791,1.26532,1.26238,1.25911,1.25552,1.25161,1.24739,1.24288,1.23809,1.23303,1.22772,1.22218,1.21642,1.21047,1.20435,1.19807,1.19166,1.18514,1.17854,1.17188,1.16518,1.15846,1.15175,1.14507,1.13844,1.13188,1.12542,1.11906,1.11283,1.10673,1.10079,1.09502,1.08942,1.084,1.07877,1.07374,1.0689,1.06427,1.05983,1.05559,1.05154,1.04768,1.04401,1.04051,1.03718,1.034,1.03098,1.02809,1.02533,1.02268,1.02013,1.01768,1.01529,1.01297,1.01069,1.00844,1.0062,1.00396,1.0017,0.999409,0.997058,0.994631,0.99211,0.989473,0.986697,0.983761,0.980639,0.977306,0.973734,0.969895,0.965757,0.961289,0.956456,0.951223,0.94555,0.939399,0.932726,0.925488,0.91764,0.909133,0.89992,0.88995,0.879173,0.867539,0.854999,0.841504,0.827008,0.811471,0.794854,0.777128,0.75827,0.738268,0.717123,0.694848,0.671474,0.64705,0.621644,0.595349,0.568276,0.540566,0.512377,0.483892,0.45531,0.426848,0.398724,0.371162,0.344375,0.318558,0.29388},
	{1.01582,1.01671,1.01763,1.01861,1.01963,1.0207,1.02183,1.02301,1.02425,1.02555,1.02691,1.02833,1.02982,1.03139,1.03302,1.03473,1.03652,1.03838,1.04033,1.04237,1.04449,1.0467,1.049,1.0514,1.0539,1.0565,1.05919,1.06199,1.0649,1.06791,1.07103,1.07425,1.07759,1.08104,1.08459,1.08826,1.09204,1.09593,1.09993,1.10403,1.10824,1.11255,1.11696,1.12147,1.12607,1.13077,1.13554,1.1404,1.14533,1.15033,1.15539,1.16051,1.16567,1.17087,1.17609,1.18134,1.1866,1.19186,1.1971,1.20233,1.20752,1.21266,1.21775,1.22276,1.22769,1.23252,1.23724,1.24183,1.24629,1.25059,1.25472,1.25867,1.26242,1.26596,1.26928,1.27236,1.27519,1.27776,1.28005,1.28205,1.28376,1.28516,1.28624,1.287,1.28742,1.2875,1.28724,1.28663,1.28566,1.28434,1.28267,1.28064,1.27826,1.27554,1.27248,1.26908,1.26535,1.26132,1.25697,1.25234,1.24743,1.24227,1.23686,1.23123,1.2254,1.21938,1.21321,1.20689,1.20046,1.19394,1.18735,1.18072,1.17407,1.16741,1.16078,1.1542,1.14768,1.14125,1.13493,1.12872,1.12265,1.11673,1.11097,1.10538,1.09997,1.09475,1.08973,1.08489,1.08026,1.07581,1.07157,1.06751,1.06364,1.05995,1.05644,1.05309,1.04989,1.04685,1.04393,1.04114,1.03846,1.03588,1.03338,1.03096,1.02859,1.02626,1.02395,1.02165,1.01935,1.01702,1.01464,1.0122,1.00968,1.00706,1.00431,1.00141,0.998346,0.995083,0.991595,0.987857,0.983837,0.979505,0.974827,0.969768,0.964291,0.958356,0.951923,0.944949,0.937389,0.929197,0.920324,0.910721,0.900338,0.889125,0.877031,0.864007,0.850006,0.834983,0.818898,0.801715,0.783406,0.763953,0.743347,0.721592,0.698706,0.674727,0.649708,0.623724,0.596875,0.56928,0.541085,0.512455,0.48358,0.454663,0.425923,0.397582,0.369864,0.342977,0.317113,0.292433},
	{1.01581,1.0167,1.01763,1.0186,1.01963,1.02071,1.02184,1.02302,1.02426,1.02557,1.02694,1.02837,1.02987,1.03144,1.03308,1.0348,1.0366,1.03848,1.04045,1.0425,1.04464,1.04687,1.04919,1.05162,1.05414,1.05676,1.05949,1.06232,1.06527,1.06832,1.07148,1.07475,1.07814,1.08164,1.08526,1.08899,1.09284,1.0968,1.10087,1.10506,1.10936,1.11377,1.11828,1.1229,1.12761,1.13243,1.13733,1.14232,1.1474,1.15255,1.15776,1.16304,1.16837,1.17375,1.17916,1.18461,1.19007,1.19553,1.201,1.20644,1.21187,1.21725,1.22258,1.22784,1.23303,1.23812,1.24311,1.24798,1.25271,1.25729,1.26171,1.26595,1.26999,1.27383,1.27745,1.28082,1.28395,1.28682,1.28941,1.29171,1.29371,1.2954,1.29677,1.29781,1.29851,1.29887,1.29888,1.29853,1.29783,1.29676,1.29533,1.29354,1.2914,1.28889,1.28604,1.28285,1.27932,1.27547,1.2713,1.26684,1.2621,1.25708,1.25182,1.24632,1.24061,1.23472,1.22865,1.22244,1.2161,1.20967,1.20316,1.1966,1.19002,1.18342,1.17685,1.17032,1.16384,1.15745,1.15116,1.14498,1.13894,1.13304,1.1273,1.12173,1.11634,1.11113,1.10611,1.10128,1.09664,1.0922,1.08795,1.08389,1.08001,1.07631,1.07279,1.06942,1.06621,1.06314,1.06021,1.05739,1.05469,1.05208,1.04954,1.04708,1.04467,1.0423,1.03994,1.03759,1.03523,1.03284,1.03039,1.02788,1.02528,1.02257,1.01973,1.01673,1.01355,1.01017,1.00655,1.00267,0.998505,0.99401,0.989157,0.983909,0.97823,0.972079,0.965414,0.958192,0.950367,0.941893,0.93272,0.922799,0.91208,0.900512,0.888045,0.87463,0.860219,0.84477,0.828242,0.810602,0.791824,0.771889,0.750793,0.728542,0.705158,0.680681,0.655169,0.628703,0.601383,0.573336,0.54471,0.515676,0.486426,0.457168,0.428122,0.399512,0.37156,0.344474,0.318444,0.293626},
	{1.0158,1.01668,1.01762,1.0186,1.01962,1.0207,1.02184,1.02303,1.02427,1.02558,1.02696,1.0284,1.0299,1.03148,1.03314,1.03487,1.03668,1.03857,1.04055,1.04261,1.04477,1.04702,1.04937,1.05181,1.05436,1.05701,1.05977,1.06263,1.06561,1.0687,1.0719,1.07522,1.07866,1.08221,1.08589,1.08968,1.09359,1.09763,1.10178,1.10605,1.11043,1.11493,1.11955,1.12427,1.1291,1.13403,1.13906,1.14419,1.1494,1.1547,1.16007,1.16551,1.17101,1.17657,1.18217,1.18781,1.19347,1.19915,1.20483,1.2105,1.21615,1.22177,1.22735,1.23287,1.23831,1.24367,1.24893,1.25408,1.25909,1.26396,1.26867,1.27321,1.27755,1.28169,1.28561,1.28929,1.29272,1.29589,1.29879,1.30139,1.3037,1.30569,1.30736,1.30869,1.30969,1.31033,1.31062,1.31055,1.31011,1.30931,1.30813,1.30659,1.30468,1.30241,1.29978,1.2968,1.29348,1.28982,1.28585,1.28156,1.27698,1.27213,1.26702,1.26166,1.25609,1.25031,1.24436,1.23826,1.23202,1.22568,1.21926,1.21278,1.20626,1.19973,1.19322,1.18674,1.18031,1.17396,1.1677,1.16156,1.15554,1.14967,1.14395,1.13839,1.13301,1.1278,1.12279,1.11796,1.11332,1.10888,1.10462,1.10055,1.09666,1.09294,1.0894,1.08601,1.08277,1.07967,1.07671,1.07386,1.07111,1.06845,1.06587,1.06336,1.06089,1.05845,1.05602,1.0536,1.05115,1.04866,1.04612,1.0435,1.04078,1.03794,1.03496,1.03181,1.02847,1.02491,1.0211,1.01701,1.01261,1.00788,1.00276,0.99723,0.991247,0.98477,0.977755,0.970159,0.961935,0.953035,0.94341,0.93301,0.921783,0.90968,0.89665,0.882645,0.86762,0.851531,0.834341,0.81602,0.796544,0.775901,0.754089,0.731121,0.707025,0.681847,0.655656,0.628537,0.600603,0.571986,0.542845,0.513359,0.483726,0.454158,0.424881,0.396118,0.368089,0.340999,0.315026,0.29032},
	{1.01578,1.01667,1.0176,1.01858,1.01961,1.0207,1.02183,1.02303,1.02428,1.02559,1.02697,1.02841,1.02993,1.03152,1.03318,1.03492,1.03674,1.03864,1.04063,1.04271,1.04489,1.04715,1.04952,1.05199,1.05456,1.05724,1.06002,1.06292,1.06593,1.06906,1.0723,1.07566,1.07915,1.08275,1.08648,1.09034,1.09431,1.09842,1.10264,1.10699,1.11146,1.11606,1.12077,1.12559,1.13053,1.13558,1.14074,1.146,1.15135,1.15679,1.16232,1.16792,1.17359,1.17933,1.18511,1.19094,1.19681,1.20269,1.20859,1.21449,1.22038,1.22624,1.23207,1.23784,1.24355,1.24918,1.25471,1.26014,1.26544,1.2706,1.27561,1.28044,1.28509,1.28954,1.29377,1.29776,1.30151,1.30499,1.3082,1.31112,1.31374,1.31604,1.31802,1.31966,1.32095,1.32189,1.32247,1.32268,1.32253,1.32199,1.32109,1.3198,1.31815,1.31612,1.31372,1.31097,1.30786,1.30441,1.30063,1.29653,1.29214,1.28745,1.2825,1.27729,1.27186,1.26622,1.26039,1.2544,1.24828,1.24203,1.2357,1.2293,1.22286,1.2164,1.20995,1.20353,1.19715,1.19085,1.18464,1.17853,1.17254,1.16669,1.161,1.15546,1.15009,1.1449,1.13989,1.13507,1.13044,1.12599,1.12173,1.11765,1.11375,1.11002,1.10646,1.10306,1.0998,1.09668,1.09369,1.09081,1.08803,1.08534,1.08272,1.08016,1.07764,1.07515,1.07267,1.07018,1.06766,1.0651,1.06248,1.05977,1.05695,1.05401,1.05091,1.04764,1.04417,1.04046,1.0365,1.03225,1.02767,1.02274,1.01742,1.01167,1.00544,0.998712,0.991424,0.983536,0.975002,0.965772,0.955797,0.945026,0.933408,0.920893,0.907432,0.892975,0.877479,0.860901,0.843206,0.824365,0.804356,0.78317,0.760809,0.737288,0.71264,0.686916,0.660188,0.632548,0.604112,0.57502,0.545434,0.515537,0.485533,0.455636,0.426075,0.397071,0.368846,0.3416,0.315509,0.290715},
	{1.01575,1.01665,1.01758,1.01856,1.0196,1.02068,1.02182,1.02302,1.02428,1.02559,1.02698,1.02843,1.02995,1.03154,1.03321,1.03496,1.03679,1.0387,1.04071,1.0428,1.04499,1.04727,1.04966,1.05215,1.05474,1.05744,1.06025,1.06318,1.06622,1.06939,1.07267,1.07607,1.0796,1.08326,1.08704,1.09096,1.095,1.09917,1.10346,1.10789,1.11245,1.11713,1.12194,1.12687,1.13191,1.13708,1.14236,1.14774,1.15323,1.15882,1.1645,1.17026,1.17611,1.18202,1.18799,1.19401,1.20008,1.20617,1.21229,1.21842,1.22454,1.23064,1.23672,1.24275,1.24872,1.25462,1.26044,1.26615,1.27174,1.2772,1.2825,1.28765,1.2926,1.29736,1.30191,1.30622,1.31029,1.3141,1.31763,1.32087,1.32381,1.32643,1.32872,1.33068,1.33228,1.33353,1.33441,1.33492,1.33505,1.3348,1.33417,1.33316,1.33176,1.32999,1.32783,1.32531,1.32243,1.3192,1.31563,1.31173,1.30752,1.30301,1.29822,1.29318,1.28789,1.28239,1.2767,1.27083,1.26481,1.25868,1.25244,1.24613,1.23977,1.23339,1.227,1.22064,1.21432,1.20806,1.20189,1.19582,1.18987,1.18405,1.17837,1.17285,1.1675,1.16232,1.15732,1.1525,1.14786,1.14341,1.13915,1.13506,1.13115,1.1274,1.12382,1.12039,1.11711,1.11396,1.11093,1.10801,1.10519,1.10244,1.09977,1.09715,1.09457,1.09201,1.08945,1.08687,1.08427,1.08161,1.07887,1.07605,1.0731,1.07002,1.06678,1.06334,1.05969,1.05579,1.05162,1.04714,1.04233,1.03714,1.03153,1.02548,1.01894,1.01186,1.0042,0.995921,0.986967,0.977291,0.966843,0.955572,0.943427,0.930357,0.916314,0.90125,0.885122,0.86789,0.849521,0.829988,0.809275,0.787375,0.764296,0.740059,0.714704,0.688288,0.660891,0.632614,0.603581,0.573941,0.543863,0.513537,0.483174,0.452993,0.42322,0.394082,0.365794,0.338551,0.312521,0.287833},
	{1.01573,1.01662,1.01756,1.01854,1.01958,1.02066,1.02181,1.02301,1.02427,1.02559,1.02697,1.02843,1.02996,1.03156,1.03323,1.03499,1.03683,1.03875,1.04077,1.04287,1.04508,1.04738,1.04978,1.05229,1.0549,1.05763,1.06046,1.06342,1.06649,1.06969,1.07301,1.07646,1.08003,1.08373,1.08757,1.09154,1.09564,1.09987,1.10424,1.10875,1.11339,1.11816,1.12306,1.12809,1.13324,1.13852,1.14392,1.14943,1.15506,1.16079,1.16662,1.17254,1.17855,1.18464,1.1908,1.19701,1.20328,1.20959,1.21592,1.22227,1.22863,1.23498,1.24131,1.2476,1.25384,1.26001,1.26611,1.27211,1.278,1.28375,1.28937,1.29482,1.3001,1.30518,1.31004,1.31468,1.31908,1.32322,1.32708,1.33065,1.33392,1.33687,1.33949,1.34177,1.3437,1.34526,1.34645,1.34727,1.3477,1.34775,1.34741,1.34667,1.34555,1.34404,1.34214,1.33987,1.33722,1.33422,1.33086,1.32717,1.32316,1.31884,1.31423,1.30935,1.30423,1.29887,1.29332,1.28758,1.28169,1.27566,1.26953,1.26331,1.25704,1.25074,1.24443,1.23813,1.23187,1.22567,1.21954,1.21352,1.2076,1.20181,1.19616,1.19067,1.18533,1.18017,1.17518,1.17037,1.16574,1.16129,1.15702,1.15292,1.149,1.14525,1.14165,1.1382,1.1349,1.13172,1.12866,1.12571,1.12285,1.12007,1.11735,1.11469,1.11205,1.10943,1.10681,1.10416,1.10148,1.09873,1.09591,1.09299,1.08994,1.08674,1.08337,1.0798,1.076,1.07194,1.0676,1.06294,1.05793,1.05253,1.0467,1.0404,1.0336,1.02624,1.01829,1.00969,1.0004,0.99037,0.979545,0.967877,0.955314,0.941806,0.927304,0.911761,0.895136,0.877389,0.858489,0.838412,0.817144,0.79468,0.771033,0.746227,0.720306,0.693332,0.66539,0.636587,0.60705,0.576934,0.546414,0.515685,0.484958,0.454458,0.424411,0.395043,0.36657,0.339181,0.313039,0.288269},
	{1.0157,1.01659,1.01753,1.01852,1.01955,1.02064,1.02179,1.02299,1.02425,1.02558,1.02697,1.02843,1.02996,1.03156,1.03325,1.03501,1.03686,1.03879,1.04082,1.04294,1.04515,1.04747,1.04988,1.05241,1.05504,1.05779,1.06065,1.06364,1.06674,1.06997,1.07332,1.07681,1.08043,1.08418,1.08806,1.09208,1.09624,1.10054,1.10498,1.10956,1.11428,1.11913,1.12413,1.12925,1.13451,1.13991,1.14542,1.15106,1.15682,1.1627,1.16868,1.17476,1.18093,1.1872,1.19354,1.19994,1.20641,1.21293,1.21948,1.22606,1.23265,1.23925,1.24583,1.25238,1.25889,1.26534,1.27172,1.27801,1.2842,1.29026,1.29619,1.30196,1.30755,1.31296,1.31816,1.32313,1.32786,1.33234,1.33654,1.34045,1.34405,1.34734,1.3503,1.35291,1.35517,1.35706,1.35858,1.35971,1.36046,1.36081,1.36077,1.36033,1.35949,1.35825,1.35662,1.3546,1.35221,1.34944,1.34631,1.34284,1.33903,1.33491,1.33049,1.32579,1.32083,1.31564,1.31023,1.30463,1.29886,1.29295,1.28693,1.28082,1.27464,1.26842,1.26219,1.25596,1.24977,1.24362,1.23755,1.23156,1.22569,1.21993,1.21431,1.20884,1.20353,1.19838,1.1934,1.18859,1.18397,1.17952,1.17524,1.17114,1.16721,1.16344,1.15982,1.15635,1.15302,1.14981,1.14672,1.14373,1.14083,1.138,1.13523,1.1325,1.1298,1.12711,1.12441,1.12168,1.11891,1.11607,1.11314,1.11009,1.10692,1.10358,1.10006,1.09633,1.09236,1.08812,1.08357,1.0787,1.07345,1.06779,1.06169,1.05511,1.04799,1.04031,1.032,1.02303,1.01334,1.00289,0.991615,0.979476,0.966418,0.95239,0.937346,0.921239,0.904027,0.885676,0.866154,0.845442,0.823526,0.800409,0.776106,0.750647,0.724081,0.696477,0.667926,0.638541,0.608457,0.577835,0.546857,0.515721,0.484645,0.453856,0.423581,0.394046,0.365459,0.338009,0.311851,0.287098},
	{1.01567,1.01656,1.0175,1.01849,1.01952,1.02061,1.02176,1.02296,1.02423,1.02556,1.02695,1.02841,1.02995,1.03156,1.03325,1.03502,1.03687,1.03882,1.04085,1.04298,1.04521,1.04754,1.04997,1.05251,1.05517,1.05794,1.06082,1.06383,1.06696,1.07022,1.07361,1.07713,1.08079,1.08459,1.08852,1.0926,1.09681,1.10117,1.10568,1.11033,1.11512,1.12006,1.12515,1.13037,1.13573,1.14123,1.14687,1.15263,1.15853,1.16454,1.17067,1.1769,1.18325,1.18968,1.1962,1.2028,1.20947,1.2162,1.22297,1.22977,1.2366,1.24344,1.25028,1.25709,1.26387,1.2706,1.27727,1.28386,1.29035,1.29672,1.30296,1.30905,1.31497,1.32071,1.32624,1.33156,1.33663,1.34145,1.346,1.35026,1.35421,1.35785,1.36115,1.36411,1.36671,1.36894,1.37079,1.37226,1.37333,1.374,1.37426,1.37412,1.37358,1.37263,1.37128,1.36953,1.36739,1.36488,1.36199,1.35874,1.35516,1.35124,1.34702,1.34251,1.33772,1.33269,1.32744,1.32199,1.31636,1.31058,1.30467,1.29867,1.29259,1.28646,1.28031,1.27416,1.26803,1.26195,1.25593,1.25,1.24416,1.23845,1.23286,1.22742,1.22213,1.21699,1.21203,1.20723,1.20261,1.19816,1.19389,1.18978,1.18584,1.18205,1.17842,1.17493,1.17157,1.16834,1.16521,1.16218,1.15924,1.15636,1.15354,1.15076,1.14799,1.14524,1.14246,1.13965,1.13679,1.13385,1.13081,1.12766,1.12436,1.12089,1.11722,1.11333,1.10919,1.10477,1.10003,1.09494,1.08947,1.08357,1.07721,1.07035,1.06294,1.05494,1.04629,1.03696,1.02689,1.01604,1.00434,0.991757,0.97823,0.963713,0.948156,0.931517,0.913754,0.894834,0.874729,0.853419,0.830898,0.807168,0.782251,0.756179,0.729008,0.700811,0.671686,0.64175,0.611145,0.580038,0.548615,0.51708,0.485653,0.454563,0.424039,0.394305,0.365567,0.338008,0.311777,0.286982},
	{1.01563,1.01652,1.01746,1.01845,1.01949,1.02058,1.02173,1.02293,1.0242,1.02553,1.02693,1.02839,1.02993,1.03155,1.03324,1.03502,1.03688,1.03883,1.04088,1.04302,1.04525,1.0476,1.05004,1.0526,1.05527,1.05806,1.06097,1.064,1.06716,1.07045,1.07387,1.07743,1.08113,1.08496,1.08894,1.09307,1.09734,1.10176,1.10633,1.11105,1.11592,1.12094,1.12611,1.13143,1.1369,1.1425,1.14825,1.15414,1.16016,1.16631,1.17259,1.17898,1.18549,1.19209,1.1988,1.20559,1.21246,1.21939,1.22638,1.23341,1.24048,1.24756,1.25465,1.26173,1.26879,1.2758,1.28276,1.28964,1.29644,1.30312,1.30968,1.3161,1.32235,1.32843,1.3343,1.33996,1.34539,1.35056,1.35546,1.36008,1.36439,1.36838,1.37204,1.37536,1.37831,1.38089,1.38308,1.38489,1.38629,1.38729,1.38788,1.38806,1.38782,1.38717,1.38611,1.38464,1.38278,1.38052,1.37788,1.37488,1.37152,1.36783,1.36381,1.35949,1.3549,1.35004,1.34496,1.33966,1.33417,1.32852,1.32274,1.31686,1.31088,1.30486,1.29879,1.29273,1.28667,1.28066,1.2747,1.26882,1.26303,1.25735,1.2518,1.24639,1.24112,1.23601,1.23106,1.22628,1.22166,1.21721,1.21294,1.20882,1.20487,1.20107,1.19742,1.19391,1.19053,1.18726,1.1841,1.18103,1.17805,1.17512,1.17225,1.16941,1.16658,1.16375,1.16089,1.158,1.15504,1.152,1.14886,1.14558,1.14215,1.13855,1.13473,1.13068,1.12637,1.12175,1.11681,1.11151,1.1058,1.09965,1.09302,1.08587,1.07816,1.06983,1.06084,1.05114,1.04068,1.02942,1.01729,1.00425,0.990243,0.975226,0.959149,0.94197,0.923648,0.904152,0.883457,0.861545,0.838412,0.814066,0.788531,0.761844,0.734067,0.705278,0.675578,0.645093,0.61397,0.58238,0.550515,0.518583,0.486808,0.455419,0.424647,0.394714,0.365824,0.338153,0.311846,0.287002},
	{1.01559,1.01648,1.01742,1.01841,1.01945,1.02054,1.02169,1.0229,1.02417,1.0255,1.0269,1.02837,1.02991,1.03153,1.03323,1.03501,1.03688,1.03884,1.04089,1.04304,1.04529,1.04764,1.0501,1.05267,1.05536,1.05817,1.0611,1.06415,1.06734,1.07065,1.07411,1.0777,1.08143,1.08531,1.08933,1.09351,1.09784,1.10231,1.10695,1.11173,1.11668,1.12178,1.12703,1.13244,1.138,1.14372,1.14958,1.15559,1.16174,1.16803,1.17444,1.18099,1.18766,1.19444,1.20132,1.2083,1.21536,1.22251,1.22971,1.23698,1.24428,1.25161,1.25895,1.2663,1.27363,1.28092,1.28817,1.29536,1.30246,1.30946,1.31635,1.32309,1.32969,1.3361,1.34233,1.34834,1.35412,1.35966,1.36492,1.3699,1.37458,1.37894,1.38297,1.38664,1.38996,1.3929,1.39545,1.39761,1.39936,1.4007,1.40162,1.40213,1.40221,1.40187,1.40111,1.39994,1.39835,1.39637,1.394,1.39124,1.38813,1.38466,1.38086,1.37676,1.37236,1.36769,1.36277,1.35764,1.35231,1.3468,1.34116,1.33539,1.32953,1.32361,1.31764,1.31166,1.30569,1.29974,1.29385,1.28802,1.28229,1.27666,1.27115,1.26577,1.26053,1.25544,1.25051,1.24574,1.24113,1.23669,1.23241,1.22829,1.22433,1.22052,1.21685,1.21332,1.20991,1.20662,1.20342,1.20032,1.19729,1.19431,1.19139,1.18848,1.18559,1.18269,1.17976,1.17678,1.17373,1.17059,1.16733,1.16394,1.16038,1.15663,1.15267,1.14845,1.14396,1.13916,1.13402,1.1285,1.12256,1.11616,1.10927,1.10184,1.09382,1.08517,1.07584,1.06578,1.05494,1.04326,1.03071,1.01722,1.00275,0.987246,0.970662,0.952955,0.93409,0.914034,0.892764,0.870268,0.846542,0.821599,0.795464,0.768182,0.739816,0.710451,0.680194,0.649174,0.617543,0.585479,0.553176,0.520847,0.488718,0.457022,0.425987,0.395835,0.366768,0.338957,0.312541,0.287612},
	{1.01555,1.01644,1.01738,1.01837,1.01941,1.0205,1.02165,1.02285,1.02412,1.02546,1.02686,1.02833,1.02988,1.0315,1.0332,1.03499,1.03686,1.03883,1.04089,1.04304,1.0453,1.04767,1.05014,1.05273,1.05543,1.05825,1.0612,1.06428,1.06749,1.07083,1.07431,1.07794,1.08171,1.08562,1.08969,1.09391,1.09829,1.10282,1.10752,1.11237,1.11739,1.12256,1.1279,1.1334,1.13906,1.14487,1.15085,1.15697,1.16325,1.16967,1.17623,1.18293,1.18975,1.1967,1.20376,1.21093,1.21819,1.22554,1.23297,1.24046,1.248,1.25557,1.26317,1.27078,1.27839,1.28597,1.29351,1.301,1.30842,1.31574,1.32295,1.33003,1.33697,1.34373,1.35031,1.35668,1.36283,1.36873,1.37436,1.37972,1.38477,1.3895,1.39391,1.39796,1.40164,1.40495,1.40787,1.41039,1.4125,1.4142,1.41547,1.41631,1.41672,1.41671,1.41626,1.41539,1.41411,1.41241,1.41031,1.40782,1.40495,1.40173,1.39816,1.39427,1.39008,1.3856,1.38087,1.37591,1.37074,1.36539,1.35988,1.35424,1.34851,1.34269,1.33683,1.33094,1.32505,1.31918,1.31335,1.30759,1.30191,1.29632,1.29085,1.28551,1.2803,1.27523,1.27032,1.26556,1.26096,1.25652,1.25224,1.24811,1.24414,1.24031,1.23662,1.23306,1.22962,1.22629,1.22306,1.2199,1.21682,1.21379,1.21079,1.20782,1.20484,1.20185,1.19883,1.19574,1.19258,1.18931,1.18593,1.18239,1.17867,1.17476,1.17061,1.1662,1.1615,1.15647,1.15108,1.14529,1.13907,1.13237,1.12516,1.11738,1.10899,1.09995,1.09021,1.07971,1.0684,1.05624,1.04317,1.02914,1.01411,0.998013,0.980815,0.962475,0.942955,0.922229,0.900274,0.877081,0.852653,0.827005,0.800169,0.772195,0.743152,0.713132,0.682249,0.650639,0.618462,0.5859,0.553155,0.520444,0.487996,0.456045,0.424819,0.394538,0.365396,0.33756,0.311157,0.286272},
	{1.0155,1.01639,1.01733,1.01832,1.01936,1.02045,1.0216,1.02281,1.02408,1.02541,1.02682,1.02829,1.02984,1.03146,1.03317,1.03496,1.03684,1.03881,1.04087,1.04304,1.04531,1.04768,1.05016,1.05276,1.05548,1.05832,1.06129,1.06438,1.06761,1.07098,1.07449,1.07815,1.08195,1.0859,1.09001,1.09428,1.09871,1.10329,1.10805,1.11296,1.11805,1.1233,1.12872,1.1343,1.14005,1.14597,1.15205,1.15829,1.16469,1.17125,1.17795,1.18479,1.19178,1.19889,1.20613,1.21348,1.22094,1.2285,1.23614,1.24385,1.25163,1.25945,1.26731,1.27519,1.28307,1.29094,1.29878,1.30657,1.3143,1.32195,1.32949,1.33691,1.34419,1.35131,1.35825,1.36499,1.3715,1.37778,1.38379,1.38953,1.39496,1.40008,1.40487,1.4093,1.41338,1.41707,1.42037,1.42326,1.42574,1.4278,1.42943,1.43062,1.43138,1.4317,1.43158,1.43103,1.43005,1.42865,1.42684,1.42462,1.42202,1.41905,1.41573,1.41206,1.40809,1.40382,1.39928,1.3945,1.38951,1.38431,1.37896,1.37346,1.36785,1.36215,1.3564,1.35061,1.34481,1.33902,1.33327,1.32758,1.32195,1.31642,1.311,1.30569,1.30051,1.29547,1.29058,1.28584,1.28125,1.27682,1.27254,1.26841,1.26443,1.26058,1.25688,1.2533,1.24983,1.24646,1.24319,1.23999,1.23686,1.23378,1.23072,1.22768,1.22463,1.22156,1.21845,1.21527,1.212,1.20862,1.20511,1.20144,1.19758,1.19351,1.18919,1.1846,1.17971,1.17447,1.16886,1.16283,1.15635,1.14938,1.14187,1.13378,1.12506,1.11567,1.10555,1.09466,1.08294,1.07035,1.05682,1.04231,1.02677,1.01015,0.992413,0.973511,0.953414,0.932095,0.909535,0.885728,0.860678,0.834406,0.806947,0.778357,0.748708,0.718098,0.686646,0.654494,0.621805,0.588769,0.55559,0.522489,0.489698,0.457451,0.425976,0.395491,0.366187,0.338224,0.311725,0.286768},
	{1.01545,1.01634,1.01728,1.01827,1.0193,1.0204,1.02155,1.02275,1.02402,1.02536,1.02676,1.02824,1.02979,1.03142,1.03313,1.03492,1.0368,1.03878,1.04085,1.04302,1.04529,1.04768,1.05017,1.05278,1.05551,1.05837,1.06135,1.06447,1.06772,1.07111,1.07465,1.07833,1.08216,1.08615,1.0903,1.09461,1.09908,1.10372,1.10853,1.11351,1.11866,1.12398,1.12948,1.13515,1.14099,1.14701,1.1532,1.15955,1.16607,1.17275,1.17959,1.18659,1.19372,1.201,1.20842,1.21596,1.22361,1.23137,1.23923,1.24717,1.25518,1.26325,1.27136,1.27951,1.28767,1.29582,1.30396,1.31206,1.32011,1.32808,1.33595,1.34372,1.35135,1.35883,1.36613,1.37324,1.38013,1.38679,1.39319,1.39932,1.40514,1.41066,1.41584,1.42067,1.42513,1.42922,1.43291,1.43619,1.43905,1.44148,1.44348,1.44504,1.44616,1.44683,1.44705,1.44683,1.44617,1.44508,1.44356,1.44164,1.43931,1.43661,1.43353,1.43011,1.42637,1.42231,1.41798,1.41339,1.40857,1.40355,1.39835,1.393,1.38753,1.38196,1.37632,1.37064,1.36493,1.35923,1.35356,1.34793,1.34237,1.3369,1.33152,1.32625,1.32111,1.3161,1.31123,1.3065,1.30193,1.2975,1.29322,1.28908,1.28509,1.28123,1.2775,1.27389,1.27039,1.26699,1.26367,1.26043,1.25724,1.25409,1.25096,1.24784,1.24471,1.24155,1.23833,1.23504,1.23165,1.22815,1.22449,1.22067,1.21664,1.21239,1.20788,1.20309,1.19797,1.19249,1.18662,1.18032,1.17355,1.16626,1.15842,1.14997,1.14088,1.13108,1.12054,1.1092,1.097,1.08391,1.06986,1.0548,1.03869,1.02148,1.00312,0.983587,0.962837,0.940849,0.917609,0.893112,0.867368,0.840403,0.812255,0.782986,0.752676,0.721427,0.689365,0.65664,0.62342,0.5899,0.55629,0.522816,0.48971,0.457208,0.425536,0.39491,0.365513,0.337501,0.310987,0.286039},
	{1.01539,1.01628,1.01722,1.01821,1.01925,1.02034,1.02149,1.02269,1.02397,1.0253,1.02671,1.02818,1.02973,1.03136,1.03307,1.03487,1.03676,1.03873,1.04081,1.04299,1.04527,1.04766,1.05016,1.05278,1.05553,1.0584,1.06139,1.06453,1.0678,1.07121,1.07477,1.07848,1.08235,1.08637,1.09056,1.09491,1.09942,1.10411,1.10898,1.11401,1.11923,1.12462,1.13019,1.13595,1.14188,1.14799,1.15428,1.16074,1.16738,1.17419,1.18117,1.1883,1.1956,1.20304,1.21063,1.21835,1.22619,1.23416,1.24223,1.25039,1.25864,1.26695,1.27533,1.28374,1.29217,1.30062,1.30906,1.31747,1.32583,1.33413,1.34235,1.35046,1.35845,1.36629,1.37396,1.38145,1.38872,1.39577,1.40256,1.40908,1.41531,1.42123,1.42682,1.43205,1.43692,1.44141,1.4455,1.44918,1.45243,1.45526,1.45764,1.45958,1.46106,1.46209,1.46267,1.46279,1.46246,1.4617,1.4605,1.45887,1.45684,1.45441,1.4516,1.44843,1.44492,1.4411,1.43698,1.4326,1.42797,1.42313,1.41809,1.4129,1.40757,1.40213,1.39662,1.39105,1.38545,1.37984,1.37425,1.3687,1.36321,1.3578,1.35247,1.34725,1.34215,1.33717,1.33232,1.32762,1.32305,1.31863,1.31435,1.31022,1.30621,1.30234,1.29859,1.29496,1.29143,1.28799,1.28463,1.28133,1.27809,1.27488,1.27168,1.26849,1.26527,1.26202,1.2587,1.2553,1.25179,1.24816,1.24436,1.24039,1.2362,1.23177,1.22708,1.22208,1.21674,1.21103,1.20491,1.19834,1.19128,1.18369,1.17552,1.16673,1.15726,1.14708,1.13613,1.12435,1.1117,1.09812,1.08357,1.06799,1.05133,1.03355,1.01461,0.99447,0.973098,0.950474,0.926587,0.901435,0.875033,0.847408,0.818605,0.788691,0.75775,0.725891,0.693245,0.659966,0.626231,0.592238,0.5582,0.524347,0.490913,0.458135,0.426238,0.395432,0.3659,0.337789,0.311204,0.286209},
	{1.01533,1.01622,1.01716,1.01815,1.01918,1.02027,1.02142,1.02263,1.0239,1.02524,1.02664,1.02812,1.02967,1.0313,1.03301,1.03481,1.0367,1.03868,1.04076,1.04294,1.04523,1.04763,1.05014,1.05277,1.05552,1.0584,1.06142,1.06456,1.06785,1.07129,1.07487,1.07861,1.0825,1.08656,1.09078,1.09517,1.09973,1.10446,1.10938,1.11447,1.11975,1.12521,1.13085,1.13668,1.1427,1.14891,1.1553,1.16187,1.16862,1.17556,1.18267,1.18994,1.19739,1.20499,1.21275,1.22065,1.22869,1.23686,1.24514,1.25353,1.26201,1.27057,1.2792,1.28788,1.29659,1.30533,1.31406,1.32278,1.33147,1.3401,1.34866,1.35712,1.36547,1.37368,1.38173,1.38959,1.39726,1.4047,1.4119,1.41882,1.42546,1.43179,1.43779,1.44344,1.44872,1.45362,1.45813,1.46221,1.46588,1.4691,1.47188,1.47421,1.47608,1.47748,1.47842,1.4789,1.47893,1.47849,1.47762,1.47631,1.47458,1.47244,1.46991,1.467,1.46375,1.46016,1.45627,1.4521,1.44768,1.44302,1.43816,1.43313,1.42796,1.42266,1.41728,1.41183,1.40634,1.40083,1.39534,1.38987,1.38445,1.3791,1.37383,1.36866,1.36359,1.35865,1.35383,1.34915,1.3446,1.34019,1.33591,1.33177,1.32776,1.32387,1.3201,1.31644,1.31287,1.30939,1.30599,1.30264,1.29933,1.29606,1.29279,1.28951,1.28621,1.28285,1.27943,1.27591,1.27228,1.2685,1.26456,1.26042,1.25606,1.25144,1.24654,1.24133,1.23576,1.2298,1.22341,1.21655,1.20919,1.20127,1.19276,1.1836,1.17374,1.16315,1.15176,1.13952,1.12639,1.11231,1.09723,1.08109,1.06386,1.04549,1.02593,1.00515,0.983128,0.959841,0.935279,0.909447,0.88236,0.854053,0.824574,0.793995,0.762408,0.729925,0.696685,0.662847,0.628593,0.594127,0.559666,0.525442,0.491693,0.458653,0.426548,0.395584,0.365936,0.337747,0.311115,0.286092},
	{1.01527,1.01616,1.0171,1.01808,1.01912,1.02021,1.02135,1.02256,1.02383,1.02516,1.02657,1.02804,1.0296,1.03123,1.03294,1.03474,1.03663,1.03862,1.0407,1.04288,1.04518,1.04758,1.0501,1.05274,1.0555,1.05839,1.06142,1.06458,1.06789,1.07134,1.07494,1.0787,1.08262,1.08671,1.09096,1.09539,1.09999,1.10477,1.10973,1.11488,1.12022,1.12574,1.13146,1.13737,1.14347,1.14976,1.15625,1.16293,1.1698,1.17685,1.18409,1.19151,1.1991,1.20686,1.21479,1.22287,1.2311,1.23946,1.24796,1.25657,1.26528,1.27409,1.28297,1.29192,1.30091,1.30994,1.31897,1.32801,1.33702,1.34598,1.35488,1.3637,1.37241,1.38099,1.38942,1.39768,1.40574,1.41358,1.42119,1.42853,1.43558,1.44233,1.44875,1.45483,1.46054,1.46586,1.47079,1.4753,1.47938,1.48302,1.48621,1.48894,1.4912,1.493,1.49432,1.49517,1.49555,1.49547,1.49494,1.49395,1.49254,1.4907,1.48846,1.48583,1.48284,1.47951,1.47586,1.47191,1.4677,1.46324,1.45858,1.45372,1.44871,1.44356,1.43832,1.43299,1.42762,1.42222,1.41682,1.41144,1.4061,1.40082,1.39561,1.39049,1.38548,1.38057,1.37578,1.37112,1.36659,1.36219,1.35792,1.35377,1.34975,1.34585,1.34206,1.33837,1.33477,1.33125,1.3278,1.3244,1.32104,1.31769,1.31435,1.31099,1.30759,1.30414,1.3006,1.29697,1.2932,1.28929,1.28519,1.28089,1.27635,1.27155,1.26644,1.261,1.2552,1.24899,1.24233,1.23518,1.22751,1.21927,1.21041,1.20088,1.19064,1.17963,1.16781,1.15511,1.1415,1.12691,1.11131,1.09462,1.07682,1.05786,1.03769,1.01629,0.993626,0.969686,0.944462,0.91796,0.890202,0.861225,0.831084,0.799854,0.767632,0.734537,0.700713,0.666324,0.631558,0.596622,0.561739,0.527142,0.493071,0.45976,0.427432,0.39629,0.366505,0.338213,0.311505,0.286426},
	{1.01521,1.01609,1.01703,1.01801,1.01904,1.02013,1.02128,1.02248,1.02375,1.02509,1.02649,1.02796,1.02952,1.03115,1.03286,1.03466,1.03655,1.03854,1.04062,1.04281,1.04511,1.04752,1.05004,1.05269,1.05546,1.05836,1.0614,1.06457,1.0679,1.07137,1.07499,1.07877,1.08272,1.08683,1.09111,1.09558,1.10021,1.10504,1.11005,1.11525,1.12064,1.12623,1.13201,1.138,1.14418,1.15056,1.15714,1.16392,1.1709,1.17807,1.18544,1.19299,1.20073,1.20865,1.21674,1.225,1.23341,1.24198,1.25068,1.25951,1.26845,1.2775,1.28664,1.29586,1.30513,1.31444,1.32378,1.33313,1.34246,1.35176,1.36101,1.37019,1.37926,1.38822,1.39704,1.40569,1.41415,1.4224,1.43042,1.43818,1.44566,1.45284,1.45969,1.4662,1.47235,1.47811,1.48347,1.48841,1.49292,1.49699,1.5006,1.50375,1.50642,1.50862,1.51033,1.51157,1.51232,1.51261,1.51242,1.51178,1.51069,1.50917,1.50724,1.5049,1.50219,1.49912,1.49572,1.49201,1.48802,1.48377,1.4793,1.47463,1.46978,1.4648,1.4597,1.45451,1.44926,1.44398,1.43868,1.43339,1.42813,1.42292,1.41778,1.41272,1.40775,1.40288,1.39813,1.39349,1.38897,1.38458,1.38031,1.37616,1.37213,1.36821,1.3644,1.36067,1.35704,1.35347,1.34996,1.3465,1.34307,1.33964,1.33621,1.33276,1.32926,1.32569,1.32203,1.31826,1.31434,1.31027,1.306,1.30151,1.29677,1.29174,1.28641,1.28072,1.27464,1.26814,1.26118,1.25371,1.24569,1.23707,1.22781,1.21787,1.20718,1.19571,1.18339,1.17018,1.15602,1.14087,1.12467,1.10737,1.08892,1.0693,1.04844,1.02634,1.00296,0.978293,0.952331,0.925088,0.896589,0.866877,0.836011,0.804075,0.77117,0.737424,0.702985,0.668025,0.632738,0.597336,0.562045,0.527103,0.49275,0.459219,0.42673,0.395481,0.365636,0.337322,0.310621,0.285568},
	{1.01514,1.01602,1.01695,1.01794,1.01897,1.02005,1.0212,1.0224,1.02367,1.025,1.0264,1.02788,1.02943,1.03106,1.03277,1.03457,1.03646,1.03845,1.04054,1.04273,1.04503,1.04744,1.04997,1.05262,1.0554,1.05831,1.06136,1.06455,1.06788,1.07137,1.07501,1.07881,1.08278,1.08692,1.09123,1.09573,1.1004,1.10526,1.11032,1.11557,1.12102,1.12666,1.13251,1.13857,1.14483,1.15129,1.15797,1.16485,1.17193,1.17922,1.18671,1.1944,1.20228,1.21035,1.21861,1.22704,1.23564,1.2444,1.2533,1.26235,1.27153,1.28082,1.29021,1.29969,1.30924,1.31885,1.32849,1.33815,1.34781,1.35745,1.36705,1.37658,1.38603,1.39537,1.40457,1.41362,1.42249,1.43116,1.4396,1.44779,1.4557,1.46332,1.47062,1.47757,1.48416,1.49037,1.49618,1.50156,1.50652,1.51102,1.51507,1.51864,1.52174,1.52435,1.52648,1.52811,1.52926,1.52992,1.5301,1.52982,1.52907,1.52788,1.52627,1.52424,1.52181,1.51902,1.51588,1.51242,1.50867,1.50464,1.50038,1.4959,1.49124,1.48642,1.48148,1.47644,1.47132,1.46616,1.46097,1.45578,1.45061,1.44549,1.44041,1.43541,1.4305,1.42567,1.42096,1.41635,1.41185,1.40748,1.40321,1.39907,1.39503,1.39109,1.38726,1.38351,1.37983,1.37623,1.37267,1.36915,1.36565,1.36216,1.35865,1.3551,1.3515,1.34783,1.34405,1.34015,1.3361,1.33187,1.32744,1.32277,1.31784,1.31261,1.30706,1.30113,1.29481,1.28804,1.28079,1.27302,1.26467,1.25571,1.24609,1.23576,1.22467,1.21277,1.2,1.18632,1.17166,1.15599,1.13925,1.12139,1.10236,1.08213,1.06067,1.03793,1.0139,0.988574,0.961946,0.934032,0.904861,0.874481,0.842955,0.810372,0.776838,0.742485,0.707469,0.671965,0.636172,0.600305,0.564596,0.529283,0.494606,0.460799,0.42808,0.396642,0.366645,0.338209,0.31141,0.286278},
	{1.01506,1.01595,1.01688,1.01786,1.01889,1.01997,1.02111,1.02231,1.02358,1.02491,1.02631,1.02778,1.02933,1.03096,1.03267,1.03447,1.03636,1.03835,1.04044,1.04263,1.04493,1.04735,1.04988,1.05254,1.05532,1.05824,1.0613,1.0645,1.06784,1.07134,1.075,1.07882,1.08281,1.08697,1.09131,1.09584,1.10055,1.10545,1.11055,1.11584,1.12134,1.12705,1.13296,1.13908,1.14541,1.15196,1.15873,1.1657,1.17289,1.18029,1.18791,1.19572,1.20375,1.21197,1.22038,1.22898,1.23776,1.24672,1.25583,1.26509,1.2745,1.28403,1.29368,1.30342,1.31325,1.32315,1.33309,1.34307,1.35305,1.36303,1.37298,1.38287,1.3927,1.40242,1.41202,1.42147,1.43076,1.43984,1.44871,1.45734,1.46569,1.47375,1.4815,1.48891,1.49596,1.50262,1.50889,1.51473,1.52014,1.5251,1.52959,1.53361,1.53714,1.54018,1.54273,1.54478,1.54633,1.54738,1.54795,1.54803,1.54765,1.5468,1.54551,1.5438,1.54168,1.53918,1.53632,1.53312,1.52961,1.52582,1.52177,1.5175,1.51303,1.50839,1.50362,1.49873,1.49375,1.48871,1.48364,1.47856,1.47349,1.46845,1.46345,1.45851,1.45365,1.44888,1.4442,1.43962,1.43515,1.43079,1.42653,1.42238,1.41834,1.41439,1.41053,1.40674,1.40303,1.39938,1.39577,1.39219,1.38862,1.38504,1.38144,1.3778,1.37409,1.3703,1.36639,1.36235,1.35814,1.35375,1.34914,1.34428,1.33915,1.3337,1.3279,1.32172,1.31512,1.30806,1.3005,1.2924,1.2837,1.27437,1.26436,1.25361,1.24207,1.22971,1.21645,1.20225,1.18707,1.17083,1.15351,1.13504,1.11539,1.09452,1.07239,1.04897,1.02425,0.998224,0.970887,0.94226,0.912379,0.881293,0.849074,0.815814,0.781625,0.746647,0.711038,0.67498,0.638679,0.602353,0.566235,0.530568,0.495592,0.461538,0.428623,0.397034,0.366926,0.338411,0.311557,0.286387},
	{1.01499,1.01587,1.0168,1.01777,1.0188,1.01988,1.02102,1.02222,1.02348,1.02481,1.02621,1.02768,1.02922,1.03085,1.03256,1.03436,1.03625,1.03824,1.04033,1.04252,1.04483,1.04724,1.04978,1.05244,1.05523,1.05815,1.06122,1.06442,1.06778,1.07129,1.07497,1.0788,1.08281,1.087,1.09136,1.09591,1.10066,1.10559,1.11073,1.11607,1.12162,1.12738,1.13335,1.13954,1.14594,1.15257,1.15942,1.16649,1.17378,1.18129,1.18902,1.19697,1.20513,1.21349,1.22206,1.23083,1.23979,1.24894,1.25825,1.26773,1.27736,1.28713,1.29703,1.30704,1.31714,1.32733,1.33758,1.34787,1.35818,1.3685,1.3788,1.38906,1.39926,1.40937,1.41937,1.42923,1.43893,1.44845,1.45775,1.46682,1.47562,1.48414,1.49235,1.50022,1.50773,1.51486,1.5216,1.52791,1.53379,1.53921,1.54417,1.54864,1.55262,1.55611,1.55909,1.56157,1.56354,1.565,1.56596,1.56643,1.56642,1.56593,1.56499,1.56361,1.56181,1.55961,1.55704,1.55412,1.55087,1.54732,1.54351,1.53945,1.53519,1.53074,1.52613,1.52141,1.51658,1.51168,1.50674,1.50177,1.4968,1.49185,1.48693,1.48207,1.47727,1.47255,1.46791,1.46337,1.45892,1.45458,1.45033,1.44619,1.44213,1.43817,1.43429,1.43048,1.42673,1.42303,1.41937,1.41572,1.41208,1.40843,1.40474,1.401,1.39719,1.39327,1.38924,1.38506,1.3807,1.37614,1.37136,1.36631,1.36097,1.3553,1.34927,1.34284,1.33597,1.32862,1.32076,1.31233,1.30329,1.29359,1.28319,1.27203,1.26007,1.24725,1.23352,1.21883,1.20312,1.18635,1.16846,1.14941,1.12916,1.10766,1.08489,1.06082,1.03544,1.00873,0.980717,0.951407,0.920843,0.88908,0.856193,0.82228,0.787458,0.751872,0.715685,0.679084,0.642277,0.60549,0.568956,0.53292,0.497623,0.463294,0.430148,0.398368,0.368104,0.339462,0.312502,0.287242},
	{1.01491,1.01579,1.01671,1.01768,1.01871,1.01979,1.02092,1.02212,1.02338,1.0247,1.0261,1.02757,1.02911,1.03074,1.03245,1.03424,1.03613,1.03812,1.04021,1.0424,1.0447,1.04712,1.04966,1.05232,1.05512,1.05804,1.06111,1.06433,1.06769,1.07122,1.0749,1.07876,1.08278,1.08699,1.09138,1.09595,1.10072,1.10569,1.11087,1.11625,1.12184,1.12765,1.13368,1.13993,1.14641,1.15311,1.16004,1.1672,1.17459,1.18221,1.19005,1.19812,1.20641,1.21493,1.22365,1.23259,1.24172,1.25105,1.26057,1.27026,1.28011,1.29012,1.30027,1.31054,1.32092,1.33139,1.34194,1.35255,1.36319,1.37385,1.38451,1.39513,1.40571,1.41621,1.42661,1.43688,1.44701,1.45696,1.4667,1.47622,1.48548,1.49446,1.50313,1.51148,1.51947,1.52708,1.53429,1.54108,1.54744,1.55334,1.55877,1.56371,1.56816,1.5721,1.57554,1.57846,1.58086,1.58274,1.58412,1.58499,1.58536,1.58525,1.58467,1.58363,1.58217,1.58028,1.57801,1.57538,1.5724,1.56911,1.56554,1.56171,1.55766,1.55341,1.54899,1.54443,1.53976,1.53501,1.53019,1.52534,1.52048,1.51563,1.51079,1.50601,1.50127,1.4966,1.49201,1.4875,1.48308,1.47876,1.47452,1.47037,1.46631,1.46233,1.45842,1.45457,1.45078,1.44703,1.4433,1.43959,1.43587,1.43212,1.42833,1.42448,1.42054,1.41649,1.41231,1.40797,1.40344,1.3987,1.39371,1.38844,1.38287,1.37695,1.37065,1.36393,1.35676,1.34909,1.34087,1.33207,1.32264,1.31253,1.30169,1.29007,1.27762,1.26429,1.25002,1.23477,1.21847,1.20109,1.18257,1.16286,1.14193,1.11974,1.09626,1.07146,1.04533,1.01788,0.989103,0.959037,0.927723,0.895219,0.861607,0.826991,0.791493,0.755265,0.718476,0.68132,0.644009,0.606771,0.569846,0.533478,0.497909,0.463365,0.430058,0.398164,0.367825,0.339141,0.312162,0.286897},
	{1.01482,1.0157,1.01662,1.01759,1.01861,1.01969,1.02082,1.02202,1.02327,1.02459,1.02598,1.02745,1.02899,1.03061,1.03232,1.03411,1.036,1.03798,1.04007,1.04226,1.04457,1.04698,1.04952,1.05219,1.05498,1.05792,1.06099,1.06421,1.06758,1.07112,1.07481,1.07868,1.08272,1.08694,1.09135,1.09595,1.10075,1.10575,1.11096,1.11638,1.12202,1.12788,1.13396,1.14027,1.14681,1.15358,1.16059,1.16784,1.17532,1.18304,1.191,1.19919,1.20761,1.21627,1.22514,1.23424,1.24355,1.25306,1.26277,1.27267,1.28275,1.29299,1.30338,1.31391,1.32457,1.33533,1.34618,1.3571,1.36808,1.37908,1.39009,1.40108,1.41204,1.42293,1.43374,1.44443,1.45498,1.46536,1.47556,1.48553,1.49525,1.5047,1.51385,1.52268,1.53116,1.53926,1.54696,1.55425,1.56109,1.56748,1.5734,1.57882,1.58375,1.58817,1.59207,1.59544,1.59829,1.60061,1.60241,1.6037,1.60447,1.60475,1.60455,1.60387,1.60275,1.6012,1.59925,1.59691,1.59422,1.5912,1.58788,1.58429,1.58046,1.57642,1.5722,1.56782,1.56332,1.55872,1.55404,1.54932,1.54457,1.53982,1.53508,1.53036,1.5257,1.52109,1.51654,1.51207,1.50768,1.50337,1.49914,1.49499,1.49092,1.48692,1.48299,1.4791,1.47526,1.47146,1.46766,1.46387,1.46007,1.45623,1.45233,1.44836,1.44429,1.4401,1.43576,1.43125,1.42654,1.4216,1.4164,1.41091,1.40509,1.3989,1.39232,1.38531,1.37781,1.3698,1.36122,1.35204,1.3422,1.33166,1.32036,1.30826,1.29531,1.28144,1.26662,1.25079,1.23389,1.21587,1.1967,1.17632,1.15469,1.13178,1.10756,1.08202,1.05514,1.02692,0.997375,0.966543,0.934467,0.901214,0.866871,0.831545,0.79537,0.758499,0.72111,0.683401,0.645591,0.607911,0.570603,0.533913,0.498082,0.463334,0.429875,0.397877,0.367473,0.338753,0.31176,0.286495},
	{1.01474,1.01561,1.01653,1.0175,1.01851,1.01959,1.02072,1.0219,1.02316,1.02447,1.02586,1.02732,1.02886,1.03048,1.03218,1.03397,1.03586,1.03784,1.03992,1.04211,1.04441,1.04683,1.04937,1.05203,1.05483,1.05777,1.06084,1.06407,1.06745,1.07099,1.0747,1.07857,1.08263,1.08687,1.09129,1.09592,1.10074,1.10577,1.11101,1.11647,1.12214,1.12805,1.13418,1.14055,1.14715,1.15399,1.16108,1.16841,1.17598,1.1838,1.19186,1.20017,1.20872,1.21751,1.22653,1.23579,1.24526,1.25496,1.26486,1.27497,1.28526,1.29574,1.30638,1.31717,1.3281,1.33914,1.35029,1.36153,1.37283,1.38417,1.39554,1.4069,1.41824,1.42953,1.44075,1.45186,1.46284,1.47367,1.48431,1.49475,1.50494,1.51487,1.52451,1.53382,1.5428,1.5514,1.5596,1.56739,1.57474,1.58163,1.58805,1.59397,1.59939,1.6043,1.60868,1.61252,1.61583,1.61861,1.62085,1.62257,1.62376,1.62445,1.62463,1.62434,1.62358,1.62237,1.62075,1.61873,1.61634,1.6136,1.61055,1.60721,1.60362,1.5998,1.59578,1.5916,1.58727,1.58283,1.5783,1.57372,1.56909,1.56445,1.5598,1.55518,1.55059,1.54604,1.54155,1.53712,1.53276,1.52847,1.52425,1.52011,1.51603,1.51201,1.50805,1.50413,1.50025,1.49639,1.49253,1.48867,1.48478,1.48084,1.47684,1.47275,1.46855,1.46422,1.45973,1.45505,1.45016,1.44502,1.43961,1.43389,1.42782,1.42138,1.41452,1.4072,1.39939,1.39104,1.38211,1.37255,1.36231,1.35134,1.3396,1.32703,1.31358,1.2992,1.28384,1.26744,1.24995,1.23132,1.21152,1.19048,1.16818,1.14458,1.11966,1.09339,1.06578,1.03682,1.00654,0.974972,0.942164,0.908189,0.873139,0.837129,0.800296,0.7628,0.724824,0.686571,0.648264,0.610138,0.572438,0.535409,0.499293,0.46431,0.430664,0.39852,0.368004,0.339199,0.312142,0.286825},
	{1.01465,1.01552,1.01643,1.01739,1.01841,1.01948,1.0206,1.02179,1.02303,1.02435,1.02573,1.02719,1.02872,1.03034,1.03204,1.03382,1.0357,1.03768,1.03976,1.04195,1.04425,1.04666,1.0492,1.05187,1.05466,1.0576,1.06068,1.06391,1.06729,1.07084,1.07455,1.07844,1.0825,1.08676,1.0912,1.09584,1.10069,1.10574,1.11101,1.1165,1.12222,1.12816,1.13434,1.14076,1.14743,1.15433,1.16149,1.1689,1.17656,1.18448,1.19264,1.20106,1.20974,1.21866,1.22782,1.23723,1.24687,1.25675,1.26684,1.27715,1.28766,1.29836,1.30924,1.32029,1.33149,1.34282,1.35427,1.36582,1.37745,1.38914,1.40086,1.41259,1.42432,1.436,1.44763,1.45916,1.47058,1.48186,1.49296,1.50386,1.51454,1.52495,1.53508,1.5449,1.55438,1.56349,1.57221,1.58051,1.58838,1.59578,1.60272,1.60915,1.61508,1.62048,1.62536,1.62969,1.63348,1.63673,1.63943,1.64159,1.64322,1.64433,1.64492,1.64502,1.64464,1.6438,1.64252,1.64083,1.63875,1.63631,1.63354,1.63047,1.62713,1.62354,1.61974,1.61576,1.61162,1.60735,1.60298,1.59854,1.59404,1.58952,1.58498,1.58045,1.57594,1.57146,1.56703,1.56265,1.55832,1.55406,1.54986,1.54572,1.54164,1.53761,1.53362,1.52967,1.52575,1.52183,1.51791,1.51398,1.51,1.50597,1.50187,1.49766,1.49333,1.48886,1.48422,1.47937,1.4743,1.46897,1.46335,1.4574,1.4511,1.4444,1.43727,1.42966,1.42154,1.41286,1.40357,1.39364,1.38301,1.37163,1.35945,1.34643,1.3325,1.31762,1.30173,1.28478,1.26672,1.24751,1.22709,1.20542,1.18247,1.15821,1.13261,1.10565,1.07734,1.04768,1.01669,0.984405,0.950893,0.916223,0.880492,0.843819,0.806348,0.768242,0.729688,0.690896,0.652091,0.613512,0.575404,0.538015,0.501584,0.466332,0.432457,0.40012,0.369442,0.3405,0.313325,0.287903},
	{1.01455,1.01542,1.01633,1.01729,1.0183,1.01936,1.02048,1.02166,1.02291,1.02422,1.0256,1.02705,1.02858,1.03019,1.03188,1.03366,1.03554,1.03751,1.03959,1.04177,1.04407,1.04648,1.04901,1.05168,1.05447,1.05741,1.06049,1.06372,1.06711,1.07066,1.07438,1.07827,1.08235,1.08661,1.09107,1.09573,1.10059,1.10567,1.11097,1.11649,1.12224,1.12822,1.13445,1.14092,1.14764,1.15461,1.16183,1.16932,1.17706,1.18507,1.19333,1.20186,1.21066,1.21971,1.22901,1.23857,1.24837,1.25842,1.2687,1.27921,1.28993,1.30086,1.31198,1.32328,1.33475,1.34636,1.35811,1.36997,1.38193,1.39396,1.40604,1.41814,1.43025,1.44234,1.45438,1.46634,1.4782,1.48993,1.5015,1.51287,1.52403,1.53494,1.54557,1.5559,1.56589,1.57552,1.58477,1.5936,1.60199,1.60993,1.61739,1.62435,1.6308,1.63672,1.64211,1.64695,1.65123,1.65497,1.65814,1.66077,1.66285,1.66439,1.66541,1.66592,1.66593,1.66547,1.66456,1.66321,1.66146,1.65934,1.65687,1.65408,1.651,1.64766,1.64409,1.64032,1.63638,1.63229,1.6281,1.62381,1.61945,1.61506,1.61063,1.6062,1.60178,1.59738,1.59301,1.58869,1.58441,1.58018,1.576,1.57188,1.5678,1.56376,1.55976,1.55578,1.55182,1.54785,1.54388,1.53987,1.53582,1.5317,1.5275,1.52319,1.51874,1.51413,1.50935,1.50434,1.4991,1.49359,1.48777,1.48161,1.47508,1.46814,1.46075,1.45287,1.44445,1.43546,1.42585,1.41557,1.40457,1.39281,1.38023,1.36678,1.3524,1.33706,1.32068,1.30323,1.28465,1.26489,1.24391,1.22167,1.19812,1.17325,1.14702,1.11943,1.09048,1.06016,1.02852,0.995585,0.961419,0.926101,0.889732,0.852433,0.814353,0.775658,0.73654,0.697211,0.657899,0.618845,0.580298,0.542504,0.505704,0.470116,0.435937,0.403325,0.372399,0.34323,0.315845,0.290228},
	{1.01445,1.01531,1.01622,1.01718,1.01818,1.01924,1.02036,1.02153,1.02277,1.02408,1.02545,1.0269,1.02842,1.03003,1.03171,1.03349,1.03536,1.03733,1.0394,1.04158,1.04387,1.04628,1.04881,1.05147,1.05427,1.0572,1.06028,1.06351,1.0669,1.07045,1.07417,1.07807,1.08216,1.08643,1.0909,1.09557,1.10046,1.10556,1.11088,1.11642,1.12221,1.12823,1.13449,1.14101,1.14778,1.15481,1.1621,1.16965,1.17748,1.18557,1.19394,1.20257,1.21148,1.22065,1.23009,1.2398,1.24976,1.25998,1.27044,1.28114,1.29207,1.30322,1.31458,1.32613,1.33786,1.34976,1.3618,1.37398,1.38626,1.39863,1.41106,1.42354,1.43604,1.44853,1.46098,1.47337,1.48568,1.49786,1.50989,1.52175,1.5334,1.54481,1.55595,1.5668,1.57731,1.58747,1.59725,1.60662,1.61556,1.62404,1.63203,1.63954,1.64652,1.65298,1.65889,1.66424,1.66904,1.67328,1.67695,1.68005,1.6826,1.6846,1.68606,1.68699,1.68742,1.68735,1.68681,1.68583,1.68443,1.68263,1.68047,1.67797,1.67516,1.67208,1.66876,1.66521,1.66148,1.65759,1.65357,1.64945,1.64524,1.64098,1.63667,1.63235,1.62802,1.62371,1.61941,1.61514,1.6109,1.60671,1.60255,1.59844,1.59436,1.59031,1.58629,1.58228,1.57827,1.57426,1.57022,1.56614,1.562,1.55779,1.55348,1.54904,1.54446,1.53971,1.53476,1.52959,1.52416,1.51845,1.51241,1.50602,1.49925,1.49204,1.48437,1.47618,1.46745,1.45812,1.44815,1.43749,1.4261,1.41392,1.40089,1.38698,1.37213,1.35627,1.33937,1.32137,1.30222,1.28188,1.26029,1.23742,1.21323,1.1877,1.16081,1.13254,1.10289,1.07188,1.03954,1.0059,0.971046,0.935044,0.898004,0.860051,0.821338,0.782038,0.742344,0.702473,0.662657,0.623139,0.584169,0.545995,0.508854,0.472967,0.438525,0.405683,0.374554,0.345205,0.317659,0.291894},
	{1.01435,1.01521,1.01611,1.01706,1.01806,1.01912,1.02023,1.0214,1.02263,1.02393,1.0253,1.02674,1.02826,1.02986,1.03154,1.03331,1.03518,1.03714,1.03921,1.04138,1.04366,1.04607,1.04859,1.05125,1.05404,1.05697,1.06005,1.06328,1.06666,1.07022,1.07394,1.07785,1.08194,1.08622,1.0907,1.09538,1.10028,1.1054,1.11074,1.11631,1.12212,1.12817,1.13448,1.14104,1.14785,1.15494,1.16229,1.16991,1.17781,1.18599,1.19444,1.20318,1.21219,1.22149,1.23106,1.2409,1.25102,1.2614,1.27204,1.28293,1.29407,1.30543,1.31702,1.32882,1.34081,1.35299,1.36532,1.3778,1.39041,1.40311,1.4159,1.42875,1.44163,1.45452,1.46739,1.48021,1.49296,1.5056,1.5181,1.53044,1.54259,1.5545,1.56616,1.57752,1.58857,1.59927,1.60958,1.6195,1.62898,1.63801,1.64656,1.65461,1.66214,1.66913,1.67558,1.68147,1.68679,1.69154,1.69571,1.69931,1.70233,1.7048,1.70671,1.70808,1.70892,1.70926,1.70911,1.70849,1.70744,1.70598,1.70413,1.70193,1.69941,1.69659,1.6935,1.69019,1.68666,1.68297,1.67912,1.67515,1.67109,1.66695,1.66276,1.65853,1.65428,1.65003,1.64578,1.64155,1.63734,1.63315,1.62899,1.62485,1.62074,1.61665,1.61256,1.60848,1.60438,1.60026,1.5961,1.59189,1.5876,1.58322,1.57872,1.57408,1.56928,1.56429,1.55908,1.55362,1.54789,1.54184,1.53546,1.52869,1.52151,1.51387,1.50573,1.49706,1.48781,1.47792,1.46737,1.45609,1.44404,1.43117,1.41743,1.40275,1.3871,1.37041,1.35265,1.33374,1.31366,1.29234,1.26975,1.24586,1.22062,1.19402,1.16604,1.13667,1.10592,1.07382,1.04039,1.00569,0.969799,0.932802,0.894817,0.855983,0.81646,0.776434,0.736109,0.69571,0.655475,0.615652,0.576492,0.538243,0.501134,0.465377,0.43115,0.39859,0.367793,0.338808,0.311639,0.286249},
	{1.01424,1.0151,1.01599,1.01694,1.01794,1.01899,1.02009,1.02126,1.02249,1.02378,1.02514,1.02658,1.02809,1.02968,1.03136,1.03312,1.03498,1.03694,1.03899,1.04116,1.04344,1.04584,1.04836,1.05101,1.05379,1.05672,1.05979,1.06302,1.0664,1.06996,1.07368,1.07759,1.08168,1.08597,1.09046,1.09515,1.10006,1.10519,1.11055,1.11614,1.12198,1.12806,1.1344,1.141,1.14786,1.155,1.1624,1.17009,1.17806,1.18632,1.19486,1.20369,1.21281,1.22222,1.23192,1.2419,1.25216,1.2627,1.27352,1.2846,1.29593,1.30751,1.31933,1.33137,1.34363,1.35607,1.3687,1.38148,1.39441,1.40746,1.4206,1.43382,1.44709,1.46038,1.47367,1.48692,1.50012,1.51322,1.5262,1.53903,1.55168,1.5641,1.57628,1.58818,1.59977,1.61102,1.62189,1.63237,1.64241,1.65201,1.66112,1.66974,1.67783,1.68539,1.69239,1.69883,1.70469,1.70998,1.71467,1.71878,1.72231,1.72526,1.72765,1.72948,1.73076,1.73153,1.73179,1.73157,1.73089,1.72979,1.72828,1.7264,1.72418,1.72164,1.71883,1.71576,1.71247,1.70898,1.70534,1.70155,1.69765,1.69366,1.6896,1.68549,1.68135,1.67719,1.67302,1.66885,1.66469,1.66054,1.65641,1.65228,1.64817,1.64407,1.63996,1.63584,1.6317,1.62752,1.6233,1.619,1.61462,1.61012,1.60551,1.60073,1.59579,1.59063,1.58525,1.5796,1.57367,1.5674,1.56078,1.55376,1.5463,1.53838,1.52994,1.52094,1.51134,1.5011,1.49016,1.47848,1.466,1.45268,1.43847,1.4233,1.40714,1.38992,1.3716,1.35212,1.33144,1.30951,1.28629,1.26174,1.23584,1.20856,1.17989,1.14982,1.11837,1.08555,1.05141,1.016,0.97941,0.941723,0.903063,0.863574,0.823421,0.782793,0.741898,0.700964,0.660234,0.619956,0.580384,0.541763,0.504324,0.468275,0.43379,0.401003,0.370006,0.34084,0.313505,0.287962},
	{1.01413,1.01498,1.01588,1.01682,1.01781,1.01885,1.01995,1.02111,1.02233,1.02362,1.02497,1.0264,1.02791,1.02949,1.03116,1.03292,1.03477,1.03672,1.03877,1.04093,1.0432,1.04559,1.0481,1.05075,1.05353,1.05645,1.05951,1.06274,1.06612,1.06967,1.07339,1.0773,1.08139,1.08568,1.09018,1.09488,1.0998,1.10494,1.11031,1.11593,1.12178,1.12789,1.13426,1.1409,1.1478,1.15498,1.16244,1.17019,1.17822,1.18655,1.19518,1.2041,1.21332,1.22284,1.23266,1.24277,1.25318,1.26388,1.27486,1.28612,1.29765,1.30944,1.32149,1.33377,1.34627,1.35899,1.3719,1.38499,1.39824,1.41162,1.42512,1.43871,1.45236,1.46606,1.47976,1.49345,1.5071,1.52067,1.53413,1.54745,1.5606,1.57355,1.58626,1.59869,1.61083,1.62264,1.63407,1.64512,1.65574,1.66591,1.6756,1.6848,1.69347,1.7016,1.70918,1.71618,1.7226,1.72843,1.73367,1.73831,1.74236,1.74581,1.74869,1.75099,1.75274,1.75394,1.75463,1.75482,1.75453,1.7538,1.75264,1.7511,1.74919,1.74695,1.74442,1.74161,1.73856,1.7353,1.73186,1.72826,1.72453,1.7207,1.71678,1.71279,1.70875,1.70468,1.70059,1.69649,1.69238,1.68826,1.68415,1.68004,1.67592,1.6718,1.66766,1.6635,1.6593,1.65506,1.65074,1.64635,1.64186,1.63724,1.63249,1.62756,1.62244,1.61711,1.61152,1.60566,1.59949,1.59298,1.58608,1.57878,1.57102,1.56276,1.55397,1.54461,1.53462,1.52396,1.51259,1.50045,1.48749,1.47367,1.45892,1.4432,1.42646,1.40864,1.38969,1.36956,1.34821,1.32559,1.30166,1.27638,1.24973,1.22169,1.19225,1.1614,1.12916,1.09555,1.06063,1.02445,0.987091,0.948658,0.909277,0.869093,0.828281,0.787033,0.745562,0.704102,0.662896,0.622198,0.58226,0.543327,0.505628,0.469367,0.434711,0.401789,0.370685,0.341433,0.314027,0.28842},
	{1.01402,1.01486,1.01575,1.01669,1.01767,1.01871,1.0198,1.02096,1.02217,1.02345,1.0248,1.02622,1.02772,1.02929,1.03096,1.03271,1.03455,1.03649,1.03853,1.04068,1.04295,1.04533,1.04783,1.05047,1.05324,1.05615,1.05922,1.06243,1.06581,1.06935,1.07308,1.07698,1.08107,1.08536,1.08986,1.09456,1.09949,1.10464,1.11003,1.11566,1.12153,1.12767,1.13406,1.14073,1.14767,1.15489,1.1624,1.1702,1.1783,1.1867,1.1954,1.20441,1.21373,1.22335,1.23329,1.24353,1.25407,1.26492,1.27607,1.2875,1.29923,1.31122,1.32348,1.336,1.34876,1.36174,1.37494,1.38833,1.40189,1.41561,1.42946,1.44341,1.45745,1.47155,1.48568,1.4998,1.5139,1.52794,1.54188,1.5557,1.56936,1.58283,1.59608,1.60907,1.62176,1.63413,1.64615,1.65777,1.66898,1.67974,1.69002,1.69981,1.70908,1.7178,1.72596,1.73354,1.74054,1.74694,1.75273,1.75792,1.7625,1.76648,1.76987,1.77266,1.77489,1.77656,1.77769,1.7783,1.77843,1.77808,1.7773,1.77611,1.77453,1.77261,1.77036,1.76783,1.76503,1.76201,1.75879,1.75539,1.75185,1.74818,1.74441,1.74056,1.73664,1.73267,1.72867,1.72464,1.72058,1.71652,1.71243,1.70834,1.70423,1.7001,1.69593,1.69174,1.68749,1.68318,1.67879,1.67431,1.66972,1.66499,1.6601,1.65503,1.64975,1.64425,1.63847,1.63241,1.62602,1.61926,1.61212,1.60454,1.59649,1.58793,1.57881,1.5691,1.55875,1.5477,1.53592,1.52335,1.50995,1.49565,1.48041,1.46418,1.4469,1.42853,1.409,1.38827,1.3663,1.34304,1.31846,1.29251,1.26518,1.23644,1.20628,1.17472,1.14175,1.10743,1.07178,1.03488,0.996806,0.957674,0.91761,0.876766,0.835318,0.793465,0.751422,0.709426,0.667725,0.626572,0.586222,0.546919,0.50889,0.472336,0.437421,0.40427,0.372961,0.343525,0.315949,0.290185},
	{1.0139,1.01474,1.01562,1.01655,1.01753,1.01856,1.01965,1.0208,1.022,1.02328,1.02462,1.02603,1.02752,1.02909,1.03074,1.03248,1.03432,1.03625,1.03828,1.04042,1.04268,1.04505,1.04755,1.05017,1.05294,1.05584,1.05889,1.0621,1.06547,1.06901,1.07273,1.07663,1.08072,1.08501,1.0895,1.09421,1.09914,1.1043,1.10969,1.11533,1.12123,1.12738,1.1338,1.14049,1.14746,1.15472,1.16228,1.17013,1.17828,1.18675,1.19552,1.20461,1.21402,1.22375,1.23379,1.24415,1.25483,1.26583,1.27713,1.28874,1.30064,1.31284,1.32531,1.33806,1.35106,1.36431,1.37778,1.39147,1.40534,1.41939,1.43359,1.44791,1.46233,1.47683,1.49138,1.50594,1.52049,1.53499,1.54942,1.56374,1.57792,1.59192,1.6057,1.61924,1.6325,1.64545,1.65804,1.67026,1.68206,1.69342,1.70431,1.7147,1.72457,1.73389,1.74265,1.75083,1.75842,1.7654,1.77177,1.77752,1.78265,1.78717,1.79108,1.79438,1.7971,1.79925,1.80083,1.80189,1.80243,1.80249,1.80209,1.80126,1.80003,1.79843,1.79649,1.79423,1.7917,1.78893,1.78593,1.78274,1.77938,1.77588,1.77226,1.76855,1.76475,1.76088,1.75696,1.75299,1.74899,1.74496,1.7409,1.73681,1.73269,1.72853,1.72433,1.72008,1.71576,1.71137,1.70688,1.70228,1.69755,1.69268,1.68762,1.68238,1.6769,1.67118,1.66517,1.65885,1.65219,1.64514,1.63768,1.62977,1.62137,1.61243,1.60291,1.59277,1.58197,1.57045,1.55816,1.54507,1.53111,1.51623,1.50039,1.48353,1.46559,1.44653,1.42629,1.40483,1.3821,1.35806,1.33267,1.3059,1.27773,1.24814,1.21713,1.1847,1.15086,1.11567,1.07917,1.04142,1.00253,0.962595,0.921765,0.880194,0.838065,0.795582,0.752968,0.710463,0.668318,0.626788,0.586128,0.546582,0.50837,0.471689,0.436695,0.403502,0.37218,0.342749,0.315191,0.289448},
	{1.01378,1.01461,1.01549,1.01641,1.01739,1.01841,1.01949,1.02063,1.02183,1.0231,1.02443,1.02583,1.02731,1.02887,1.03052,1.03225,1.03407,1.03599,1.03802,1.04015,1.04239,1.04476,1.04724,1.04986,1.05261,1.0555,1.05855,1.06175,1.06511,1.06864,1.07235,1.07624,1.08033,1.08461,1.0891,1.09381,1.09874,1.1039,1.10931,1.11496,1.12086,1.12703,1.13347,1.14018,1.14719,1.15448,1.16207,1.16997,1.17818,1.1867,1.19554,1.20471,1.2142,1.22402,1.23417,1.24465,1.25546,1.26659,1.27805,1.28982,1.30191,1.3143,1.32698,1.33995,1.3532,1.3667,1.38045,1.39443,1.40861,1.42299,1.43753,1.45222,1.46703,1.48192,1.49689,1.51189,1.52689,1.54187,1.55679,1.57161,1.58631,1.60084,1.61517,1.62928,1.64311,1.65664,1.66983,1.68264,1.69505,1.70703,1.71853,1.72954,1.74003,1.74998,1.75936,1.76815,1.77635,1.78393,1.79089,1.79722,1.80293,1.80801,1.81246,1.8163,1.81953,1.82218,1.82425,1.82577,1.82676,1.82724,1.82724,1.8268,1.82593,1.82467,1.82306,1.82111,1.81887,1.81635,1.8136,1.81064,1.80749,1.80418,1.80073,1.79717,1.7935,1.78976,1.78594,1.78206,1.77813,1.77416,1.77014,1.76607,1.76197,1.75781,1.75359,1.74931,1.74495,1.7405,1.73595,1.73127,1.72644,1.72145,1.71628,1.71089,1.70526,1.69937,1.69318,1.68666,1.67978,1.67251,1.66481,1.65663,1.64794,1.63871,1.62888,1.61841,1.60725,1.59536,1.58269,1.56919,1.55481,1.53949,1.52318,1.50583,1.48739,1.46781,1.44703,1.42501,1.40171,1.37708,1.35108,1.32369,1.29489,1.26466,1.23299,1.19989,1.16539,1.12952,1.09234,1.05393,1.01436,0.973774,0.932295,0.890089,0.847342,0.804261,0.761071,0.718016,0.675349,0.633326,0.592203,0.552222,0.513606,0.476549,0.441204,0.407686,0.376059,0.346343,0.318515,0.292518},
	{1.01365,1.01448,1.01535,1.01627,1.01723,1.01825,1.01933,1.02046,1.02165,1.02291,1.02423,1.02563,1.0271,1.02865,1.03028,1.032,1.03382,1.03573,1.03774,1.03986,1.04209,1.04444,1.04692,1.04952,1.05226,1.05515,1.05818,1.06137,1.06472,1.06824,1.07194,1.07582,1.0799,1.08418,1.08867,1.09337,1.0983,1.10346,1.10887,1.11452,1.12044,1.12662,1.13307,1.13981,1.14683,1.15416,1.16178,1.16972,1.17798,1.18656,1.19546,1.2047,1.21427,1.22418,1.23443,1.24501,1.25594,1.26721,1.27881,1.29074,1.303,1.31558,1.32847,1.34166,1.35514,1.36889,1.38291,1.39717,1.41166,1.42636,1.44124,1.45629,1.47147,1.48677,1.50215,1.51758,1.53304,1.54848,1.56389,1.57922,1.59443,1.6095,1.62439,1.63905,1.65346,1.66758,1.68136,1.69479,1.70781,1.72041,1.73254,1.74418,1.7553,1.76587,1.77588,1.7853,1.79411,1.80231,1.80988,1.81681,1.8231,1.82875,1.83376,1.83814,1.8419,1.84506,1.84762,1.84961,1.85105,1.85197,1.85239,1.85234,1.85184,1.85094,1.84965,1.84801,1.84605,1.8438,1.8413,1.83856,1.83561,1.83249,1.8292,1.82578,1.82225,1.81861,1.81488,1.81107,1.8072,1.80326,1.79926,1.79519,1.79107,1.78688,1.78261,1.77827,1.77382,1.76927,1.7646,1.75978,1.7548,1.74964,1.74427,1.73867,1.73282,1.72668,1.72022,1.71341,1.70621,1.69861,1.69054,1.68198,1.67289,1.66322,1.65293,1.64198,1.63031,1.61788,1.60464,1.59055,1.57554,1.55956,1.54257,1.52451,1.50533,1.48498,1.46341,1.44057,1.41642,1.39092,1.36404,1.33574,1.30602,1.27485,1.24223,1.20819,1.17275,1.13594,1.09784,1.05851,1.01807,0.97664,0.93436,0.891403,0.847963,0.804254,0.760506,0.716968,0.673897,0.63155,0.590181,0.550031,0.511315,0.474219,0.438887,0.405419,0.373872,0.344252,0.316527,0.290632},
	{1.01352,1.01434,1.01521,1.01612,1.01708,1.01809,1.01916,1.02028,1.02146,1.02271,1.02403,1.02541,1.02687,1.02841,1.03004,1.03175,1.03355,1.03545,1.03745,1.03956,1.04178,1.04412,1.04658,1.04917,1.0519,1.05477,1.05779,1.06096,1.0643,1.06781,1.0715,1.07537,1.07944,1.08371,1.08819,1.09289,1.09781,1.10297,1.10838,1.11403,1.11995,1.12614,1.13261,1.13936,1.1464,1.15375,1.16141,1.16939,1.17768,1.18631,1.19527,1.20457,1.21422,1.22421,1.23455,1.24524,1.25629,1.26768,1.27942,1.29151,1.30393,1.31669,1.32978,1.34318,1.35689,1.37089,1.38517,1.39972,1.41451,1.42953,1.44475,1.46015,1.47571,1.49141,1.5072,1.52307,1.53898,1.5549,1.57079,1.58663,1.60237,1.61798,1.63343,1.64866,1.66366,1.67837,1.69277,1.70681,1.72047,1.7337,1.74647,1.75875,1.77052,1.78174,1.7924,1.80246,1.81192,1.82075,1.82894,1.83649,1.84339,1.84964,1.85523,1.86018,1.8645,1.86819,1.87127,1.87376,1.87569,1.87706,1.87792,1.87829,1.87819,1.87766,1.87673,1.87542,1.87378,1.87182,1.86958,1.86709,1.86438,1.86146,1.85837,1.85512,1.85173,1.84823,1.84461,1.84091,1.83711,1.83324,1.82928,1.82525,1.82115,1.81696,1.81268,1.8083,1.80381,1.7992,1.79445,1.78955,1.78446,1.77918,1.77368,1.76793,1.76191,1.75559,1.74893,1.74191,1.73449,1.72663,1.7183,1.70946,1.70007,1.69009,1.67946,1.66816,1.65612,1.6433,1.62965,1.61513,1.59967,1.58323,1.56575,1.54719,1.52749,1.50659,1.48446,1.46104,1.4363,1.41019,1.38268,1.35374,1.32336,1.29153,1.25824,1.22352,1.18739,1.1499,1.11111,1.0711,1.02998,0.987881,0.944943,0.901344,0.857279,0.812966,0.768639,0.724549,0.680953,0.638111,0.596277,0.555693,0.516571,0.479098,0.443415,0.40962,0.377766,0.347858,0.319862,0.293711},
	{1.01339,1.0142,1.01506,1.01596,1.01692,1.01792,1.01898,1.02009,1.02127,1.02251,1.02381,1.02519,1.02664,1.02817,1.02978,1.03148,1.03327,1.03516,1.03715,1.03924,1.04145,1.04377,1.04622,1.0488,1.05151,1.05437,1.05737,1.06053,1.06386,1.06736,1.07103,1.07489,1.07895,1.0832,1.08767,1.09236,1.09728,1.10243,1.10784,1.11349,1.11941,1.1256,1.13208,1.13884,1.1459,1.15327,1.16095,1.16896,1.17729,1.18596,1.19497,1.20433,1.21405,1.22411,1.23454,1.24533,1.25648,1.26799,1.27987,1.2921,1.30468,1.31762,1.33089,1.3445,1.35843,1.37267,1.38721,1.40203,1.41712,1.43245,1.448,1.46375,1.47969,1.49577,1.51197,1.52827,1.54463,1.56102,1.57741,1.59375,1.61001,1.62617,1.64217,1.65798,1.67356,1.68888,1.70389,1.71855,1.73284,1.74671,1.76013,1.77307,1.78549,1.79737,1.80868,1.8194,1.82951,1.83899,1.84782,1.856,1.86352,1.87037,1.87656,1.88209,1.88697,1.89121,1.89482,1.89782,1.90023,1.90208,1.90338,1.90418,1.90448,1.90433,1.90376,1.90279,1.90146,1.8998,1.89783,1.89559,1.89309,1.89038,1.88747,1.88438,1.88114,1.87775,1.87424,1.87061,1.86688,1.86304,1.85911,1.85509,1.85097,1.84674,1.84241,1.83796,1.83339,1.82867,1.82379,1.81874,1.81349,1.80803,1.80232,1.79635,1.79008,1.78349,1.77655,1.76922,1.76146,1.75325,1.74454,1.7353,1.72548,1.71503,1.70393,1.69211,1.67954,1.66615,1.65192,1.63677,1.62066,1.60354,1.58536,1.56606,1.54559,1.52391,1.50096,1.4767,1.45109,1.4241,1.39568,1.36582,1.3345,1.30172,1.26748,1.2318,1.19472,1.15629,1.11657,1.07566,1.03366,0.990722,0.946991,0.902651,0.857905,0.812977,0.768108,0.723551,0.679567,0.636417,0.594354,0.553611,0.5144,0.476894,0.441226,0.407483,0.375704,0.345886,0.317984,0.291926},
	{1.01325,1.01406,1.01491,1.01581,1.01675,1.01775,1.01879,1.0199,1.02107,1.0223,1.02359,1.02496,1.0264,1.02791,1.02951,1.0312,1.03298,1.03485,1.03683,1.03891,1.0411,1.04341,1.04584,1.04841,1.0511,1.05394,1.05693,1.06008,1.06339,1.06687,1.07053,1.07437,1.07841,1.08266,1.08711,1.09179,1.0967,1.10184,1.10724,1.11289,1.11881,1.125,1.13147,1.13824,1.14532,1.1527,1.1604,1.16843,1.1768,1.18551,1.19457,1.20398,1.21375,1.22389,1.2344,1.24528,1.25653,1.26815,1.28015,1.29252,1.30526,1.31836,1.33182,1.34563,1.35978,1.37425,1.38904,1.40413,1.4195,1.43514,1.45102,1.46713,1.48343,1.4999,1.51652,1.53324,1.55005,1.56691,1.58379,1.60064,1.61744,1.63414,1.6507,1.66709,1.68327,1.6992,1.71483,1.73013,1.74507,1.75959,1.77368,1.78728,1.80038,1.81293,1.82492,1.83632,1.8471,1.85724,1.86674,1.87557,1.88373,1.89122,1.89803,1.90417,1.90963,1.91445,1.91861,1.92215,1.92508,1.92742,1.92921,1.93045,1.93119,1.93145,1.93127,1.93066,1.92967,1.92832,1.92665,1.92468,1.92245,1.91996,1.91726,1.91436,1.91129,1.90805,1.90467,1.90115,1.89751,1.89375,1.88987,1.88588,1.88178,1.87756,1.87322,1.86874,1.86412,1.85934,1.85438,1.84923,1.84388,1.83828,1.83243,1.8263,1.81986,1.81307,1.80592,1.79836,1.79036,1.78189,1.7729,1.76336,1.75322,1.74245,1.73099,1.7188,1.70584,1.69205,1.67738,1.66178,1.64521,1.62761,1.60892,1.5891,1.56809,1.54584,1.52231,1.49746,1.47124,1.44361,1.41455,1.38404,1.35205,1.31859,1.28367,1.2473,1.20952,1.17039,1.12997,1.08837,1.04568,1.00207,0.957669,0.912681,0.867305,0.821769,0.776317,0.731204,0.686693,0.643046,0.600515,0.559336,0.519718,0.481832,0.44581,0.411735,0.379646,0.349535,0.321357,0.295038},
	{1.0131,1.01391,1.01475,1.01564,1.01658,1.01757,1.01861,1.0197,1.02086,1.02208,1.02336,1.02472,1.02614,1.02765,1.02924,1.03091,1.03267,1.03453,1.03649,1.03856,1.04074,1.04303,1.04545,1.04799,1.05067,1.0535,1.05647,1.0596,1.06289,1.06635,1.06999,1.07382,1.07785,1.08207,1.08651,1.09118,1.09607,1.1012,1.10659,1.11223,1.11814,1.12433,1.1308,1.13757,1.14465,1.15205,1.15976,1.16782,1.17621,1.18495,1.19404,1.2035,1.21333,1.22353,1.23411,1.24507,1.25642,1.26815,1.28027,1.29277,1.30565,1.31891,1.33254,1.34654,1.3609,1.3756,1.39063,1.40598,1.42164,1.43757,1.45378,1.47022,1.48689,1.50374,1.52076,1.53791,1.55517,1.5725,1.58986,1.60722,1.62454,1.64179,1.65892,1.67589,1.69267,1.7092,1.72547,1.74141,1.75699,1.77218,1.78693,1.80122,1.81499,1.82824,1.84091,1.85299,1.86446,1.87528,1.88545,1.89495,1.90377,1.9119,1.91935,1.9261,1.93218,1.93758,1.94231,1.9464,1.94986,1.95271,1.95498,1.95669,1.95787,1.95855,1.95876,1.95853,1.95789,1.95687,1.9555,1.9538,1.95182,1.94957,1.94708,1.94436,1.94145,1.93836,1.9351,1.93168,1.92812,1.92442,1.92058,1.91661,1.91251,1.90827,1.90389,1.89936,1.89466,1.88978,1.88471,1.87943,1.87392,1.86816,1.86212,1.85578,1.8491,1.84207,1.83464,1.82679,1.81848,1.80967,1.80032,1.7904,1.77986,1.76866,1.75675,1.74409,1.73063,1.71631,1.7011,1.68493,1.66776,1.64953,1.6302,1.6097,1.588,1.56503,1.54076,1.51514,1.48814,1.45971,1.42983,1.39848,1.36565,1.33133,1.29555,1.25831,1.21967,1.17968,1.13842,1.09599,1.0525,1.00811,0.962971,0.91728,0.871249,0.825106,0.779103,0.733498,0.688554,0.644535,0.601692,0.560257,0.520432,0.482386,0.44624,0.412072,0.37991,0.349741,0.321512,0.295148},
	{1.01296,1.01375,1.01459,1.01547,1.0164,1.01738,1.01841,1.0195,1.02064,1.02185,1.02312,1.02447,1.02588,1.02737,1.02895,1.03061,1.03236,1.0342,1.03615,1.0382,1.04036,1.04263,1.04503,1.04756,1.05022,1.05303,1.05598,1.05909,1.06236,1.0658,1.06943,1.07323,1.07724,1.08145,1.08587,1.09052,1.09539,1.10051,1.10588,1.11151,1.11741,1.12359,1.13006,1.13683,1.14391,1.15131,1.15903,1.1671,1.17551,1.18428,1.1934,1.2029,1.21278,1.22304,1.23368,1.24472,1.25615,1.26798,1.2802,1.29283,1.30585,1.31926,1.33306,1.34724,1.3618,1.37672,1.39198,1.40759,1.42352,1.43975,1.45627,1.47305,1.49007,1.50731,1.52473,1.5423,1.56,1.57779,1.59564,1.61351,1.63136,1.64916,1.66685,1.68441,1.70179,1.71895,1.73585,1.75244,1.76869,1.78455,1.79998,1.81495,1.82943,1.84337,1.85675,1.86953,1.8817,1.89322,1.90409,1.91428,1.92377,1.93258,1.94068,1.94808,1.95478,1.96079,1.96612,1.97078,1.97479,1.97818,1.98096,1.98316,1.98481,1.98593,1.98656,1.98672,1.98645,1.98577,1.98472,1.98333,1.98162,1.97962,1.97736,1.97485,1.97213,1.96919,1.96607,1.96277,1.95931,1.95569,1.95192,1.94799,1.94391,1.93967,1.93527,1.9307,1.92595,1.921,1.91585,1.91046,1.90482,1.89892,1.89272,1.88619,1.87932,1.87208,1.86442,1.85631,1.84773,1.83863,1.82898,1.81873,1.80785,1.79628,1.78399,1.77092,1.75703,1.74227,1.72659,1.70993,1.69225,1.6735,1.65361,1.63255,1.61026,1.58668,1.56179,1.53553,1.50786,1.47876,1.44819,1.41614,1.38259,1.34755,1.31103,1.27306,1.23368,1.19296,1.15096,1.1078,1.0636,1.0185,0.972673,0.926318,0.879646,0.832893,0.78631,0.740158,0.694704,0.65021,0.606929,0.56509,0.524896,0.486511,0.450054,0.415598,0.383169,0.352749,0.324285,0.297698},
	{1.01281,1.01359,1.01442,1.0153,1.01622,1.01719,1.01821,1.01928,1.02042,1.02162,1.02288,1.02421,1.02561,1.02709,1.02865,1.0303,1.03203,1.03386,1.03579,1.03782,1.03996,1.04222,1.0446,1.04711,1.04975,1.05254,1.05547,1.05856,1.06181,1.06523,1.06883,1.07261,1.0766,1.08078,1.08519,1.08981,1.09467,1.09977,1.10512,1.11073,1.11662,1.12278,1.12924,1.13601,1.14308,1.15048,1.15821,1.16628,1.17471,1.18349,1.19265,1.20218,1.21209,1.2224,1.2331,1.24421,1.25572,1.26763,1.27996,1.2927,1.30585,1.3194,1.33336,1.34772,1.36246,1.37759,1.39308,1.40894,1.42513,1.44165,1.45847,1.47558,1.49295,1.51056,1.52837,1.54636,1.5645,1.58275,1.60108,1.61945,1.63783,1.65617,1.67443,1.69258,1.71056,1.72834,1.74588,1.76312,1.78003,1.79657,1.81269,1.82836,1.84354,1.85819,1.87228,1.88578,1.89866,1.9109,1.92247,1.93336,1.94355,1.95303,1.9618,1.96986,1.9772,1.98384,1.98978,1.99503,1.99961,2.00354,2.00685,2.00955,2.01167,2.01325,2.01431,2.01487,2.01498,2.01466,2.01394,2.01285,2.01142,2.00967,2.00764,2.00534,2.00279,2.00002,1.99703,1.99385,1.99048,1.98693,1.9832,1.9793,1.97522,1.97097,1.96653,1.9619,1.95707,1.95203,1.94675,1.94123,1.93543,1.92935,1.92294,1.9162,1.90909,1.90158,1.89363,1.88522,1.87631,1.86686,1.85684,1.84619,1.83488,1.82287,1.81011,1.79655,1.78214,1.76684,1.75059,1.73335,1.71505,1.69566,1.67511,1.65335,1.63035,1.60604,1.58039,1.55335,1.52488,1.49496,1.46356,1.43066,1.39626,1.36036,1.32297,1.28412,1.24388,1.20229,1.15944,1.11545,1.07044,1.02457,0.978003,0.930949,0.883625,0.836271,0.789144,0.742505,0.696625,0.651765,0.608177,0.566086,0.52569,0.487145,0.450564,0.416011,0.383505,0.353021,0.324501,0.297863},
	{1.01265,1.01343,1.01425,1.01512,1.01603,1.01699,1.018,1.01907,1.02019,1.02137,1.02262,1.02394,1.02533,1.0268,1.02834,1.02997,1.03169,1.0335,1.03541,1.03743,1.03955,1.04179,1.04415,1.04664,1.04926,1.05202,1.05493,1.05799,1.06122,1.06462,1.06819,1.07196,1.07592,1.08008,1.08446,1.08906,1.09389,1.09897,1.1043,1.10989,1.11576,1.12191,1.12835,1.1351,1.14217,1.14956,1.15729,1.16537,1.1738,1.1826,1.19177,1.20132,1.21127,1.22162,1.23237,1.24353,1.25511,1.26711,1.27953,1.29238,1.30564,1.31933,1.33344,1.34796,1.36289,1.37821,1.39393,1.41002,1.42647,1.44326,1.46039,1.47782,1.49553,1.5135,1.5317,1.5501,1.56867,1.58738,1.60619,1.62507,1.64397,1.66285,1.68168,1.70042,1.71901,1.73741,1.75559,1.77349,1.79108,1.8083,1.82513,1.8415,1.8574,1.87277,1.88759,1.90182,1.91543,1.9284,1.94069,1.9523,1.96321,1.97339,1.98286,1.99159,1.9996,2.00689,2.01345,2.01932,2.02449,2.02899,2.03285,2.03607,2.0387,2.04075,2.04226,2.04325,2.04376,2.04381,2.04344,2.04268,2.04155,2.04007,2.03829,2.03621,2.03387,2.03127,2.02844,2.02539,2.02212,2.01865,2.01499,2.01113,2.00708,2.00282,1.99837,1.9937,1.98881,1.98369,1.97831,1.97267,1.96673,1.96049,1.95391,1.94697,1.93964,1.9319,1.9237,1.91501,1.9058,1.89604,1.88568,1.87468,1.86299,1.85058,1.8374,1.8234,1.80853,1.79275,1.77599,1.75822,1.73938,1.71941,1.69827,1.67591,1.65227,1.62732,1.60099,1.57327,1.5441,1.51345,1.48132,1.44767,1.41251,1.37583,1.33767,1.29805,1.25702,1.21466,1.17105,1.1263,1.08055,1.03395,0.986678,0.938948,0.89098,0.843017,0.795318,0.748149,0.701779,0.656472,0.612479,0.570022,0.529296,0.490456,0.453607,0.418811,0.386082,0.35539,0.326675,0.299852},
	{1.01249,1.01326,1.01408,1.01493,1.01583,1.01678,1.01779,1.01884,1.01995,1.02113,1.02236,1.02367,1.02504,1.02649,1.02802,1.02963,1.03134,1.03313,1.03502,1.03702,1.03912,1.04134,1.04368,1.04615,1.04874,1.05148,1.05437,1.05741,1.06061,1.06398,1.06753,1.07127,1.0752,1.07933,1.08368,1.08826,1.09307,1.09812,1.10342,1.10899,1.11483,1.12096,1.12739,1.13412,1.14117,1.14855,1.15627,1.16435,1.17278,1.18158,1.19077,1.20034,1.21031,1.22069,1.23149,1.2427,1.25434,1.26641,1.27891,1.29185,1.30523,1.31904,1.33328,1.34796,1.36306,1.37857,1.39449,1.41081,1.42751,1.44457,1.46198,1.47972,1.49777,1.51609,1.53467,1.55348,1.57247,1.59163,1.61092,1.63029,1.64971,1.66913,1.68853,1.70784,1.72704,1.74607,1.76489,1.78345,1.80172,1.81963,1.83716,1.85425,1.87087,1.88697,1.90252,1.91749,1.93184,1.94555,1.95858,1.97092,1.98255,1.99346,2.00363,2.01305,2.02174,2.02969,2.0369,2.04339,2.04917,2.05425,2.05867,2.06243,2.06557,2.06811,2.07007,2.0715,2.07242,2.07286,2.07284,2.0724,2.07158,2.07038,2.06885,2.06699,2.06485,2.06242,2.05974,2.05681,2.05364,2.05025,2.04663,2.0428,2.03875,2.03448,2.02999,2.02526,2.02028,2.01505,2.00954,2.00375,1.99764,1.9912,1.9844,1.97722,1.96963,1.96159,1.95308,1.94407,1.9345,1.92436,1.91359,1.90216,1.89003,1.87714,1.86346,1.84893,1.83351,1.81714,1.79978,1.78138,1.76188,1.74123,1.71939,1.69629,1.6719,1.64616,1.61904,1.59049,1.56048,1.52897,1.49596,1.46142,1.42536,1.38778,1.34871,1.30818,1.26625,1.223,1.17851,1.13291,1.08633,1.03894,0.990926,0.942496,0.893882,0.845331,0.797106,0.749475,0.702711,0.657075,0.612815,0.57015,0.529268,0.490314,0.453387,0.41854,0.385776,0.355061,0.326326,0.299488},
	{1.01233,1.01309,1.0139,1.01474,1.01563,1.01657,1.01756,1.01861,1.01971,1.02087,1.02209,1.02338,1.02474,1.02618,1.02769,1.02929,1.03097,1.03275,1.03462,1.03659,1.03868,1.04087,1.04319,1.04563,1.04821,1.05092,1.05378,1.05679,1.05997,1.06331,1.06683,1.07054,1.07444,1.07855,1.08287,1.08741,1.09219,1.09721,1.10248,1.10802,1.11384,1.11994,1.12635,1.13306,1.14009,1.14745,1.15516,1.16322,1.17165,1.18045,1.18964,1.19922,1.20921,1.21961,1.23044,1.2417,1.25339,1.26552,1.27809,1.29112,1.30459,1.31852,1.33289,1.34771,1.36297,1.37866,1.39478,1.41131,1.42825,1.44557,1.46326,1.4813,1.49967,1.51835,1.5373,1.5565,1.57592,1.59552,1.61527,1.63514,1.65507,1.67504,1.695,1.7149,1.73471,1.75436,1.77383,1.79306,1.812,1.83062,1.84886,1.86667,1.88403,1.90087,1.91718,1.9329,1.94801,1.96247,1.97626,1.98935,2.00173,2.01337,2.02427,2.03442,2.0438,2.05244,2.06032,2.06746,2.07387,2.07956,2.08456,2.08889,2.09256,2.09562,2.09807,2.09996,2.10131,2.10215,2.10252,2.10244,2.10193,2.10104,2.09978,2.09817,2.09624,2.09401,2.0915,2.08871,2.08567,2.08237,2.07884,2.07505,2.07103,2.06677,2.06226,2.05749,2.05245,2.04714,2.04153,2.03562,2.02937,2.02276,2.01578,2.0084,2.00058,1.9923,1.98353,1.97422,1.96435,1.95388,1.94277,1.93097,1.91844,1.90515,1.89103,1.87605,1.86015,1.84329,1.82542,1.80647,1.78642,1.76519,1.74274,1.71902,1.69399,1.66759,1.63979,1.61054,1.57981,1.54758,1.51382,1.47853,1.4417,1.40335,1.36349,1.32218,1.27947,1.23543,1.19017,1.14381,1.09648,1.04837,0.99965,0.950546,0.901289,0.852129,0.803335,0.755176,0.707925,0.661844,0.61718,0.57415,0.532939,0.493688,0.456492,0.421398,0.388406,0.357478,0.328543,0.301515},
	{1.01216,1.01292,1.01371,1.01455,1.01543,1.01636,1.01734,1.01837,1.01946,1.0206,1.02181,1.02309,1.02443,1.02585,1.02735,1.02893,1.03059,1.03235,1.0342,1.03616,1.03822,1.04039,1.04268,1.0451,1.04765,1.05033,1.05317,1.05615,1.05929,1.06261,1.0661,1.06977,1.07364,1.07772,1.082,1.08652,1.09126,1.09625,1.10149,1.107,1.11278,1.11885,1.12523,1.13191,1.13891,1.14625,1.15394,1.16199,1.1704,1.17919,1.18838,1.19796,1.20796,1.21838,1.22923,1.24052,1.25225,1.26443,1.27707,1.29018,1.30374,1.31776,1.33225,1.34721,1.36261,1.37847,1.39478,1.41152,1.42868,1.44625,1.46421,1.48254,1.50122,1.52023,1.53955,1.55914,1.57896,1.599,1.61922,1.63957,1.66002,1.68052,1.70104,1.72153,1.74193,1.76222,1.78234,1.80223,1.82186,1.84118,1.86014,1.87869,1.89678,1.91438,1.93145,1.94794,1.96381,1.97905,1.99361,2.00747,2.02061,2.033,2.04465,2.05553,2.06564,2.07498,2.08355,2.09136,2.09842,2.10474,2.11035,2.11526,2.11949,2.12307,2.12603,2.1284,2.1302,2.13147,2.13223,2.13251,2.13235,2.13177,2.13079,2.12945,2.12775,2.12573,2.12339,2.12076,2.11785,2.11466,2.1112,2.10747,2.10348,2.09923,2.0947,2.0899,2.0848,2.07941,2.0737,2.06766,2.06127,2.0545,2.04734,2.03974,2.0317,2.02317,2.01413,2.00454,1.99436,1.98356,1.97209,1.95992,1.947,1.93329,1.91874,1.9033,1.88692,1.86956,1.85116,1.83168,1.81106,1.78925,1.76619,1.74185,1.71617,1.68911,1.66062,1.63067,1.59923,1.56626,1.53176,1.49571,1.45811,1.41898,1.37834,1.33624,1.29274,1.24792,1.20189,1.15476,1.10669,1.05784,1.00842,0.958642,0.908741,0.858973,0.809609,0.760921,0.713182,0.666656,0.621587,0.578191,0.536649,0.4971,0.459632,0.424289,0.391066,0.359923,0.330785,0.303565},
	{1.01199,1.01274,1.01352,1.01435,1.01522,1.01614,1.0171,1.01812,1.0192,1.02033,1.02153,1.02279,1.02412,1.02552,1.027,1.02856,1.0302,1.03194,1.03377,1.0357,1.03774,1.03989,1.04215,1.04455,1.04707,1.04972,1.05253,1.05548,1.05859,1.06188,1.06533,1.06897,1.07281,1.07685,1.0811,1.08557,1.09028,1.09523,1.10044,1.10591,1.11165,1.11769,1.12403,1.13068,1.13765,1.14496,1.15262,1.16064,1.16904,1.17781,1.18699,1.19657,1.20657,1.21699,1.22786,1.23917,1.25093,1.26315,1.27585,1.28901,1.30265,1.31677,1.33136,1.34643,1.36198,1.37799,1.39447,1.4114,1.42877,1.44658,1.46479,1.4834,1.50239,1.52173,1.54139,1.56136,1.58159,1.60205,1.62272,1.64355,1.6645,1.68553,1.7066,1.72767,1.74868,1.76959,1.79035,1.81092,1.83123,1.85126,1.87093,1.89021,1.90906,1.92742,1.94525,1.96251,1.97917,1.99519,2.01053,2.02517,2.03908,2.05225,2.06465,2.07628,2.08713,2.09719,2.10646,2.11496,2.12269,2.12966,2.13588,2.14139,2.14619,2.15032,2.1538,2.15666,2.15893,2.16063,2.1618,2.16246,2.16265,2.16239,2.1617,2.16062,2.15916,2.15735,2.1552,2.15272,2.14993,2.14684,2.14346,2.13978,2.13582,2.13157,2.12702,2.12217,2.11701,2.11152,2.1057,2.09952,2.09296,2.08601,2.07863,2.07081,2.06252,2.05372,2.04438,2.03447,2.02395,2.01278,2.00093,1.98834,1.97499,1.96082,1.94579,1.92985,1.91295,1.89504,1.87607,1.85599,1.83475,1.81229,1.78857,1.76354,1.73716,1.70936,1.68013,1.64941,1.61718,1.58341,1.54809,1.51121,1.47277,1.43279,1.3913,1.34836,1.30401,1.25835,1.21149,1.16356,1.1147,1.0651,1.01496,0.964495,0.913954,0.863594,0.813688,0.764512,0.716338,0.66943,0.62403,0.58035,0.538567,0.498811,0.461166,0.425669,0.392309,0.361041,0.331786,0.304456},
	{1.01181,1.01255,1.01333,1.01414,1.015,1.01591,1.01687,1.01787,1.01893,1.02005,1.02123,1.02248,1.02379,1.02517,1.02663,1.02817,1.0298,1.03151,1.03332,1.03523,1.03724,1.03937,1.04161,1.04397,1.04646,1.04909,1.05186,1.05478,1.05786,1.06111,1.06453,1.06814,1.07193,1.07593,1.08014,1.08458,1.08924,1.09415,1.09932,1.10475,1.11046,1.11645,1.12275,1.12936,1.1363,1.14357,1.1512,1.15919,1.16755,1.17631,1.18546,1.19503,1.20502,1.21544,1.22631,1.23763,1.24941,1.26167,1.2744,1.28762,1.30132,1.31552,1.33021,1.34539,1.36106,1.37721,1.39385,1.41096,1.42854,1.44656,1.46502,1.4839,1.50317,1.52283,1.54283,1.56316,1.58378,1.60467,1.62578,1.64708,1.66853,1.69009,1.71171,1.73335,1.75496,1.7765,1.79791,1.81914,1.84015,1.86088,1.88128,1.90131,1.92091,1.94004,1.95866,1.97671,1.99416,2.01097,2.02712,2.04256,2.05726,2.07122,2.08441,2.09681,2.10842,2.11923,2.12923,2.13844,2.14686,2.1545,2.16138,2.16751,2.17291,2.17762,2.18164,2.18502,2.18778,2.18994,2.19155,2.19262,2.19318,2.19327,2.1929,2.19211,2.19092,2.18934,2.18739,2.18509,2.18246,2.1795,2.17622,2.17263,2.16872,2.16449,2.15995,2.15509,2.14988,2.14434,2.13843,2.13215,2.12547,2.11837,2.11083,2.10282,2.09432,2.08529,2.07571,2.06553,2.05472,2.04325,2.03107,2.01815,2.00443,1.98988,1.97445,1.95809,1.94075,1.92238,1.90293,1.88236,1.8606,1.83761,1.81334,1.78774,1.76076,1.73237,1.70251,1.67115,1.63827,1.60383,1.56783,1.53025,1.49111,1.45041,1.4082,1.36453,1.31945,1.27306,1.22547,1.17682,1.12724,1.07694,1.0261,0.974969,0.923777,0.872791,0.822286,0.772538,0.723823,0.676404,0.630523,0.586392,0.544186,0.504034,0.466016,0.430168,0.396476,0.364894,0.335341,0.30773},
	{1.01163,1.01236,1.01313,1.01393,1.01478,1.01568,1.01662,1.01761,1.01866,1.01976,1.02093,1.02216,1.02345,1.02482,1.02626,1.02778,1.02938,1.03108,1.03286,1.03474,1.03673,1.03883,1.04104,1.04337,1.04584,1.04843,1.05117,1.05406,1.0571,1.06031,1.06369,1.06726,1.07102,1.07498,1.07915,1.08354,1.08816,1.09302,1.09814,1.10353,1.10919,1.11514,1.12139,1.12796,1.13485,1.14208,1.14967,1.15762,1.16595,1.17467,1.1838,1.19334,1.20331,1.21372,1.22458,1.23591,1.2477,1.25998,1.27274,1.28599,1.29975,1.31401,1.32878,1.34405,1.35983,1.37612,1.3929,1.41018,1.42794,1.44617,1.46486,1.48399,1.50354,1.5235,1.54383,1.56451,1.58551,1.6068,1.62835,1.65011,1.67204,1.69412,1.71628,1.73849,1.76069,1.78285,1.8049,1.8268,1.8485,1.86994,1.89107,1.91184,1.9322,1.95211,1.9715,1.99035,2.00861,2.02623,2.04318,2.05943,2.07494,2.0897,2.10369,2.11687,2.12926,2.14083,2.15158,2.16151,2.17064,2.17897,2.18651,2.19328,2.1993,2.2046,2.20918,2.2131,2.21636,2.219,2.22105,2.22254,2.22349,2.22393,2.22389,2.2234,2.22247,2.22113,2.2194,2.21728,2.2148,2.21197,2.20879,2.20527,2.20141,2.19721,2.19267,2.18778,2.18253,2.17691,2.17091,2.1645,2.15768,2.15042,2.14269,2.13448,2.12575,2.11647,2.10661,2.09614,2.08502,2.07321,2.06067,2.04737,2.03325,2.01828,2.00241,1.98558,1.96776,1.94889,1.92891,1.90779,1.88546,1.86189,1.83701,1.81078,1.78315,1.75409,1.72354,1.69148,1.65787,1.6227,1.58595,1.54761,1.50769,1.46621,1.42322,1.37875,1.33288,1.28571,1.23734,1.18791,1.13759,1.08655,1.03501,0.983198,0.931361,0.879765,0.828689,0.778411,0.729207,0.68134,0.635053,0.590552,0.548011,0.507555,0.469259,0.433155,0.399225,0.367419,0.337656,0.309845},
	{1.01145,1.01217,1.01292,1.01372,1.01456,1.01544,1.01637,1.01735,1.01838,1.01947,1.02062,1.02183,1.02311,1.02445,1.02588,1.02737,1.02896,1.03063,1.03239,1.03425,1.03621,1.03827,1.04046,1.04276,1.04519,1.04775,1.05046,1.05331,1.05631,1.05948,1.06283,1.06635,1.07007,1.07398,1.0781,1.08245,1.08702,1.09184,1.09691,1.10224,1.10785,1.11375,1.11995,1.12647,1.13331,1.1405,1.14804,1.15594,1.16423,1.17291,1.182,1.19151,1.20145,1.21184,1.22268,1.234,1.24579,1.25807,1.27085,1.28413,1.29793,1.31224,1.32707,1.34243,1.3583,1.3747,1.39161,1.40904,1.42697,1.44539,1.46429,1.48366,1.50347,1.52371,1.54436,1.56537,1.58674,1.60842,1.63038,1.65259,1.675,1.69758,1.72027,1.74304,1.76583,1.78859,1.81128,1.83384,1.85622,1.87836,1.90022,1.92173,1.94286,1.96353,1.98372,2.00337,2.02243,2.04086,2.05863,2.07569,2.09202,2.1076,2.12238,2.13637,2.14954,2.16188,2.17339,2.18407,2.19392,2.20295,2.21117,2.21859,2.22524,2.23113,2.2363,2.24075,2.24453,2.24766,2.25016,2.25207,2.25341,2.25422,2.25451,2.25432,2.25366,2.25256,2.25103,2.2491,2.24677,2.24406,2.24097,2.23752,2.2337,2.22951,2.22495,2.22001,2.2147,2.20898,2.20286,2.19631,2.18932,2.18187,2.17392,2.16547,2.15647,2.1469,2.13673,2.12592,2.11444,2.10225,2.08931,2.07557,2.061,2.04555,2.02917,2.01182,1.99344,1.97399,1.95342,1.93167,1.9087,1.88445,1.85887,1.83192,1.80356,1.77373,1.7424,1.70953,1.6751,1.63909,1.60148,1.56228,1.52149,1.47913,1.43525,1.3899,1.34316,1.29512,1.2459,1.19565,1.14452,1.09272,1.04045,0.987958,0.93549,0.883316,0.831719,0.78098,0.731374,0.683162,0.636585,0.591845,0.549107,0.50849,0.470063,0.433848,0.399821,0.367928,0.338085,0.310199},
	{1.01126,1.01197,1.01271,1.0135,1.01432,1.01519,1.01611,1.01708,1.01809,1.01917,1.0203,1.02149,1.02275,1.02408,1.02548,1.02696,1.02852,1.03016,1.0319,1.03373,1.03566,1.0377,1.03985,1.04212,1.04452,1.04705,1.04971,1.05253,1.05549,1.05862,1.06192,1.0654,1.06907,1.07294,1.07701,1.0813,1.08583,1.09059,1.0956,1.10088,1.10644,1.11228,1.11843,1.12489,1.13167,1.1388,1.14629,1.15414,1.16238,1.17101,1.18005,1.18952,1.19942,1.20978,1.22059,1.23189,1.24367,1.25594,1.26872,1.28202,1.29584,1.31019,1.32507,1.34049,1.35645,1.37295,1.38998,1.40754,1.42562,1.44422,1.46332,1.48291,1.50297,1.52348,1.54442,1.56576,1.58748,1.60954,1.63191,1.65455,1.67743,1.7005,1.72372,1.74704,1.77041,1.79378,1.8171,1.84033,1.86339,1.88624,1.90883,1.93109,1.95298,1.97445,1.99543,2.01589,2.03577,2.05503,2.07364,2.09154,2.10871,2.12512,2.14074,2.15555,2.16953,2.18267,2.19497,2.20642,2.21702,2.22679,2.23571,2.24383,2.25114,2.25766,2.26343,2.26847,2.2728,2.27645,2.27944,2.28181,2.28358,2.28479,2.28545,2.28559,2.28524,2.28442,2.28314,2.28142,2.27929,2.27674,2.27379,2.27044,2.2667,2.26257,2.25804,2.25312,2.24778,2.24203,2.23584,2.22921,2.22211,2.21453,2.20644,2.19782,2.18863,2.17886,2.16846,2.1574,2.14565,2.13318,2.11993,2.10588,2.09096,2.07516,2.0584,2.04065,2.02187,2.00199,1.98096,1.95875,1.93529,1.91054,1.88445,1.85697,1.82805,1.79765,1.76573,1.73227,1.69722,1.66058,1.62232,1.58246,1.54099,1.49795,1.45338,1.40733,1.35989,1.31114,1.26121,1.21025,1.15842,1.10592,1.05296,0.999789,0.946659,0.893843,0.841623,0.790283,0.7401,0.691337,0.644234,0.598992,0.555777,0.514707,0.475849,0.439226,0.404811,0.372549,0.342355,0.314138},
	{1.01107,1.01177,1.0125,1.01328,1.01409,1.01495,1.01585,1.0168,1.0178,1.01886,1.01997,1.02115,1.02239,1.0237,1.02508,1.02653,1.02807,1.02969,1.0314,1.0332,1.0351,1.03711,1.03923,1.04147,1.04383,1.04632,1.04895,1.05172,1.05465,1.05773,1.06099,1.06442,1.06804,1.07185,1.07588,1.08012,1.08458,1.08929,1.09425,1.09947,1.10496,1.11074,1.11683,1.12322,1.12995,1.13702,1.14444,1.15223,1.16041,1.16898,1.17797,1.18738,1.19724,1.20755,1.21833,1.22959,1.24134,1.2536,1.26637,1.27967,1.2935,1.30787,1.32279,1.33825,1.35427,1.37085,1.38798,1.40565,1.42387,1.44263,1.4619,1.48169,1.50198,1.52274,1.54396,1.5656,1.58765,1.61007,1.63283,1.6559,1.67922,1.70277,1.72649,1.75034,1.77428,1.79824,1.82219,1.84605,1.86979,1.89334,1.91664,1.93965,1.9623,1.98454,2.00632,2.02758,2.04828,2.06837,2.0878,2.10654,2.12455,2.14179,2.15824,2.17388,2.18868,2.20263,2.21572,2.22794,2.2393,2.24979,2.25944,2.26824,2.27621,2.28338,2.28975,2.29537,2.30025,2.30441,2.30789,2.31071,2.31291,2.3145,2.31552,2.31599,2.31593,2.31537,2.31432,2.3128,2.31083,2.30842,2.30558,2.3023,2.29861,2.2945,2.28996,2.285,2.27959,2.27375,2.26745,2.26067,2.2534,2.24562,2.23731,2.22844,2.21898,2.20891,2.19819,2.18678,2.17467,2.16179,2.14813,2.13362,2.11824,2.10194,2.08466,2.06637,2.04701,2.02654,2.00489,1.98203,1.9579,1.93246,1.90564,1.87741,1.84773,1.81654,1.78381,1.74951,1.71361,1.6761,1.63696,1.5962,1.55383,1.50987,1.46439,1.41743,1.36908,1.31944,1.26864,1.21684,1.1642,1.11092,1.05724,1.00339,0.94964,0.896264,0.843547,0.791776,0.741226,0.692157,0.644807,0.599371,0.556007,0.514825,0.475882,0.439192,0.404725,0.372418,0.342184,0.31393}
};
