/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.datastax.oss.driver.api.querybuilder.term;

import com.datastax.oss.driver.api.core.CqlIdentifier;
import com.datastax.oss.driver.api.core.cql.Statement;
import com.datastax.oss.driver.api.core.type.DataType;
import com.datastax.oss.driver.api.querybuilder.BuildableQuery;
import com.datastax.oss.driver.api.querybuilder.CqlSnippet;
import com.datastax.oss.driver.api.querybuilder.QueryBuilder;
import com.datastax.oss.driver.api.querybuilder.relation.Relation;
import com.datastax.oss.driver.api.querybuilder.select.OngoingSelection;
import com.datastax.oss.driver.api.querybuilder.select.Selector;

/**
 * A simple expression that doesn't reference columns.
 *
 * <p>It is used as an argument to certain {@linkplain Selector selectors} (for example the indices
 * in a {@linkplain OngoingSelection#range(Selector, Term, Term) range}), or as the right operand of
 * {@linkplain Relation relations}.
 *
 * <p>To create a term, call one of the static factory methods in {@link QueryBuilder}:
 *
 * <ul>
 *   <li>{@link QueryBuilder#literal(Object) literal()} to inline a Java object into the query
 *       string;
 *   <li>{@link QueryBuilder#function(CqlIdentifier, CqlIdentifier, Iterable) function()} to invoke
 *       a built-in or user-defined function;
 *   <li>an arithmetic operator combining other terms: {@link QueryBuilder#add(Term, Term) add()},
 *       {@link QueryBuilder#subtract(Term, Term) subtract()}, {@link QueryBuilder#negate(Term)
 *       negate()}, {@link QueryBuilder#multiply(Term, Term) multiply()}, {@link
 *       QueryBuilder#divide(Term, Term) divide()} or {@link QueryBuilder#remainder(Term, Term)
 *       remainder()};
 *   <li>{@link QueryBuilder#typeHint(Term, DataType) typeHint()} to coerce another term to a
 *       particular CQL type;
 *   <li>{@link QueryBuilder#raw(String) raw()} for a raw CQL snippet.
 * </ul>
 *
 * Note that some of these methods have multiple overloads.
 */
public interface Term extends CqlSnippet {

  /**
   * Whether the term is idempotent.
   *
   * <p>That is, whether it always produces the same result when used multiple times. For example,
   * the literal {@code 1} is idempotent, the function call {@code now()} isn't.
   *
   * <p>This is used internally by the query builder to compute the {@link Statement#isIdempotent()}
   * flag on the statements generated by {@link BuildableQuery#build()}. If a term is ambiguous (for
   * example a raw snippet or a call to a user function), the builder is pessimistic and assumes the
   * term is not idempotent.
   */
  boolean isIdempotent();
}
